from dataclasses import dataclass
from datetime import timedelta

from .export_helper import randomize_start_time
from .cursor_provider import create_cursor_provider
from .task import TaskParams
from mail.shiva.stages.api.props.logger import get_uid_logger
from mail.shiva.stages.api.props.services.archive import ArchiveStorage
from mail.shiva.stages.api.settings.s3api import S3ApiSettings
from mail.shiva.stages.api.settings.tvm import TvmSettings
from mail.python.theatre.detail.tvm import TvmServiceTickets

log = get_uid_logger(__name__)


async def get_next_user_for_purge(conn, purge_ttl, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users mu LEFT JOIN mail.archives ma USING (uid)
                 WHERE is_here
                   AND NOT is_deleted
                   AND mu.state = 'active'
                   AND ma.state in ('archivation_in_progress', 'restoration_complete')
                   AND mu.last_state_update < (now() - %(purge_ttl)s)
                   AND ma.updated < (now() - %(purge_ttl)s)
                   AND uid > %(uid)s
                 ORDER BY uid
                 LIMIT 1
            ''',
            dict(
                purge_ttl=purge_ttl,
                uid=uid,
            )
        )
        res = await cur.fetchone()
        return res and res['uid']


async def purge_user_archive(conn, archive_storage, uid):
    try:
        user_keys = await archive_storage.list_user_objects(uid)
        await archive_storage.delete_objects(user_keys)

        async with conn.cursor() as cur:
            await cur.execute(
                '''
                DELETE FROM mail.archives
                WHERE uid = %(uid)s
                ''',
                dict(
                    uid=uid,
                )
            )
            log.info('successfully purge user archive', uid=uid)
    except Exception as exc:
        log.exception(f'Got exception: {exc}', uid=uid)


@dataclass
class PurgeArchivesParams(TaskParams):
    task_name: str = 'purge_archives'
    purge_ttl: timedelta = timedelta(days=10)
    tvm_ids: TvmSettings = None
    tvm_tickets: TvmServiceTickets = None
    s3api_settings: S3ApiSettings = None
    max_delay: int = 3600


async def shard_purge_archives(params: PurgeArchivesParams, stats):
    await randomize_start_time(max_delay=params.max_delay)

    archive_storage = ArchiveStorage(
        s3api_settings=params.s3api_settings,
        s3_id=params.tvm_ids.s3_id,
        tvm=params.tvm_tickets,
        stats=stats,
    )

    async with create_cursor_provider(params, stats) as conn:
        uid = 0
        while True:
            uid = await get_next_user_for_purge(
                conn=conn,
                purge_ttl=params.purge_ttl,
                uid=uid,
            )
            if not uid:
                break
            await purge_user_archive(conn, archive_storage, uid)
