import aiopg
import psycopg2
from psycopg2.extras import RealDictCursor
from dataclasses import dataclass
from datetime import timedelta

from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn

from .task import TaskParams
from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)
PURGE_TIMEOUT = 3000


@dataclass
class Backup:
    uid: int = None
    backup_id: int = None


async def get_backups_for_purge(conn, purge_ttl, chunk_size):
    while True:
        async with conn.cursor() as cur:
            await cur.execute(
                '''
                SELECT uid, backup_id
                  FROM backup.backups bb
                  JOIN mail.users mu USING (uid)
                 WHERE bb.state in ('inactive', 'error')
                   AND bb.updated < (now() - %(purge_ttl)s)
                   AND mu.is_here
                 LIMIT %(chunk_size)s
                ''',
                dict(
                    purge_ttl=purge_ttl,
                    chunk_size=chunk_size,
                )
            )
            chunk = [Backup(**r) async for r in cur]
            yield chunk
            if len(chunk) < chunk_size:
                return


async def purge_backup(conn, backup: Backup):
    try:
        async with conn.cursor(timeout=PURGE_TIMEOUT) as cur:
            await cur.execute(
                '''SELECT * FROM code.purge_old_backup(%(uid)s, %(backup_id)s)''',
                dict(
                    uid=backup.uid,
                    backup_id=backup.backup_id
                )
            )
            log.info(f'purge_backup: successfully purge user backup {backup.backup_id}', uid=backup.uid)
    except psycopg2.Error:
        log.exception('exception while purging backup', uid=backup.uid)


@dataclass
class PurgeBackupsParams(TaskParams):
    task_name: str = 'purge_backups'
    purge_ttl: timedelta = timedelta(days=30)
    chunk_size: int = 100


async def shard_purge_backups(params: PurgeBackupsParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats), cursor_factory=RealDictCursor) as conn:
        async for backups_chunk in get_backups_for_purge(
            conn=conn,
            purge_ttl=params.purge_ttl,
            chunk_size=params.chunk_size,
        ):
            for backup in backups_chunk:
                await purge_backup(conn, backup)
