import aiopg
import psycopg2
from itertools import groupby
from operator import itemgetter
from dataclasses import dataclass
from datetime import timedelta

from .helpers import chunks
from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams
from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)
PURGE_TIMEOUT = 600


async def get_deleted(conn, purge_ttl, chunk_size):
    while True:
        async with conn.cursor() as cur:
            await cur.execute(
                '''
                SELECT *
                  FROM (
                    SELECT uid, mid
                      FROM mail.deleted_box
                      JOIN mail.users USING (uid)
                     WHERE deleted_date < (now() - %(purge_ttl)s)
                       AND is_here
                     LIMIT %(chunk_size)s) dbc
                 ORDER BY uid, mid
                ''',
                dict(
                    purge_ttl=purge_ttl,
                    chunk_size=chunk_size
                )
            )
            deleted = [tuple(row) async for row in cur]
            yield deleted
            if len(deleted) < chunk_size:
                log.info(f'Got only {len(deleted)} items, chunk_size {chunk_size} - time to sleep')
                break


async def purge(conn, uid, mids):
    log.info(f'purge mids={mids}', uid=uid)
    try:
        async with conn.cursor(timeout=PURGE_TIMEOUT) as cur:
            await cur.execute(
                '''SELECT * FROM code.purge_deleted_messages(%(uid)s, %(mids)s)''',
                dict(
                    uid=uid,
                    mids=mids
                )
            )
            log.info(f'successfully purged {len(mids)} messages', uid=uid)
    except psycopg2.Error:
        log.exception('exception while purging messages', uid=uid)


async def purge_chunk(conn, uids_mids, delete_chunk_size):
    for uid, data_by_uid in groupby(uids_mids, itemgetter(0)):
        for data_chunk in chunks(data_by_uid, delete_chunk_size):
            await purge(conn, uid, [r[1] for r in data_chunk])


@dataclass
class PurgeDeletedBoxParams(TaskParams):
    task_name: str = 'purge_deleted_box'
    purge_ttl: timedelta = timedelta(days=190)
    get_deleted_chunk_size: int = 10000
    delete_chunk_size: int = 1000


async def shard_purge_deleted_box(params: PurgeDeletedBoxParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats)) as conn:
        async for deleted_chunk in get_deleted(
            conn,
            params.purge_ttl,
            params.get_deleted_chunk_size
        ):
            await purge_chunk(conn, deleted_chunk, params.delete_chunk_size)
