import logging
import aiopg
from dataclasses import dataclass

from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams
from .purge_deleted_box import purge_chunk

log = logging.getLogger(__name__)
GET_DELETED_TIMEOUT = 600


async def get_deleted_with_synced_attribute(conn, synced_deleted_chunk_size, deleted_chunk_size):
    uid = 0
    mid = 0
    while True:
        async with conn.cursor(timeout=GET_DELETED_TIMEOUT) as cur:
            await cur.execute(
                '''
                SELECT uid, mid FROM
                  (SELECT uid, mid, attributes
                     FROM mail.deleted_box
                     JOIN mail.messages USING (uid, mid)
                     JOIN mail.users USING (uid)
                    WHERE (uid, mid) > (%(uid)s, %(mid)s)
                      AND is_here
                    ORDER BY uid, mid
                    LIMIT %(deleted_chunk_size)s) as dm
                 WHERE 'synced' = ANY(attributes)
                 ORDER BY uid, mid
                 LIMIT %(synced_deleted_chunk_size)s
                ''',
                dict(
                    uid=uid,
                    mid=mid,
                    synced_deleted_chunk_size=synced_deleted_chunk_size,
                    deleted_chunk_size=deleted_chunk_size,
                )
            )

            chunk = [tuple(row) async for row in cur]
            if chunk:
                uid = chunk[-1][0]
                mid = chunk[-1][1]
                yield chunk
            else:
                await cur.execute(
                    '''
                    SELECT uid, mid FROM
                      (SELECT uid, mid, attributes
                         FROM mail.deleted_box
                         JOIN mail.messages USING (uid, mid)
                         JOIN mail.users USING (uid)
                        WHERE (uid, mid) > (%(uid)s, %(mid)s)
                          AND is_here
                        ORDER BY uid, mid
                        LIMIT %(deleted_chunk_size)s) as dm
                     ORDER BY uid DESC, mid DESC
                     LIMIT 1
                    ''',
                    dict(
                        uid=uid,
                        mid=mid,
                        deleted_chunk_size=deleted_chunk_size,
                    )
                )
                chunk = [tuple(row) async for row in cur]
                if chunk:
                    uid = chunk[-1][0]
                    mid = chunk[-1][1]
                else:
                    return


@dataclass
class PurgeSyncedDeletedBoxParams(TaskParams):
    task_name: str = 'purge_synced_deleted_box'
    get_synced_deleted_chunk_size: int = 10000
    get_deleted_chunk_size: int = 500000
    delete_chunk_size: int = 1000


async def shard_purge_synced_deleted_box(params: PurgeSyncedDeletedBoxParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats)) as conn:
        async for deleted_chunk in get_deleted_with_synced_attribute(
            conn,
            params.get_synced_deleted_chunk_size,
            params.get_deleted_chunk_size,
        ):
            await purge_chunk(conn, deleted_chunk, params.delete_chunk_size)
