import tenacity
from dataclasses import dataclass
from datetime import datetime

import yt.wrapper as yt
from yql.api.v1.client import YqlClient

from .helpers import run_in_executor, db_general_retry
from .task import TaskParams
from .freeze_helper import db_update_user_state, db_update_user_state_with_notifies_count, db_get_user_state, is_live_user
from .deactivate_users import YtFreezingParams, create_yt_table, write_users_to_yt, read_users_from_yt, check_yt_table, remove_yt_table
from .export_helper import randomize_start_time
from .cursor_provider import create_cursor_provider, locked_transactional_cursor
from .notify_helper import Templates, Metrics, NotifyParams, quick_send

from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)
GET_USERS_TIMEOUT = 1000
YQL_RETRY_DELAY = 300
FAKE_NOTIFIES_COUNT_FOR_REACTIVATION = -5


@db_general_retry
async def get_special_users_chunk(conn, last_uid, chunk_size):
    async with conn.cursor(timeout=GET_USERS_TIMEOUT) as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users u
                 WHERE is_here
                   AND NOT is_deleted
                   AND state IN ('inactive', 'notified')
                   AND notifies_count != %(notifies_count)s
                   AND uid > %(last_uid)s
                   AND (
                           uid BETWEEN 1130000000000000 AND 1139999999999999
                        OR EXISTS (SELECT 1 FROM mailish.accounts a WHERE a.uid = u.uid)
                        OR EXISTS (
                           SELECT 1
                             FROM filters.actions a
                             JOIN filters.rules r
                            USING (uid, rule_id)
                            WHERE a.uid = u.uid
                              AND enabled
                              AND oper IN ('forward', 'forwardwithstore', 'reply', 'notify'))
                        OR EXISTS (
                           SELECT 1
                             FROM settings.settings s
                            WHERE s.uid = u.uid
                              AND (value->'single_settings'->'has_pro_interface' = '"on"'
                               OR value->'single_settings'->'is_ad_disabled_via_billing' = '"on"')))
                 ORDER BY uid
                 LIMIT %(chunk_size)s
            ''',
            dict(
                last_uid=last_uid,
                notifies_count=FAKE_NOTIFIES_COUNT_FOR_REACTIVATION,
                chunk_size=chunk_size,
            )
        )
        return [r['uid'] async for r in cur]


async def get_special_users(conn, chunk_size):
    last_uid = 0
    while True:
        chunk = await get_special_users_chunk(conn, last_uid, chunk_size)

        if chunk:
            last_uid = chunk[-1]
            yield chunk

        if len(chunk) < chunk_size:
            return


@db_general_retry
async def get_inactive_users_chunk(conn, last_uid, chunk_size):
    async with conn.cursor(timeout=GET_USERS_TIMEOUT) as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users u
                 WHERE is_here
                   AND NOT is_deleted
                   AND state IN ('inactive', 'notified')
                   AND notifies_count != %(notifies_count)s
                   AND uid > %(last_uid)s
                 ORDER BY uid
                 LIMIT %(chunk_size)s
            ''',
            dict(
                last_uid=last_uid,
                notifies_count=FAKE_NOTIFIES_COUNT_FOR_REACTIVATION,
                chunk_size=chunk_size,
            )
        )
        return [r['uid'] async for r in cur]


async def get_inactive_users(conn, chunk_size):
    last_uid = 0
    while True:
        chunk = await get_inactive_users_chunk(conn, last_uid, chunk_size)

        if chunk:
            last_uid = chunk[-1]
            yield chunk

        if len(chunk) < chunk_size:
            return


@tenacity.retry(
    reraise=True,
    wait=tenacity.wait_random(min=YQL_RETRY_DELAY, max=YQL_RETRY_DELAY * 3),
    stop=tenacity.stop_after_attempt(3))
async def calculate_active_users(yql_config, db_users_table, passport_active_users_table, active_users_table, tmp_folder):
    def _proc(yql_config, db_users_table, passport_active_users_table, active_users_table, tmp_folder):
        query = f'''
            PRAGMA yt.TmpFolder = "{tmp_folder}";
            INSERT INTO `{active_users_table}` WITH TRUNCATE
                 SELECT s.uid as uid
                   FROM `{db_users_table}` as s
                   LEFT SEMI JOIN `{passport_active_users_table}` as a
                     ON s.uid = a.uid
               ORDER BY uid;
        '''
        with YqlClient(**yql_config) as yql_client:
            request = yql_client.query(query=query, syntax_version=1)
            request.run()
            request.wait_progress()
            if not request.is_success:
                log.error('YQL request failed: ' + str(request.get_results()))
                raise RuntimeError(f'YQL request failed. Status: {request.status}')

    return await run_in_executor(_proc, yql_config, db_users_table, passport_active_users_table, active_users_table, tmp_folder)


@dataclass
class ReactivateUsersParams(TaskParams, YtFreezingParams, NotifyParams):
    task_name: str = 'reactivate_users'
    chunk_size: int = 100000
    mail_templates: Templates = Templates(
        default='activate-def',
        ru='activate-ru',
        en='activate-en',
        tr='activate-tr')


@db_general_retry
async def update_user_state(conn, uid, params: NotifyParams, metrics: Metrics):
    async with locked_transactional_cursor(conn, uid) as cur:
        user_state = await db_get_user_state(cur, uid)
        if not is_live_user(user_state) or user_state.state not in ['inactive', 'notified', 'frozen', 'archived']:
            log.info('user was skipped, because his state was changed', uid=uid)
        elif user_state.state in ['frozen', 'archived']:
            await db_update_user_state_with_notifies_count(cur, user_state, user_state.state, FAKE_NOTIFIES_COUNT_FOR_REACTIVATION)
            log.warning(f'user with state={user_state.state} cannot be reactivated', uid=uid)
        else:
            await db_update_user_state(cur, user_state, 'active')
            return user_state
        return None


async def reactivate_user(conn, uid, params: ReactivateUsersParams, metrics: Metrics):
    try:
        old_user_state = await update_user_state(conn, uid, params, metrics)
        if old_user_state and old_user_state.state == 'notified' and old_user_state.notifies_count > 0:
            await quick_send(params, uid, params.mail_templates, metrics, recovery_send=False)
    except Exception as exc:
        log.exception(f'Got exception: {exc}', uid=uid)
        return
    log.info('user was successfully reactivated', uid=uid)


async def reactivate_special_users(params: ReactivateUsersParams, metrics: Metrics):
    special_users = set()
    async with create_cursor_provider(params, metrics) as conn:
        async for users_chunk in get_special_users(
            conn=conn,
            chunk_size=params.chunk_size,
        ):
            for uid in users_chunk:
                await reactivate_user(conn, uid, params, metrics)
                special_users.add(uid)
    log.info(f'Reactivated {len(special_users)} special users')
    return special_users


async def reactivate_active_users(params: ReactivateUsersParams, date_prefix: str, special_users: set, metrics: Metrics):
    yt_client = yt.YtClient(**params.yt_config)
    passport_active_users_table = f'{params.table_prefix}/active_users_dump_{date_prefix}'
    db_users_table = f'{params.table_prefix}/reactivate/db_users_{date_prefix}_{params.shard_id}'
    active_users_table = f'{params.table_prefix}/reactivate/active_users_{date_prefix}_{params.shard_id}'
    tmp_folder = f'{params.table_prefix}/tmp'

    await check_yt_table(yt_client, passport_active_users_table)

    try:
        await create_yt_table(yt_client, db_users_table)
        async with create_cursor_provider(params, metrics) as conn:
            async for users_chunk in get_inactive_users(
                conn=conn,
                chunk_size=params.chunk_size,
            ):
                data = [{'uid': uid} for uid in users_chunk if uid not in special_users]
                await write_users_to_yt(yt_client, db_users_table, data)
                log.info(f'Written {len(data)} uids to YT')

            await calculate_active_users(params.yql_config, db_users_table, passport_active_users_table, active_users_table, tmp_folder)
            active_users = await read_users_from_yt(yt_client, active_users_table, None)
            log.info(f'Read {len(active_users)} active users from YT')

            for uid in active_users:
                await reactivate_user(conn, uid, params, metrics)
    finally:
        await remove_yt_table(yt_client, db_users_table)
        await remove_yt_table(yt_client, active_users_table)


async def shard_reactivate_users(params: ReactivateUsersParams, stats):
    metrics = Metrics(stats)
    date_prefix = datetime. today().strftime('%Y-%m-%d')
    await randomize_start_time(max_delay=params.max_delay)
    special_users = await reactivate_special_users(params, metrics)
    await reactivate_active_users(params, date_prefix, special_users, metrics)
