import yt.wrapper as yt

import logging
import datetime
import aiopg
from psycopg2.extras import RealDictCursor
from dataclasses import dataclass, field as class_field
from typing import List, Set
from .helpers import run_in_executor
from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams
from .export_helper import (
    create_yt_table,
    write_data_to_yt,
    get_start_id,
    randomize_start_time,
    ExportParams,
)

log = logging.getLogger(__name__)


@dataclass
class UserSettings:
    uid: int = None
    value: int = None


async def read_users_settings(conn, from_uid, chunk_size):
    while True:
        async with conn.cursor() as cur:
            await cur.execute(
                '''
                SELECT uid, s.value
                  FROM settings.settings s JOIN
                       mail.users u using(uid)
                 WHERE uid >= %(from_uid)s
                   AND u.is_here
                   AND NOT u.is_deleted
                 ORDER BY uid
                 LIMIT %(chunk_size)s
                ''',
                dict(
                    from_uid=from_uid,
                    chunk_size=chunk_size,
                )
            )
            chunk = [UserSettings(**r) async for r in cur]

            if chunk:
                from_uid = chunk[-1].uid + 1
                yield chunk

            if len(chunk) < chunk_size:
                return


@dataclass
class SettingsFields:
    all: List[str] = class_field(default_factory=list)
    private: List[str] = class_field(default_factory=list)
    unknown: Set[str] = class_field(default_factory=set)


async def read_fields(yt_client, public_field_list_path, private_field_list_path):
    def _proc(field_list_path):
        current = []

        for line in list(yt_client.read_file(field_list_path)):
            line = line.decode('utf-8').strip()
            current.append(line)

        return current

    fields = SettingsFields()
    fields.all = await run_in_executor(_proc, public_field_list_path)
    fields.private = await run_in_executor(_proc, private_field_list_path)
    fields.all.extend(fields.private)
    return fields


async def create_yt_settings_table(yt_client, table, fields):
    standard_fields = {
        'uid': 'int64',
        'from_shard_id': 'int64',
        'signs_count': 'int64',
    }
    await create_yt_table(yt_client, table, fields=standard_fields, auto_fields=set(fields.all))


def mask(settings, field):
    if field in settings:
        settings[field] = str(hash(settings[field]))


def remove_privacy(shard_id, uid, value, fields):
    signs_count = len(value['signs']) if 'signs' in value else 0
    settings = value['single_settings']
    unknown_fields = set(settings).difference(fields.all)
    if unknown_fields:
        fields.unknown.update(unknown_fields)
    for field in fields.private:
        mask(settings, field)
    for field in unknown_fields:
        mask(settings, field)
    settings['from_shard_id'] = shard_id
    settings['uid'] = uid
    settings['signs_count'] = signs_count
    return settings


async def export_interval(conn, chunk_size, table_prefix, shard_id, from_uid, fields, yt_client):
    date_prefix = datetime.datetime.today().strftime('%Y-%m-%d')
    table = f'{table_prefix}{date_prefix}/{shard_id}'

    if from_uid == 0:
        from_uid = await get_start_id(yt_client, table) + 1
    log.info(f'Process will be resumed from uid = {from_uid}')

    await create_yt_settings_table(yt_client, table, fields)

    async for users in read_users_settings(conn, from_uid, chunk_size):
        result = []
        for user in users:
            if user.value is not None:
                result.append(remove_privacy(shard_id, user.uid, user.value, fields))
        await write_data_to_yt(yt_client, table, result)
        log.info(f'successfully written {len(result)} lines to YT')

    if fields.unknown:
        log.warn(f'Unknown field found: {fields.unknown}')


@dataclass
class SettingsExportParams(TaskParams, ExportParams):
    task_name: str = 'settings_export'
    table_prefix: str = '//home/mail-logs/mail-settings/'
    public_field_list_path: str = '//home/mail-logs/mail-settings/field_list_public.txt'
    private_field_list_path: str = '//home/mail-logs/mail-settings/field_list_private.txt'


async def shard_settings_export(params: SettingsExportParams, stats):
    await randomize_start_time(max_delay=params.max_delay)
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats), cursor_factory=RealDictCursor) as conn:
        log.info(f'Start settings_export with chunk size {params.chunk_size}')
        yt_client = yt.YtClient(**params.yt_config)
        fields = await read_fields(yt_client, params.public_field_list_path, params.private_field_list_path)
        await export_interval(
            conn=conn,
            chunk_size=params.chunk_size,
            table_prefix=params.table_prefix,
            shard_id=params.shard_id,
            from_uid=params.from_uid,
            fields=fields,
            yt_client=yt_client,
        )
