import psycopg2
from dataclasses import dataclass
from datetime import timedelta

from .helpers import db_general_retry
from .task import TaskParams
from .freeze_helper import db_update_user_state_with_notifies_count, db_get_user_state, is_live_user
from .cursor_provider import create_cursor_provider, locked_transactional_cursor
from mail.shiva.stages.api.props.logger import get_uid_logger

log = get_uid_logger(__name__)


@db_general_retry
async def get_users_chunk(conn, state_ttl, last_uid, chunk_size):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users
                 WHERE is_here
                   AND NOT is_deleted
                   AND state = 'inactive'
                   AND last_state_update < (now() - %(state_ttl)s)
                   AND uid > %(last_uid)s
                 ORDER BY uid
                 LIMIT %(chunk_size)s
            ''',
            dict(
                state_ttl=state_ttl,
                last_uid=last_uid,
                chunk_size=chunk_size,
            )
        )
        return [rec['uid'] async for rec in cur]


async def get_users(conn, state_ttl, chunk_size, users_count):
    last_uid = 0
    while True:
        if chunk_size > users_count:
            chunk_size = users_count
        users_count -= chunk_size

        chunk = await get_users_chunk(conn, state_ttl, last_uid, chunk_size)

        if chunk:
            last_uid = chunk[-1]
            yield chunk

        if len(chunk) < chunk_size or users_count == 0:
            return


async def start_freezing_user(conn, uid):
    try:
        async with locked_transactional_cursor(conn, uid) as cur:
            user_state = await db_get_user_state(cur, uid)
            if not is_live_user(user_state) or user_state.state != 'inactive':
                log.info('user was skipped, because his state was changed', uid=uid)
                return
            await db_update_user_state_with_notifies_count(cur, user_state, 'notified', 0)
    except psycopg2.Error as exc:
        log.exception(f'Got exception: {exc}', uid=uid)
        return
    log.info('successfully started freezing user', uid=uid)


@dataclass
class StartFreezingUsersParams(TaskParams):
    task_name: str = 'start_freezing_users'
    state_ttl: timedelta = timedelta(days=1)
    chunk_size: int = 1000
    max_users_count: int = 10000


async def shard_start_freezing_users(params: StartFreezingUsersParams, stats):
    async with create_cursor_provider(params, stats) as conn:
        async for users_chunk in get_users(
            conn=conn,
            state_ttl=params.state_ttl,
            chunk_size=params.chunk_size,
            users_count=params.max_users_count,
        ):
            for uid in users_chunk:
                await start_freezing_user(conn, uid)
