import logging
from dataclasses import dataclass
from datetime import timedelta

import aiopg
from psycopg2.extras import RealDictCursor

from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams, HuskydbEngine
from .transfer_users import add_transfer_tasks

log = logging.getLogger(__name__)

PG_TIMEOUT = 600


async def get_acive_users(conn, users_count, activity_interval):
    async with conn.cursor(timeout=PG_TIMEOUT) as cur:
        await cur.execute(
            '''
            SELECT uid FROM (
              SELECT uid, count(*) as cnt
                FROM mail.change_log
                JOIN mail.users using(uid)
               WHERE change_date > now() - %(activity_interval)s
                 AND is_here
               GROUP BY uid) as activity
             ORDER BY cnt DESC
             LIMIT %(users_count)s
            ''',
            dict(
                users_count=users_count,
                activity_interval=activity_interval,
            )
        )
        return [rec['uid'] async for rec in cur]


@dataclass
class TransferActiveUsersParams(TaskParams):
    task_name: str = 'transfer_active_users'
    huskydb: HuskydbEngine = None
    users_count: int = 32000
    users_chunk_size: int = 10000
    activity_interval: timedelta = timedelta(days=7)
    task_args: str = None
    transfer_priority: int = -11
    load_type: str = 'dbaas_hot'
    shards_count: int = 5


async def shard_transfer_active_users(params: TransferActiveUsersParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats), cursor_factory=RealDictCursor) as maildb_conn:
        uids = await get_acive_users(
            conn=maildb_conn,
            users_count=params.users_count,
            activity_interval=params.activity_interval,
        )
        async with params.huskydb.connection() as huskydb_conn:
            await add_transfer_tasks(
                conn=huskydb_conn,
                from_db=params.shard_id,
                load_type=params.load_type,
                shards_count=params.shards_count,
                extra_task_args=params.task_args,
                priority=params.transfer_priority,
                users_chunk_size=params.users_chunk_size,
                uids=uids,
            )
