import aiopg
from dataclasses import dataclass
from datetime import timedelta

from mail.shiva.stages.api.props.services.sharpei import get_shard_dsn
from .task import TaskParams


async def get_users(conn, update_ttl):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users u
                 WHERE is_here
                   AND NOT EXISTS
                    (SELECT 1
                       FROM stats.users_info i
                      WHERE u.uid = i.uid AND last_update > now() - %(delay)s)
            ''',
            dict(delay=update_ttl)
        )
        return [row[0] async for row in cur]


async def get_user_db_size(conn, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            "SELECT util.mailbox_size_dynamic(%(uid)s)",
            dict(uid=uid)
        )
        return (await cur.fetchone())[0]


async def get_user_storage_size(conn, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            "SELECT sum(message_size) FROM mail.folders WHERE uid=%(uid)s",
            dict(uid=uid)
        )
        return (await cur.fetchone())[0]


async def update_user_mailbox_size(conn, uid):
    db_size = await get_user_db_size(conn, uid)
    storage_size = await get_user_storage_size(conn, uid)
    async with conn.cursor() as cur:
        await cur.execute(
            '''
                INSERT INTO stats.users_info
                    (uid, last_update, db_size, storage_size)
                VALUES (
                    %(uid)s,
                    now(),
                    %(db_size)s,
                    %(storage_size)s)
                ON CONFLICT (uid) DO UPDATE
                    SET db_size = %(db_size)s,
                        storage_size = %(storage_size)s,
                        last_update = now()
            ''',
            dict(
                uid=uid,
                db_size=db_size,
                storage_size=storage_size
            )
        )


@dataclass
class UpdateMailboxSizeParams(TaskParams):
    task_name: str = 'update_mailbox_size'
    update_ttl: timedelta = timedelta(days=14)


async def shard_update_mailbox_size(params: UpdateMailboxSizeParams, stats):
    async with aiopg.connect(await get_shard_dsn(params.sharpei, params.db_user, params.shard_id, stats)) as conn:
        for uid in await get_users(conn, params.update_ttl):
            await update_user_mailbox_size(conn, uid)
