from aiohttp import web
from datetime import timedelta

from mail.python.theatre.detail.tvm import TvmServiceTickets
from mail.shiva.stages.api.settings.app import Settings
from mail.shiva.stages.api.roles.shard_worker import ShardWorker

from .shard.cleanup_doomed import shard_cleanup_doomed, СleanupDoomedParams
from .shard.close_for_load import shard_close_for_load, CloseForLoadParams
from .shard.end_prepared_transaction import shard_end_prepared_transaction, EndPreparedTransactionParams
from .shard.init_pop3_folders import shard_init_pop3_folder, InitPop3FolderParams
from .shard.folder_archivation import shard_folder_archivation, FolderArchivationParams
from .shard.pg_partman_maintenance import shard_pg_partman_maintenance, PgPartmanMaintenanceParams
from .shard.purge_backups import shard_purge_backups, PurgeBackupsParams
from .shard.purge_chained_log import shard_purge_chained_log, PurgeChainedLogParams
from .shard.purge_deleted_box import shard_purge_deleted_box, PurgeDeletedBoxParams
from .shard.purge_synced_deleted_box import shard_purge_synced_deleted_box, PurgeSyncedDeletedBoxParams
from .shard.purge_storage import shard_purge_storage, PurgeStorageParams
from .shard.purge_user import shard_purge_deleted_user, shard_purge_transferred_user, PurgeDeletedUserParams, PurgeTransferredUserParams
from .shard.space_balancer import shard_space_balancer, SpaceBalancerParams
from .shard.transfer_users import shard_transfer_users, TransferUsersParams
from .shard.transfer_active_users import shard_transfer_active_users, TransferActiveUsersParams
from .shard.update_mailbox_size import shard_update_mailbox_size, UpdateMailboxSizeParams
from .shard.settings_export import shard_settings_export, SettingsExportParams
from .shard.pnl_estimation_export import shard_pnl_estimation_export, PnlEstimationExportParams
from .shard.task import HuskydbEngine

from .shard.reactivate_users import shard_reactivate_users, ReactivateUsersParams
from .shard.deactivate_users import shard_deactivate_users, DeactivateUsersParams
from .shard.start_freezing_users import shard_start_freezing_users, StartFreezingUsersParams
from .shard.notify_users import shard_notify_users, NotifyUsersParams
from .shard.freeze_users import shard_freeze_users, FreezeUsersParams
from .shard.archive_users import shard_archive_users, ArchiveUsersParams
from .shard.purge_archives import shard_purge_archives, PurgeArchivesParams
from .shard.clean_archives import shard_clean_archives, CleanArchivesParams

from .shard.onetime_task import shard_onetime_task, OnetimeTaskParams

from .base_api import WorkerApi


class ShardApi(WorkerApi):
    def __init__(
        self,
        settings: Settings,
        huskydb: HuskydbEngine,
        shard_worker: ShardWorker,
        tvm: TvmServiceTickets
    ):
        super().__init__(settings, huskydb, shard_worker)
        self._tvm = tvm

    def bind_routes(self, app: web.Application):
        app.router.add_get('/cleanup_doomed', self.cleanup_doomed)
        app.router.add_get('/close_for_load', self.close_for_load)
        app.router.add_get('/end_prepared_transaction', self.end_prepared_transaction)
        app.router.add_get('/init_pop3_folder', self.init_pop3_folder)
        app.router.add_get('/folder_archivation', self.folder_archivation)
        app.router.add_get('/pg_partman_maintenance', self.pg_partman_maintenance)
        app.router.add_get('/purge_backups', self.purge_backups)
        app.router.add_get('/purge_chained_log', self.purge_chained_log)
        app.router.add_get('/purge_deleted_box', self.purge_deleted_box)
        app.router.add_get('/purge_synced_deleted_box', self.purge_synced_deleted_box)
        app.router.add_get('/purge_storage', self.purge_storage)
        app.router.add_get('/purge_deleted_user', self.purge_deleted_user)
        app.router.add_get('/purge_transferred_user', self.purge_transferred_user)
        app.router.add_post('/space_balancer', self.space_balancer)
        app.router.add_post('/transfer_users', self.transfer_users)
        app.router.add_post('/transfer_active_users', self.transfer_active_users)
        app.router.add_get('/update_mailbox_size', self.update_mailbox_size)
        app.router.add_get('/settings_export', self.settings_export)
        app.router.add_get('/pnl_estimation_export', self.pnl_estimation_export)

        # jobs for freezing users
        app.router.add_get('/reactivate_users', self.reactivate_users)
        app.router.add_get('/deactivate_users', self.deactivate_users)
        app.router.add_get('/start_freezing_users', self.start_freezing_users)
        app.router.add_get('/notify_users', self.notify_users)
        app.router.add_get('/freeze_users', self.freeze_users)
        app.router.add_get('/archive_users', self.archive_users)
        app.router.add_get('/purge_archives', self.purge_archives)
        app.router.add_get('/clean_archives', self.clean_archives)

        app.router.add_get('/onetime_task', self.onetime_task)

    async def cleanup_doomed(self, request: web.Request):
        params = СleanupDoomedParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_cleanup_doomed)

    async def close_for_load(self, request: web.Request):
        params = CloseForLoadParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            huskydb=self._huskydb,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.max_transfer_ratio = float(request.query.get('max_transfer_ratio', params.max_transfer_ratio))
        params.max_registration_ratio = float(request.query.get('max_registration_ratio', params.max_registration_ratio))
        return await self.run_task(params, shard_close_for_load)

    async def end_prepared_transaction(self, request: web.Request):
        params = EndPreparedTransactionParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        if 'ttl_sc' in request.query:
            params.transaction_ttl = timedelta(seconds=int(request.query.get('ttl_sc')))
        return await self.run_task(params, shard_end_prepared_transaction)

    async def init_pop3_folder(self, request: web.Request):
        params = InitPop3FolderParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_init_pop3_folder)

    async def folder_archivation(self, request: web.Request):
        params = FolderArchivationParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_folder_archivation)

    async def pg_partman_maintenance(self, request: web.Request):
        params = PgPartmanMaintenanceParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_pg_partman_maintenance)

    async def purge_backups(self, request: web.Request):
        params = PurgeBackupsParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        if 'ttl_days' in request.query:
            params.purge_ttl = timedelta(days=int(request.query.get('ttl_days')))
        return await self.run_task(params, shard_purge_backups)

    async def purge_chained_log(self, request: web.Request):
        params = PurgeChainedLogParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_purge_chained_log)

    async def purge_deleted_box(self, request: web.Request):
        params = PurgeDeletedBoxParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        if 'ttl_days' in request.query:
            params.purge_ttl = timedelta(days=int(request.query.get('ttl_days')))
        return await self.run_task(params, shard_purge_deleted_box)

    async def purge_synced_deleted_box(self, request: web.Request):
        params = PurgeSyncedDeletedBoxParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_purge_synced_deleted_box)

    async def purge_storage(self, request: web.Request):
        params = PurgeStorageParams(
            sharpei=self._settings.sharpei,
            mgate_host=self._settings.storage.host,
            mgate_port=self._settings.storage.port,
            ca_cert_path=self._settings.storage.ca_cert_path,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            mds_id=self._settings.tvm.mds_id,
            tvm=self._tvm
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.jobs_count = int(request.query.get('jobs_count', params.jobs_count))
        params.job_no = int(request.query.get('job_no', params.job_no))
        params.delete_chunk_size = int(request.query.get('delete_chunk_size', params.delete_chunk_size))
        if 'ttl_days' in request.query:
            params.purge_ttl = timedelta(days=int(request.query.get('ttl_days')))
        return await self.run_task(params, shard_purge_storage)

    async def purge_deleted_user(self, request: web.Request):
        params = PurgeDeletedUserParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        params.jobs_count = int(request.query.get('jobs_count', params.jobs_count))
        params.job_no = int(request.query.get('job_no', params.job_no))
        params.force = request.query.get('force', params.force)
        if 'ttl_days' in request.query:
            params.purge_ttl = timedelta(days=int(request.query.get('ttl_days')))
        return await self.run_task(params, shard_purge_deleted_user)

    async def purge_transferred_user(self, request: web.Request):
        params = PurgeTransferredUserParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        params.jobs_count = int(request.query.get('jobs_count', params.jobs_count))
        params.job_no = int(request.query.get('job_no', params.job_no))
        params.force = request.query.get('force', params.force)
        if 'ttl_days' in request.query:
            params.purge_ttl = timedelta(days=int(request.query.get('ttl_days')))
        return await self.run_task(params, shard_purge_transferred_user)

    async def space_balancer(self, request: web.Request):
        params = SpaceBalancerParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            huskydb=self._huskydb,
        )
        params.transfer_ratio = float(request.query.get('transfer_ratio', params.transfer_ratio))
        params.db_used_ratio = float(request.query.get('db_used_ratio', params.db_used_ratio))
        params.db_used_without_bloat_ratio = float(request.query.get('db_used_without_bloat_ratio', params.db_used_without_bloat_ratio))
        params.shards_count = int(request.query.get('shards_count', params.shards_count))
        params.load_type = request.query.get('load_type', params.load_type)
        params.task_args = await request.text()
        return await self.run_task(params, shard_space_balancer)

    async def transfer_users(self, request: web.Request):
        params = TransferUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            huskydb=self._huskydb,
        )
        params.messages_count = int(request.query.get('messages_count', params.messages_count))
        params.min_messages_per_user = int(request.query.get('min_messages_per_user', params.min_messages_per_user))
        params.max_messages_per_user = int(request.query.get('max_messages_per_user', params.max_messages_per_user))
        params.shards_count = int(request.query.get('shards_count', params.shards_count))
        params.load_type = request.query.get('load_type', params.load_type)
        params.task_args = await request.text()
        return await self.run_task(params, shard_transfer_users)

    async def transfer_active_users(self, request: web.Request):
        params = TransferActiveUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            huskydb=self._huskydb,
        )
        params.users_count = int(request.query.get('users_count', params.users_count))
        params.shards_count = int(request.query.get('shards_count', params.shards_count))
        params.load_type = request.query.get('load_type', params.load_type)
        params.task_args = await request.text()
        return await self.run_task(params, shard_transfer_active_users)

    async def update_mailbox_size(self, request: web.Request):
        params = UpdateMailboxSizeParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        return await self.run_task(params, shard_update_mailbox_size)

    async def settings_export(self, request: web.Request):
        params = SettingsExportParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            yt_config=self._settings.yt.yt_config,
        )
        params.from_uid = int(request.query.get('from_uid', params.from_uid))
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        return await self.run_task(params, shard_settings_export)

    async def reactivate_users(self, request: web.Request):
        params = ReactivateUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            yt_config=self._settings.yt.yt_config,
            yql_config=self._settings.yt.yql_config,
            bb_settings=self._settings.blackbox,
            sendr_settings=self._settings.sendr,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        return await self.run_task(params, shard_reactivate_users)

    async def deactivate_users(self, request: web.Request):
        params = DeactivateUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            yt_config=self._settings.yt.yt_config,
            yql_config=self._settings.yt.yql_config,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.max_users_count = int(request.query.get('max_users_count', params.max_users_count))
        return await self.run_task(params, shard_deactivate_users)

    async def start_freezing_users(self, request: web.Request):
        params = StartFreezingUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id'))
        )
        params.max_users_count = int(request.query.get('max_users_count', params.max_users_count))
        return await self.run_task(params, shard_start_freezing_users)

    async def notify_users(self, request: web.Request):
        params = NotifyUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            bb_settings=self._settings.blackbox,
            sendr_settings=self._settings.sendr,
            surveillance_settings=self._settings.surveillance,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.max_users_count = int(request.query.get('max_users_count', params.max_users_count))
        return await self.run_task(params, shard_notify_users)

    async def freeze_users(self, request: web.Request):
        params = FreezeUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            passport_settings=self._settings.passport,
            surveillance_settings=self._settings.surveillance,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.max_users_count = int(request.query.get('max_users_count', params.max_users_count))
        return await self.run_task(params, shard_freeze_users)

    async def archive_users(self, request: web.Request):
        params = ArchiveUsersParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            s3api_settings=self._settings.s3api,
            huskydb=self._huskydb,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.max_users_count = int(request.query.get('max_users_count', params.max_users_count))
        params.messages_chunk_size = int(request.query.get('messages_chunk_size', params.messages_chunk_size))
        return await self.run_task(params, shard_archive_users)

    async def purge_archives(self, request: web.Request):
        params = PurgeArchivesParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            s3api_settings=self._settings.s3api,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        return await self.run_task(params, shard_purge_archives)

    async def clean_archives(self, request: web.Request):
        params = CleanArchivesParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            s3api_settings=self._settings.s3api,
        )
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        return await self.run_task(params, shard_clean_archives)

    async def pnl_estimation_export(self, request: web.Request):
        params = PnlEstimationExportParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            yt_config=self._settings.yt.yt_config,
        )
        params.from_uid = int(request.query.get('from_uid', params.from_uid))
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        return await self.run_task(params, shard_pnl_estimation_export)

    async def onetime_task(self, request: web.Request):
        params = OnetimeTaskParams(
            sharpei=self._settings.sharpei,
            shard_id=int(self.get_required_parameter(request, 'shard_id')),
            subtask_name=self.get_required_parameter(request, 'subtask'),
            settings=self._settings,
            tvm_tickets=self._tvm,
        )
        params.max_count = int(request.query.get('max_count', params.max_count))
        params.max_delay = int(request.query.get('max_delay', params.max_delay))
        params.jobs_count = int(request.query.get('jobs_count', params.jobs_count))
        params.job_no = int(request.query.get('job_no', params.job_no))
        return await self.run_task(params, shard_onetime_task)
