from dataclasses import dataclass

from mail.shiva.stages.api.roles.shard_worker import TaskStats
from mail.shiva.stages.api.props.errors import BadRequest
from mail.shiva.stages.api.props.shard.cursor_provider import create_uid_cursor_provider, ShardDbCursorProvider
from mail.shiva.stages.api.props.shard.archive_users import archive_user as archive_user_impl, get_open_shards_for_move, get_user_shard_id, User
from mail.shiva.stages.api.props.shard.task import HuskydbEngine
from mail.shiva.stages.api.props.services.archive import ArchiveStorage
from mail.shiva.stages.api.props.util.freeze_handler import FreezeUserParams
from mail.shiva.stages.api.settings.s3api import S3ApiSettings


async def get_user(conn, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
                SELECT uid, ma.state as archivation_state
                  FROM mail.users mu LEFT JOIN mail.archives ma USING (uid)
                 WHERE is_here
                   AND NOT is_deleted
                   AND mu.state = 'frozen'
                   AND (ma.state is NULL or ma.state = 'archivation_in_progress')
                   AND uid = %(uid)s
            ''',
            dict(
                uid=uid,
            )
        )
        res = await cur.fetchone()
        return res and User(**res)


@dataclass
class ArchiveUserParams(FreezeUserParams):
    db_user: str = 'sharpei'
    messages_chunk_size: int = 10000
    load_type: str = 'dbaas_hot'
    s3api_settings: S3ApiSettings = None
    huskydb: HuskydbEngine = None


async def archive_user(params: ArchiveUserParams):
    if not params.check_uid_allowed():
        raise BadRequest('uid not allowed to be modified')

    stats = TaskStats('util_archive_user')

    archive_storage = ArchiveStorage(
        s3api_settings=params.s3api_settings,
        s3_id=params.tvm_ids.s3_id,
        tvm=params.tvm_tickets,
        stats=stats,
    )

    sharddb_conn = ShardDbCursorProvider(params.sharpei, stats)
    try:
        params.shard_id = await get_user_shard_id(sharddb_conn, params.uid)
        open_shard_ids = await get_open_shards_for_move(params)
        async with create_uid_cursor_provider(params, stats) as conn:
            user = await get_user(conn, params.uid)
            if not user:
                raise BadRequest('bad user state')
            await archive_user_impl(
                conn=conn,
                sharddb_conn=sharddb_conn,
                params=params,
                user=user,
                archive_storage=archive_storage,
                messages_chunk_size=params.messages_chunk_size,
                open_shard_ids=open_shard_ids,
                stats=stats,
            )
    finally:
        await sharddb_conn.close()
