from dataclasses import dataclass

from mail.python.theatre.detail.tvm import TvmServiceTickets
from mail.python.theatre.profiling.http import ProfiledClientSession

from mail.shiva.stages.api.props.services.passport import passport_update_user_state
from mail.shiva.stages.api.roles.shard_worker import TaskStats
from mail.shiva.stages.api.settings.log import http_logger
from mail.shiva.stages.api.settings.freeze_settings import FreezeSettings
from mail.shiva.stages.api.settings.passport import PassportSettings
from mail.shiva.stages.api.settings.tvm import TvmSettings
from mail.shiva.stages.api.props.errors import BadRequest
from mail.shiva.stages.api.props.shard.cursor_provider import create_uid_cursor_provider, locked_transactional_cursor
from mail.shiva.stages.api.props.shard.task import TaskParams
from mail.shiva.stages.api.props.shard.freeze_helper import db_get_user_state, is_live_user, log_update_user_state


@dataclass
class FreezeUserParams(TaskParams):
    uid: str = None
    freeze_settings: FreezeSettings = None
    passport_settings: PassportSettings = None
    tvm_ids: TvmSettings = None
    tvm_tickets: TvmServiceTickets = None

    def check_uid_allowed(self):
        return self.freeze_settings.allowed_uids is None or self.uid in self.freeze_settings.allowed_uids


async def db_freeze_user(cur, uid):
    user = await db_get_user_state(cur, uid)
    if not is_live_user(user):
        raise RuntimeError('user is not alive in shard')
    await cur.execute(
        '''
        UPDATE mail.users
           SET state = %(new_state)s,
               last_state_update = current_timestamp
         WHERE uid = %(uid)s;
        ''',
        dict(
            uid=user.uid,
            new_state='frozen',
        )
    )
    log_update_user_state(user, 'frozen', user.notifies_count)


async def freeze_user_impl(passport_client, passport_settings, passport_tvm_ticket, conn, uid, stats):
    async with locked_transactional_cursor(conn, uid) as cur:
        await db_freeze_user(cur, uid)
        await passport_update_user_state(passport_client, passport_settings, uid, passport_tvm_ticket, stats)
        stats.increase_task_meter('passport_update_user_state_success')


async def freeze_user(params: FreezeUserParams):
    if not params.check_uid_allowed():
        raise BadRequest('uid not allowed to be modified')
    stats = TaskStats('util_freeze_user')
    passport_tvm_ticket = await params.tvm_tickets.get(params.tvm_ids.passport_id)
    async with create_uid_cursor_provider(params, stats) as conn:
        async with ProfiledClientSession(metrics=stats, logger=http_logger.get_logger(),
                                         timeout=params.passport_settings.timeout) as client:
            await freeze_user_impl(client, params.passport_settings,
                                   passport_tvm_ticket, conn, params.uid, stats)
