from dataclasses import dataclass

from mail.shiva.stages.api.roles.shard_worker import TaskStats
from mail.shiva.stages.api.props.errors import BadRequest
from mail.shiva.stages.api.props.shard.cursor_provider import create_uid_cursor_provider
from mail.shiva.stages.api.props.shard.purge_archives import purge_user_archive
from mail.shiva.stages.api.props.services.archive import ArchiveStorage
from mail.shiva.stages.api.props.util.freeze_handler import FreezeUserParams
from mail.shiva.stages.api.settings.s3api import S3ApiSettings


async def get_user(conn, uid):
    async with conn.cursor() as cur:
        await cur.execute(
            '''
                SELECT uid
                  FROM mail.users mu LEFT JOIN mail.archives ma USING (uid)
                 WHERE is_here
                   AND NOT is_deleted
                   AND mu.state = 'active'
                   AND ma.state in ('archivation_in_progress', 'restoration_complete')
                   AND uid = %(uid)s
            ''',
            dict(
                uid=uid,
            )
        )
        res = await cur.fetchone()
        return res and res['uid']


@dataclass
class PurgeArchiveParams(FreezeUserParams):
    s3api_settings: S3ApiSettings = None


async def purge_archive(params: PurgeArchiveParams):
    if not params.check_uid_allowed():
        raise BadRequest('uid not allowed to be modified')

    stats = TaskStats('util_purge_archive')

    archive_storage = ArchiveStorage(
        s3api_settings=params.s3api_settings,
        s3_id=params.tvm_ids.s3_id,
        tvm=params.tvm_tickets,
        stats=stats,
    )

    async with create_uid_cursor_provider(params, stats) as conn:
        uid = await get_user(conn, params.uid)
        if not uid:
            raise BadRequest('bad user state')
        await purge_user_archive(conn, archive_storage, uid)
