from aiohttp import web

from mail.python.theatre.detail.tvm import TvmServiceTickets
from mail.shiva.stages.api.settings.app import Settings
from .shard.task import HuskydbEngine
from .base_api import BaseApi
from .util.freeze_handler import FreezeUserParams, freeze_user
from .util.archive_handler import ArchiveUserParams, archive_user
from .util.clean_archive_handler import CleanArchiveParams, clean_archive
from .util.purge_archive_handler import PurgeArchiveParams, purge_archive
from .util.purge_transferred_user_handler import PurgeTransferredUserParams, purge_transferred_user


class UtilApi(BaseApi):
    def __init__(self, settings: Settings, huskydb: HuskydbEngine, tvm: TvmServiceTickets):
        super().__init__(settings, huskydb)
        self._tvm = tvm

    def bind_routes(self, app: web.Application):
        app.router.add_get('/freeze_user', self.freeze_user)
        app.router.add_get('/archive_user', self.archive_user)
        app.router.add_get('/clean_archive', self.clean_archive)
        app.router.add_get('/purge_archive', self.purge_archive)
        app.router.add_get('/purge_transferred_user', self.purge_transferred_user)

    async def freeze_user(self, request: web.Request):
        params = FreezeUserParams(
            sharpei=self._settings.sharpei,
            passport_settings=self._settings.passport,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            freeze_settings=self._settings.freeze_settings,
            uid=self.get_required_parameter(request, 'uid'),
        )
        await freeze_user(params)
        return 'Done'

    async def archive_user(self, request: web.Request):
        params = ArchiveUserParams(
            sharpei=self._settings.sharpei,
            passport_settings=self._settings.passport,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            s3api_settings=self._settings.s3api,
            huskydb=self._huskydb,
            freeze_settings=self._settings.freeze_settings,
            uid=self.get_required_parameter(request, 'uid'),
        )
        await archive_user(params)
        return 'Done'

    async def clean_archive(self, request: web.Request):
        params = CleanArchiveParams(
            sharpei=self._settings.sharpei,
            passport_settings=self._settings.passport,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            s3api_settings=self._settings.s3api,
            freeze_settings=self._settings.freeze_settings,
            uid=self.get_required_parameter(request, 'uid'),
        )
        await clean_archive(params)
        return 'Done'

    async def purge_archive(self, request: web.Request):
        params = PurgeArchiveParams(
            sharpei=self._settings.sharpei,
            passport_settings=self._settings.passport,
            tvm_ids=self._settings.tvm,
            tvm_tickets=self._tvm,
            s3api_settings=self._settings.s3api,
            freeze_settings=self._settings.freeze_settings,
            uid=self.get_required_parameter(request, 'uid'),
        )
        await purge_archive(params)
        return 'Done'

    async def purge_transferred_user(self, request: web.Request):
        params = PurgeTransferredUserParams(
            sharpei=self._settings.sharpei,
            freeze_settings=self._settings.freeze_settings,
            huskydb=self._huskydb,
            uid=self.get_required_parameter(request, 'uid'),
        )
        params.limit = int(request.query.get('limit', params.limit))
        await purge_transferred_user(params)
        return 'Done'
