import ujson
from aiohttp.web import Response

from mail.python.theatre.roles.director import Director as DirectorBase
from mail.python.theatre.detail.tvm import TvmServiceTickets

from mail.shiva.stages.api.props.admin_api import AdminApi
from mail.shiva.stages.api.props.shard_api import ShardApi
from mail.shiva.stages.api.props.maildb_api import MaildbApi
from mail.shiva.stages.api.props.huskydb_api import HuskydbApi
from mail.shiva.stages.api.props.queuedb_api import QueuedbApi
from mail.shiva.stages.api.props.callmebackdb_api import CallmebackdbApi
from mail.shiva.stages.api.props.util_api import UtilApi
from mail.shiva.stages.api.props.shard.task import clean_tasks, HuskydbEngine, get_running_task_stats
from mail.shiva.stages.api.settings.app import Settings

from .shard_worker import ShardWorker


class Director(DirectorBase):
    def __init__(self, settings: Settings, huskydb: HuskydbEngine, worker_name: str):
        self.worker_name = worker_name
        self.huskydb = huskydb
        self.settings = settings
        shard_worker = ShardWorker(settings.shard_worker, huskydb, worker_name)
        tasks = [shard_worker]
        tvm_tickets = TvmServiceTickets.from_conf(settings.tvm)
        if tvm_tickets:
            tasks.append(tvm_tickets)
        super().__init__(tasks=tasks)

        self.admin_api = AdminApi(settings, huskydb)
        self.shard_api = ShardApi(
            settings=settings,
            huskydb=huskydb,
            shard_worker=shard_worker,
            tvm=tvm_tickets,
        )
        self.maildb_api = MaildbApi(settings, huskydb)
        self.huskydb_api = HuskydbApi(settings, huskydb, shard_worker)
        self.queuedb_api = QueuedbApi(settings, huskydb, shard_worker)
        self.callmebackdb_api = CallmebackdbApi(settings, huskydb, shard_worker)
        self.util_api = UtilApi(settings, huskydb, tvm_tickets)

    async def start(self):
        await clean_tasks(self.huskydb, self.worker_name)
        await super().start()

    async def stop(self):
        await super().stop()
        await clean_tasks(self.huskydb, self.worker_name)

    async def unistat_handler(self, _):
        task_stats = await get_running_task_stats(self.huskydb, self.worker_name)
        meters = [[f"running_{row['task']}_ammm", row['count']] for row in task_stats
                  if 'task' in row and 'count' in row]
        return Response(
            text=ujson.dumps(self.metrics() + meters + self.maildb_api.metrics()),
            content_type='application/json'
        )
