#include "agent_dialog.h"
#include "assassin.h"
#include "chars.h"
#include "libdaemon.h"
#include "mkshn.h"
#include "msg2html.h"
#include "spam.h"

#include <mail/so/spamstop/tools/so-clients/cmpl_flags.h>
#include <mail/so/spamstop/tools/so-clients/freemail.h>
#include <mail/so/spamstop/tools/so-clients/functional_clients/FreemailShClient.h>
#include <mail/so/spamstop/tools/so-clients/kshingle.h>
#include <mail/so/spamstop/tools/so-common/parsers.h>
#include <mail/so/spamstop/tools/so-common/safe_recode.h>
#include <mail/so/spamstop/tools/so-common/shconn.h>
#include <mail/so/spamstop/tools/so-common/so_answer.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>

#include <mail/so/spamstop/sp/rengine.h>
#include <mail/so/spamstop/sp/spamstop.h>
#include <mail/so/spamstop/sp/sptop.h>

#include <mail/so/libs/protect/protect.h>

#include <mlp/mail/aspam/domen_factors/lib/domen_factors.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/html/dehtml/dehtml.h>
#include <library/cpp/html/entity/htmlentity.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/langs/langs.h>
#include <library/cpp/neh/multiclient.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/charset/unidata.h>
#include <util/charset/wide.h>
#include <util/datetime/cputimer.h>
#include <util/folder/path.h>
#include <util/generic/iterator_range.h>
#include <util/generic/scope.h>
#include <util/network/sock.h>
#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/string/subst.h>
#include <util/system/datetime.h>
#include <util/system/mutex.h>
#include <util/system/tls.h>
#include <util/thread/singleton.h>

#include <contrib/deprecated/udns/udns.h>

#include <fstream>
#include <iostream>
#include <set>

#define MAX_BODYPARTS 30
#define MAX_BODYBYTES 50 * 1024

static const int dns_init_res = dns_init(0, 0);

static const TTrueConst<THashSet<TStringBuf>> stBlindLogins{
    "abuse",
    "mail",
    "postmaster",
    "sales",
    "info",
    "pr",
    "robot",
    "reply",
    "noreply",
    "no-reply",
    "no.reply",
};

static find_tag cptag({
                          {5, "CWORDS_5"},
                          {15, "CWORDS_15"},
                          {30, "CWORDS_30"},
                          {50, "CWORDS_50"},
                          {65, "CWORDS_65"},
                          {80, "CWORDS_80"},
                          {95, "CWORDS_95"},
                          {101, "CWORDS_100"},
                      },
                      "CWORDS");

static find_tag words_tag({
                              {1, "WCOUNT_1"},
                              {2, "WCOUNT_2"},
                              {5, "WCOUNT_5"},
                              {10, "WCOUNT_5_10"},
                              {30, "WCOUNT_10_30"},
                              {100, "WCOUNT_30_100"},
                          },
                          "WCOUNT");
static find_tag cwords_tag({
                               {0, "CWCOUNT_0"},
                               {1, "CWCOUNT_1"},
                               {2, "CWCOUNT_2"},
                               {5, "CWCOUNT_5"},
                               {10, "CWCOUNT_5_10"},
                               {30, "CWCOUNT_10_30"},
                               {100, "CWCOUNT_30_100"},
                           },
                           "CWCOUNT");
static find_tag len_tag({
                            {1000, "ALEN_1000"},
                            {5000, "ALEN_5000"},
                            {15000, "ALEN_15000"},
                            {50000, "ALEN_50000"},
                            {100000, "ALEN_100000"},
                        },
                        "ALEN");
static find_tag count_tag({
                              {1, "ACNT_1"},
                              {2, "ACNT_2"},
                              {5, "ACNT_5"},
                              {10, "ACNT_5_10"},
                              {30, "ACNT_10_30"},
                              {100, "ACNT_30_100"},
                          },
                          "ACNT");
static find_tag ilen_tag({
                             {1000, "IALEN_1000"},
                             {5000, "IALEN_5000"},
                             {15000, "IALEN_15000"},
                             {50000, "IALEN_50000"},
                             {100000, "IALEN_100000"},
                         },
                         "IALEN");
static find_tag icount_tag({
                               {1, "IACNT_1"},
                               {2, "IACNT_2"},
                               {5, "IACNT_5"},
                               {10, "IACNT_5_10"},
                               {30, "IACNT_10_30"},
                               {100, "IACNT_30_100"},
                           },
                           "IACNT");

TString recognize_decode(const TGlobalContext& globalContext, const TLog& logger, const TString& t, const TString& charset_name, ELanguage* lang, bool to_utf8 = false) {
    // timeval rec_start, rec_end;

    ECharset recognized = CODES_KOI8;
    recognized = globalContext.RecognizeEncoding(t);
    ECharset destEnc = (to_utf8 ? CODES_UTF8 : CODES_KOI8);

    *lang = LANG_ENG;
    if (recognized == CODES_ASCII)
        return t;
    else {
        TString sFiltered;

        if (recognized >= 0) // do not strip and recognize for CODES_UNKNOWN, CODES_UNSUPPORTED
        {
            THtmlStripper stripper(HSM_ENTITY | HSM_SPACE | HSM_TRASH, recognized);
            if (stripper(t, sFiltered)) {
                *lang = globalContext.RecognizeLanguage(sFiltered, recognized); //  t.c_str());
            }
        }

        if (!to_utf8)
            destEnc = GetDestinationEncoding(*lang);

        if (recognized >= 0) {
            try {
                return Recode(recognized, destEnc, t);
            } catch (...) {
                logger << TLOG_WARNING << "cannot recode " << t << ':' << CurrentExceptionMessage();
                return t;
            }
        } else {
            if (charset_name.empty())
                return t;
            else {
                try {
                    return Recode(EncodingHintByName(charset_name.c_str()), destEnc, t);
                } catch (...) {
                    logger << TLOG_WARNING << "cannot recode " << t << ':' << CurrentExceptionMessage();
                    return t;
                }
            }
        }
    }
}

inline TString
remove_quote(const TString& s) {
    TString r;
    size_t p = s.find_first_of("\"'");
    if (p != s.npos) {
        r = s.substr(0, p++);
        while (p < s.length()) {
            if (s[p] != '\"' && s[p] != '\'') {
                r += s[p];
            }
            ++p;
        }
        return r;
    }
    return s;
}

TString trigrams_decode(const TGlobalContext& GlobalContext, const mimepp::String& sSource, bool to_utf8 = false) {
    ECharset Code = GlobalContext.RecognizeEncoding({sSource.data(), sSource.size()});
    ECharset dest_enc = CODES_UTF8;
    if (!to_utf8)
        dest_enc = CODES_KOI8;

    TString s(MimeppString2StringBuf(sSource));
    try {
        return Recode(Code, dest_enc, s);
    } catch (...) {
        Syslog(TLOG_WARNING) << "cannot recode " << s << ':' << CurrentExceptionMessage();
        return s;
    }
}

inline TString
ass_decode_rfc2047(const TGlobalContext& GlobalContext, TLog logger, const mimepp::String& fldName, const mimepp::String& fldBody, TString& charset, bool* is_mimed = 0, ECharset dest_enc = CODES_KOI8, bool* bBrokenRFC = nullptr) {
    if (mimepp::Strcasecmp(fldName, "X-yandex-rpop-foldername") == 0) {
        mimepp::Base64Decoder decoder; // assume rpop folder always comes UTF encoded
        charset = koi_names[0];
        const auto decoded = decoder.decode(fldBody);
        if(const auto c = CharsetByName(utf_names[0]); c != CODES_UNKNOWN) {
            return char_decode(logger, {decoded.c_str(), decoded.size()}, c, dest_enc);
        } else {
            logger << (TLOG_WARNING) << "cannot find charset " << utf_names[0];
            return {decoded.c_str(), decoded.size()};
        }
    }

    if (fldBody.find("=?") != mimepp::String::npos) // if charset defined or just padding then demime and decode
    {
        if (is_mimed)
            *is_mimed = true;

        return real_decode_rfc2047(GlobalContext, logger, remove_quote({fldBody.c_str(), fldBody.size()}), charset, dest_enc, bBrokenRFC);
    } else {
        if (is_mimed)
            *is_mimed = false;

        charset = koi_names[0]; // if no charset defined in header assume KOI8R

#ifndef _MSC_VER
        if ((mimepp::Strcasecmp(fldName, "subject") == 0) || // for "subj" and "from" w/o charset try 3grams
            (mimepp::Strcasecmp(fldName, "from") == 0)) {
            return trigrams_decode(GlobalContext, fldBody, dest_enc == CODES_UTF8);
        } else // do not decode other fields w/o charset
#endif
        {
            return {fldBody.c_str(), fldBody.size()};
        }
        //        return ch_decode (s, charset);
    }
}

inline TString
ass_decode_rfc2047(const TGlobalContext& GlobalContext, TLog logger, mimepp::Field* fld, TString& charset, bool* is_mimed = 0, ECharset dest_enc = CODES_KOI8, bool* bBrokenRFC = nullptr) {
    return ass_decode_rfc2047(GlobalContext, std::move(logger), fld->fieldName(), fld->fieldBody().text(), charset, is_mimed, dest_enc, bBrokenRFC);
}

bool is_part_of_addr(unsigned char c) {
    c = tolower(c);
    return (isdigit(c) || (c >= 'a' && c <= 'f') || c == '.' || c == ':' || c == 'i' || c == 'p' || c == 'v');
}

TString find_address(const TStringBuf& src, TIpAddr& adr, bool bRcvd = true) {
    char bracket;
    size_t p = 0;
    size_t startp = 0, endBracket, atPos;
    TIpAddr addr;
    TString d(src);

    while ((p = d.find_first_of("\r\n\t")) != TString::npos)
        d[p] = ' ';
    p = 0;

    size_t byPos = d.rfind(" by ");

    if (bRcvd && (byPos == TString::npos))
        return "";

    int startBracket = bRcvd ? byPos : d.length() - 1;

    while (!addr.IsValid() && (--startBracket > 0)) {
        p = d.find_last_of("[(", startBracket);
        if (p == d.npos)
            p = d.rfind(" ", startBracket);

        if (p == d.npos)
            return "";

        startBracket = p;
        bracket = d[startBracket];

        ++p;
        while (p < d.length() && (d[p] == ' ' || d[p] == '\t'))
            ++p;

        endBracket = d.find(bracket == '(' ? ")" : bracket == '[' ? "]" : " ", p);

        atPos = d.find_last_of("@", endBracket); // for IPs like "(sysadmin@636.lv@78.128.87.150)"
        if (atPos != TString::npos && atPos > p)
            p = ++atPos;

        startp = p;

        int have_dot = 0;
        int have_colon = 0;
        while ((have_dot < 4) && (p < d.length()) && is_part_of_addr(d[p])) {
            if (d[p] == '.')
                ++have_dot;
            if (d[p] == ':') {
                if (have_dot == 3) // valid ipv4 already found, like: [10.156.251.33:48169]
                    break;
                ++have_colon;
            }
            ++p;
        }
        if (((have_dot == 3) && ((p - startp) > 6)) || (have_colon >= 2)) {
            if (p < d.length()) // then checkup for closing bracket for IP
            {
                if ((bracket == '(') && (d[p] != ')'))
                    continue;
                if ((bracket == '[') && (d[p] != ':') && (d[p] != ']'))
                    continue;
            }
            addr.FromString(d.substr(startp, p - startp).c_str());
            if (addr.IsValid()) {
                adr = addr;
                break;
            }
        }
    }

    if (addr.IsValid()) {
        return addr.ToDNSBLString();
    }

    return "";
}

TString get_rbl_rule_name(const TString& name, int num) {
    return TStringBuilder{} << name << '_' << num;
}

class prof {
public:
    prof(const TString& name)
        : fname(name)
        , isstop(false) {
    }
    ~prof() {
        stop();
    }
    void stop() {
        if (isstop)
            return;
        isstop = true;
    }

    TSimpleTimer timer;
    TString fname;
    bool isstop;
};

void rbl_stub_ng(const TString& ip, const NFuncClient::TRbl::TResponse& response,
                 TContext& context,
                 bool bFirstRcvd) {
    if (auto val = response.IsInCombinedBlackList()) {
        SpSetRule(&context, "COMBINEDBL_" + ToString(val));
        if (bFirstRcvd) {
            if (val == 4)
                SpSetRule(&context, "F_RBL4");
            else
                SpSetRule(&context, "F_RBLN");
        } else // do not log for very first rcvt, will be logged along with other rcvds
            SpAddStatStrInt(&context, ip.c_str(), val);
    }
}

TString
run_smbox(const TGlobalContext& GlobalContext, TLog logger, TRengine* gsp, mimepp::Field* fld, const TString& decoded, const TString& charset, bool is_mime) {
    mimepp::String addr;
    SpCheckField(gsp,
                 fld->fieldName().c_str(),
                 {decoded.c_str(), decoded.length()},
                 charset.c_str(),
                 true,
                 is_mime);

    mimepp::MailboxList& mbl = dynamic_cast<mimepp::MailboxList&>(fld->fieldBody());
    for (int j = 0; j < mbl.numMailboxes(); ++j) {
        mimepp::Mailbox& mb = mbl.mailboxAt(j);
        TString chr, name;

        name = ass_decode_rfc2047(GlobalContext, logger, "from", mb.encodedDisplayName(), chr); // run_smbox called only for "from"

        SpCheckField(gsp,
                     (fld->fieldName() + ":name").c_str(),
                     {name.c_str(), name.length()},
                     chr.c_str(),
                     true,
                     is_mime);
        if (!mb.domain().empty()) {
            addr = mb.localPart() + "@" + mb.domain();
            SpCheckField(gsp,
                         (fld->fieldName() + ":addr").c_str(),
                         {addr.c_str(), addr.length()},
                         chr.c_str(),
                         true,
                         is_mime);
        }
    }
    return {addr.c_str(), addr.size()};
}

void run_spaddr(TRengine* gsp, mimepp::Field* fld, const TString& decoded, const TString& charset, bool is_mime) {
    int classId;
    mimepp::String addr;

    SpCheckField(gsp,
                 fld->fieldName().c_str(),
                 {decoded.c_str(), decoded.length()},
                 charset.c_str(),
                 true,
                 is_mime);

    auto& to = dynamic_cast<mimepp::AddressList&>(fld->fieldBody());
    for (int i = 0; i < to.numAddresses(); ++i) {
        classId = to.addressAt(i).class_().id();

        switch (classId) {
            case mimepp::GROUP_CLASS: {
                auto& mbl = dynamic_cast<mimepp::Group&>(to.addressAt(i));
                for (int j = 0; j < mbl.mailboxList().numMailboxes(); ++j) {
                    mimepp::Mailbox& mbx = mbl.mailboxList().mailboxAt(j);
                    addr = mbx.localPart() + "@" + mbx.domain();
                    SpCheckField(gsp,
                                 (fld->fieldName() + ":addr").c_str(),
                                 {addr.c_str(), addr.length()},
                                 charset.c_str(),
                                 true,
                                 is_mime);
                    if ((stBlindLogins->find(MimeppString2StringBuf(mbx.localPart())) != stBlindLogins->end()) &&
                        mimepp::Strcasecmp(fld->fieldName(), "reply-to")) // do not count blind addresses for sender reply-to
                        SpIncrementBlindTo(gsp);
                }
                break;
            }
            case mimepp::MAILBOX_CLASS: {
                mimepp::Mailbox& mb = dynamic_cast<mimepp::Mailbox&>(to.addressAt(i));
                addr = mb.localPart() + "@" + mb.domain();
                SpCheckField(gsp,
                             (fld->fieldName() + ":addr").c_str(),
                             {addr.c_str(), addr.length()},
                             charset.c_str(),
                             true,
                             is_mime);

                if ((stBlindLogins->find(MimeppString2StringBuf(mb.localPart())) != stBlindLogins->end()) && mimepp::Strcasecmp(fld->fieldName(), "reply-to"))
                    SpIncrementBlindTo(gsp);

                break;
            }
            default: {
                SpCheckField(gsp,
                             (fld->fieldName() + ":addr").c_str(),
                             {to.addressAt(i).text().c_str(), to.addressAt(i).text().length()},
                             charset.c_str(),
                             false,
                             is_mime);

                size_t atPos = to.addressAt(i).text().find_first_of(" @");
                mimepp::String sTmpLogin;
                if (atPos == mimepp::String::npos)
                    sTmpLogin = to.addressAt(i).text();
                else if (atPos > 0)
                    sTmpLogin = to.addressAt(i).text().substr(0, atPos - 1);

                if (!sTmpLogin.empty() && (stBlindLogins->find(MimeppString2StringBuf(sTmpLogin)) != stBlindLogins->end()) && mimepp::Strcasecmp(fld->fieldName(), "reply-to"))
                    SpIncrementBlindTo(gsp);
            }
        }
    }
}

TString
run_ruleset(const TGlobalContext& globalContext,
            const TLog& logger,
            TRengine* gsp,
            mimepp::Field* fld,
            const TString& decoded,
            const TString& charset,
            bool is_mime) {
    TString from = "";
    TString fname(fld->fieldName().c_str());
    fname.to_lower();

    if (fname == "to" || fname == "reply-to" || fname == "cc")
        run_spaddr(gsp, fld, decoded, charset, is_mime);
    else if (fname == "from")
        from = run_smbox(globalContext, logger, gsp, fld, decoded, charset, is_mime);
    else
        SpCheckField(gsp,
                     fld->fieldName().c_str(),
                     {decoded.c_str(), decoded.length()},
                     charset.c_str(),
                     true,
                     is_mime);

    return from;
}

static bool NeedInvertResolutionFotOutAllYaNotCaptcha(const TCheckedMessage& checkedMessage,
                                                      const TSoConfig& config,
                                                      const TRulesContext& rulesContext) {
    return config.fWebMail &&
        checkedMessage.spClass == TSpClass::SPAM &&
        rulesContext.IsRuleWorked("RCP_ALL_YA") &&
        !rulesContext.IsRuleWorked("NEED_CAPTCHA");
}

void visit_text(const TGlobalContext& GlobalContext, const NHtmlSanMisc::TDoc& part, CProf& prof, TContext& context, const TLog& logger) {
    CProfItemGuard allProf(prof.Prof("all"));
    TRengine *gsp = &context;

    TStringBuf type = part.contentType;
    TStringBuf subtype;
    if (!type.TrySplit('/', type, subtype) || !subtype) {
        SpSetRule(gsp, "UNDEF_TEXT_CONTENT_TYPE");
        subtype = part.htmlBody ? TStringBuf("html") : TStringBuf("plain");
    }

    const TString charset_name = "utf8";
    const TSpMesType tp = AsciiEqualsIgnoreCase(subtype, "html") ? spTextHtml : spTextPlain;

    prof.Prof("rec_decode").Start();
    const TString body = recognize_decode(GlobalContext,
                                          logger,
                                          part.htmlBody ? part.htmlBody : part.PureBody.Original,
                                          charset_name,
                                          &context.shinglesCounter->language,
                                          true);
    prof.Prof("rec_decode").Stop();

    const char *clear_utf8_text = nullptr;
    TBodyPartProperty prop = set_prop(part.contentType,
                                      part.attachName,
                                      TString{},
                                      TString{},
                                      TString{},
                                      false,
                                      charset_name,
                                      context.shinglesCounter->language);

    TString clear_text;
    {
        gsp->CheckBody(prof.Sub("CheckBody"), body.c_str(), (int) body.size(), tp, clear_text, &clear_utf8_text,
                       &prop);
    }
    free_prop(prop);

    if (clear_text) {
        context.shinglesCounter->feed(clear_text);
    }
}

void visit_text(const TGlobalContext& GlobalContext, const mimepp::Entity* m, CProf& prof, TContext& context, const TLog& logger) {
    CProfItemGuard allProf(prof.Prof("all"));
    TString tout;
    TRengine* gsp = &context;
    bool hasCT = m->headers().hasField("Content-Type");
    bool hasCTE = m->headers().hasField("Content-Transfer-Encoding");
    int type = mimepp::MediaType::TEXT;
    mimepp::String subtype = "plain"; //DwMime::kSubtypePlain;

    if (hasCT) {
        if (m->headers().contentType().typeAsEnum())
            type = m->headers().contentType().typeAsEnum();
        subtype = m->headers().contentType().subtype();
    }
    if (type == mimepp::MediaType::TEXT && subtype.empty()) {
        SpSetRule(gsp, "UNDEF_TEXT_CONTENT_TYPE"); //"FAKE_TEXT_CONTENT_TYPE" : "UNDEF_TEXT_CONTENT_TYPE");
        subtype = (m->body().getString().find("<a href") != mimepp::String::npos) ? "html" : "plain";
    }

    if (type != mimepp::MediaType::TEXT || !(mimepp::Strcasecmp(subtype, "plain") == 0 ||
                                             mimepp::Strcasecmp(subtype, "html") == 0 ||
                                             subtype.find("rfc822") != mimepp::String::npos))
        return;

    if (hasCTE) {
        int errCode = 0;
        const auto& bodyString = m->body().getString();
        const auto& teString = m->headers().contentTransferEncoding().type();
        {
            CProfItemGuard g(prof.Prof("decode_transfer"));
            const auto& s = decode_transfer(bodyString, teString, &errCode);
            tout.assign(s.c_str(), s.length());
        }

        if (errCode)
            SpSetRule(gsp, "BROKEN_B64");
    } else {
        const auto& s = m->body().getString();
        tout.assign(s.c_str(), s.length());
    }

    if (tout.length() > MAX_BODYBYTES)
        tout.erase(MAX_BODYBYTES - 1);

    const auto charset_name = find_charset_name(*m);
    TSpMesType tp = ((mimepp::Strcasecmp(subtype, "html") == 0) ? spTextHtml : spTextPlain);

    {
        CProfItemGuard g(prof.Prof("rec_decode"));
        tout = recognize_decode(GlobalContext, logger, tout, {charset_name.c_str(), charset_name.size()}, &context.shinglesCounter->language, true);
    }
    TString clear_text;
    const char* clear_utf8_text = nullptr;
    TBodyPartProperty prop;
    set_prop(GlobalContext, logger, m, {charset_name.c_str(), charset_name.size()}, prop, context.shinglesCounter->language);

    {
        gsp->m_cur->BodyTextPresent |= !tout.empty();
        gsp->CheckBody(prof.Sub("CheckBody"), tout.c_str(), (int)tout.size(), tp, clear_text, &clear_utf8_text,
                    &prop);
    }
    free_prop(prop);

    if (clear_text) {
        context.shinglesCounter->feed(clear_text);
    }
}

bool crc_type(int t, const char* st) {
    if (t == mimepp::MediaType::APPLICATION ||
        t == mimepp::MediaType::IMAGE ||
        t == mimepp::MediaType::AUDIO ||
        t == mimepp::MediaType::VIDEO)
        return true;

    if ((t == mimepp::MediaType::TEXT) &&
        (AsciiCompareIgnoreCase(st, "plain") != 0) && (AsciiCompareIgnoreCase(st, "html") != 0) &&
        (AsciiCompareIgnoreCase(st, "richtext") != 0) && (AsciiCompareIgnoreCase(st, "enriched") != 0))
        return true;

    return false;
}

void report_structure(const TGlobalContext& GlobalContext, const mimepp::Entity* m, TContext& context, TLog logger) {
    auto charset_name = find_charset_name(*m);
    TString ct, rawct;
    TString name;
    TString chr;
    bool need_crc = false;
    bool image_att = false;
    TRengine* gsp = &context;
    int ctCounter = 0;
    int cteCounter = 0;

    for (int i = 0; i < m->headers().numFields(); i++) {
        if (mimepp::Strcasecmp(m->headers().fieldAt(i).fieldName(), "content-type") == 0)
            ctCounter++;
        if (mimepp::Strcasecmp(m->headers().fieldAt(i).fieldName(), "content-transfer-encoding") == 0)
            cteCounter++;
    }

    if (ctCounter > 1)
        SpSetRule(gsp, "CT_CONFLICT");
    if (cteCounter > 1)
        SpSetRule(gsp, "CTE_CONFLICT");

    if (m->headers().hasField("Content-Type")) {
        rawct = m->headers().fieldBody("Content-Type").text().c_str();
        ct = decode_rfc2047(GlobalContext, logger, m->headers().contentType().text());
        name = decode_rfc2047(GlobalContext, logger, m->headers().contentType().name());
        need_crc = crc_type(m->headers().contentType().typeAsEnum(),
                            m->headers().contentType().subtype().c_str());
        image_att = m->headers().contentType().typeAsEnum() == mimepp::MediaType::IMAGE;
    }
    TString disposition;
    TString file;
    TVector<TArciveItem> archive_contents;
    if (m->headers().hasField("Content-Disposition") || is_archive(name)) {
        disposition = decode_rfc2047(GlobalContext, logger, m->headers().contentDisposition().type());
        file = decode_rfc2047(GlobalContext, logger, m->headers().contentDisposition().filename());
        if (file.empty())
            file = name;
        const char* arcSuffix = is_archive(file);
        if (arcSuffix) {
            mimepp::String data;
            int errCode = 0;
            decode_bin_transfer(m->body().getString().c_str(),
                                m->headers().contentTransferEncoding().type().c_str(),
                                &data,
                                &errCode);
            if (!data.empty()) {
                // timeval rec_start;
                // timeval rec_end;
                const TInstant& rec_start = Now();
                // gettimeofday (&rec_start, 0);
                GetArchiveContents(logger, data.c_str(), data.size(), arcSuffix, context.queueID.c_str(), gsp, &archive_contents, GlobalContext.Recoder.Get());
                //                bool need_more_data = (arc_res & NArchive::ArchParseNoEof);
                const TInstant& rec_end = Now();
                // gettimeofday (&rec_end, 0);

                auto prio = TLOG_DEBUG;
                // double time_took = ((double)(rec_end - rec_start)) / 1000000.;
                double time_took = (rec_end - rec_start).MicroSeconds() / 1000000.0;
                if (time_took > 0.1)
                    prio = TLOG_ERR;
                logger << prio << "arc contents took %5.2f sec" << time_took;
            }
        }
    }

    bool b64 = false;
    TString cte;
    if (m->headers().hasField("Content-Transfer-Encoding")) {
        b64 = m->headers().contentTransferEncoding().asEnum() == mimepp::TransferEncodingType::BASE64;
        cte = m->headers().contentTransferEncoding().type().c_str();
    }
    TBodyPartProperty prop;
    memset(&prop, 0, sizeof(prop));
    prop.pct_raw = rawct.empty() ? 0 : rawct.c_str();
    prop.pctype = ct.empty() ? 0 : ct.c_str();
    prop.pcte = cte.empty() ? 0 : cte.c_str();
    prop.pcd = disposition.empty() ? 0 : disposition.c_str();
    prop.pcharset = charset_name.empty() ? 0 : charset_name.c_str();
    prop.pname = name.empty() ? 0 : name.c_str();
    prop.pfilename = file.empty() ? 0 : file.c_str();
    prop.fBase64 = b64;
    if (!archive_contents.empty())
        prop.arc_files = &archive_contents;
    const TString b(m->body().getString().c_str());
    if (!b64) {
        mimepp::QuotedPrintableDecoder qDecoder;
        mimepp::String decodedText;
        if (cte.length() && (cte == "quoted-printable"))
            decodedText = qDecoder.decode(b.c_str());

        if (qDecoder.errorDetected() || decodedText.empty())
            SpCheckBodyPart(gsp, b.c_str(), (int)b.length(), &prop);
        else {
            SpCheckBodyPart(gsp, decodedText.c_str(), (int)decodedText.length(), &prop);
        }
    } else {
        SpCheckBodyPart(gsp, 0, 0, &prop);
    }
    if (need_crc) {
        if (!image_att) {
            if (b.length() > context.max_len) {
                context.max_len = b.length();
            }
            context.att_count++;
        } else {
            if (b.length() > context.i_max_len) {
                context.i_max_len = b.length();
            }
            context.i_att_count++;
        }
        context.calc_crc(b);
    }
    //cout<<"\nXXXX\nct: "<<ct<<" b "<<b<<endl;
}

void free_prop(TBodyPartProperty& prop) {
    if (prop.pctype) {
        free((char*)prop.pctype);
        prop.pctype = nullptr;
    }
    if (prop.pcte) {
        free((char*)prop.pcte);
        prop.pcte = nullptr;
    }
    if (prop.pcd) {
        free((char*)prop.pcd);
        prop.pcd = nullptr;
    }
    if (prop.pcharset) {
        free((char*)prop.pcharset);
        prop.pcharset = nullptr;
    }
    if (prop.pname) {
        free((char*)prop.pname);
        prop.pname = nullptr;
    }
    if (prop.pfilename) {
        free((char*)prop.pfilename);
        prop.pfilename = nullptr;
    }
}

TBodyPartProperty set_prop(const TString& contentType,
                           const TString& partName,
                           const TString& disposition,
                           const TString& filename,
                           const TString& contentTransferEncoding,
                           bool b64,
                           const TString& charset_name,
                           ELanguage lang) {

    TBodyPartProperty prop;

    if(contentType) {
        prop.pctype = strdup(contentType.c_str());
    }

    if(partName) {
        prop.pname = strdup(partName.c_str());
    }

    if(disposition) {
        prop.pcd = strdup(disposition.c_str());
    }

    if(filename) {
        prop.pfilename = strdup(filename.c_str());
    }

    if(contentTransferEncoding) {
        prop.pcte = strdup(contentTransferEncoding.c_str());
    }

    if(charset_name) {
        prop.pcharset = strdup(charset_name.c_str());
    }

    prop.fBase64 = b64;
    prop.lang = lang;

    return prop;
}

void set_prop(const TGlobalContext& GlobalContext, const TLog& logger, const mimepp::Entity* m, const TString& charset_name, TBodyPartProperty& prop, ELanguage lang) {
    TString ct;
    TString name;

    if (m->headers().hasField("Content-Type")) {
        ct = decode_rfc2047(GlobalContext, logger, m->headers().contentType().type());
        name = decode_rfc2047(GlobalContext, logger, m->headers().contentType().name());
    }

    TString disposition;
    TString file;
    if (m->headers().hasField("Content-Disposition")) {
        disposition = decode_rfc2047(GlobalContext, logger, m->headers().contentDisposition().type());
        file = decode_rfc2047(GlobalContext, logger, m->headers().contentDisposition().filename());
    }

    bool b64 = false;
    TString cte;
    if (m->headers().hasField("Content-Transfer-Encoding")) {
        b64 = m->headers().contentTransferEncoding().asEnum() == mimepp::TransferEncodingType::BASE64;
        cte = m->headers().contentTransferEncoding().type().c_str();
    }
    prop = set_prop(ct, name, disposition, file, cte, b64, charset_name, lang);
}

bool text_type_and_subtype(const mimepp::Entity* m) {
    if (m->headers().hasField("Content-Type")) {
        return (m->headers().contentType().typeAsEnum() == mimepp::MediaType::TEXT) &&
               ((mimepp::Strcasecmp(m->headers().contentType().subtype(), "plain") == 0) ||
                (mimepp::Strcasecmp(m->headers().contentType().subtype(), "html") == 0));
    }
    return false;
}

void visit_msg(const TGlobalContext& GlobalContext, const mimepp::Entity* m, CProf& prof, bool report_text, TContext& context, const TSoConfig& config, const TLog& logger, TSessionCache& cache) {
    CProfItemGuard allProf(prof.Prof("all"));
    context.deep_level++;
    if (context.deep_level > 50)
        return;

    int type = mimepp::MediaType::TEXT;
    mimepp::String subtype = "plain";
    mimepp::String tmpFrom, tmpID;

    if (m->headers().hasField("Content-Type")) {
        // check for no-rfc boundary
        mimepp::String sType = m->headers().contentType().getString();
        if (sType.length()) {
            mimepp::String sBoundary = Parseout(sType, "boundary=\"", "\"");
            if (sBoundary.find_first_of("$#@%^*;{}[]") != mimepp::String::npos)
                SpSetRule(&context, "NO_RFC1341");
        }

        if (m->headers().contentType().typeAsEnum()) // do not overwrite default type TEXT if empty CT
            type = m->headers().contentType().typeAsEnum();
        subtype = m->headers().contentType().subtype();

        if (type == mimepp::MediaType::MULTIPART) {
            if (m->body().numBodyParts() < 1) {
                m->headers().contentType().setType("text");
                m->headers().contentType().setSubtype("plain");
                type = mimepp::MediaType::TEXT;
                subtype = "plain";
            }

            if (m->body().numBodyParts() > MAX_BODYPARTS) {
                CProfItemGuard g(prof.Prof("concatbparts"));
                if (m->headers().hasField("From"))
                    tmpFrom = m->headers().from().getString();
                if (m->headers().hasField("Message-Id"))
                    tmpID = m->headers().messageId().getString();

                logger << TLOG_WARNING << "%sMultipart limit exceeded, nParts=" << m->body().numBodyParts() << " From=" << tmpFrom << " ID=" << tmpID;

                int tmpType = mimepp::MediaType::TEXT,
                    tmpCTE = mimepp::TransferEncodingType::BINARY;
                mimepp::String tmpSubtype,
                    sNewPart;
                mimepp::BodyPart* currentPart;
                mimepp::Base64Decoder dec;

                for (int i = m->body().numBodyParts() - 1; i > 0; i--) {
                    tmpType = mimepp::MediaType::TEXT;
                    currentPart = &m->body().bodyPartAt(i);
                    if (currentPart->headers().hasField("Content-Type")) {
                        tmpType = currentPart->headers().contentType().typeAsEnum();
                        tmpSubtype = currentPart->headers().contentType().subtype();
                    }

                    if ((tmpType == mimepp::MediaType::TEXT) && (mimepp::Strcasecmp(tmpSubtype, "PLAIN") == 0)) {
                        if (currentPart->headers().hasField("Content-Transfer-Encoding"))
                            tmpCTE = currentPart->headers().contentTransferEncoding().asEnum();

                        if (sNewPart.length() < MAX_BODYBYTES) {
                            if (tmpCTE == mimepp::TransferEncodingType::BASE64)
                                sNewPart += dec.decode(currentPart->body().getString());
                            else
                                sNewPart += currentPart->body().getString();
                        }
                        m->body().removeBodyPartAt(i);
                    }
                }

                if (!sNewPart.empty()) {
                    currentPart = &m->body().bodyPartAt(0);
                    currentPart->headers().deleteAllFields();
                    currentPart->headers().contentType().setString("text/plain");
                    currentPart->headers().parse();
                    currentPart->setString(sNewPart);
                    currentPart->parse();
                    SpSetRule(&context, "PARTS_10");
                }
            }
        }
    }

    if (type == mimepp::MediaType::MULTIPART) {
        if (mimepp::Strcasecmp(subtype, "alternative") == 0) {
            int last_part = m->body().numBodyParts() - 1;
            int show_text_part = -1;
            for (int i = 0; i < m->body().numBodyParts(); ++i) {
                mimepp::BodyPart& part = m->body().bodyPartAt(i);
                visit_msg(GlobalContext, &part, prof, i == last_part, context, config, logger, cache);
                if (text_type_and_subtype(&part))
                    show_text_part = i;
            }
            if ((last_part != show_text_part) && (show_text_part != -1)) {
                visit_text(GlobalContext, &(m->body().bodyPartAt(show_text_part)), prof.Sub("visit_text"), context, logger);
            }
        } else {
            for (int i = 0; i < m->body().numBodyParts(); ++i) {
                mimepp::BodyPart& part = m->body().bodyPartAt(i);
                visit_msg(GlobalContext, &part, prof, report_text, context, config, logger, cache);
            }
        }

        if (m->body().preamble().length()) {
            mimepp::BodyPart preamble(m->body().preamble());
            preamble.parse();
            //            visit_text (&preamble, vTokens, key);
            SpCheckField(&context, "preamble", {m->body().preamble().c_str(), m->body().preamble().length()}, 0, true, 0);
        }

        if (m->body().epilogue().length()) {
            mimepp::BodyPart epilogue(m->body().epilogue());
            epilogue.parse();
            {
                visit_text(GlobalContext, &epilogue, prof.Sub("visit_text"), context, logger);
            }
            SpCheckField(&context, "epilogue", {m->body().epilogue().c_str(), m->body().epilogue().length()}, 0, true, 0);
        }
    } else {
        if (type == mimepp::MediaType::MESSAGE) {
            if (m->body().message()) // if part has no body, then do not check
            {
                int nHeadersInMessage = 0;
                if (mimepp::Strcasecmp(subtype, "rfc822") == 0)
                    nHeadersInMessage = report_headers(GlobalContext, m->body().message(), prof.Sub("report_headers"), false, context, false, config, logger, cache);

                mimepp::Message* msg = m->body().message();
                if (nHeadersInMessage == 0) // fake attached message, will decode and parse as text
                {
                    SpSetRule(&context, "RFC_PART_ERR");

                    bool partHasCTE = m->headers().hasField("Content-Transfer-Encoding");

                    mimepp::BodyPart bPart;
                    if (partHasCTE) {
                        bPart.headers().contentTransferEncoding().setString(m->headers().contentTransferEncoding().type());
                        bPart.headers().contentTransferEncoding().parse();
                    }

                    bPart.headers().contentType().setString("text/html");
                    bPart.headers().contentType().parse();

                    bPart.body().setString(m->body().getString());
                    bPart.body().parse();

                    visit_text(GlobalContext, &bPart, prof.Sub("visit_text"), context, logger);
                } else
                    visit_msg(GlobalContext, msg, prof, report_text, context, config, logger, cache);
            }
        } else {
            if (type == mimepp::MediaType::TEXT) {
                if (m->body().message() && (mimepp::Strcasecmp(subtype, "rfc822-headers") == 0))
                    report_headers(GlobalContext, m->body().message(), prof.Sub("report_headers"), false, context, false, config, logger, cache);
                else {
                    report_structure(GlobalContext, m, context, logger);
                    if (report_text) {
                        visit_text(GlobalContext, m, prof.Sub("visit_text"), context, logger);
                    }
                }
            } else
                report_structure(GlobalContext, m, context, logger);
        }
    }
}

void report_rcvds(const mimepp::Entity* m, TContext& context) {
    TRengine& gsp = context;

    gsp.InitHeader();
    const mimepp::Headers& h = m->headers();
    for (int i = 0; i < h.numFields(); ++i) {
        const auto& header = h.fieldAt(i);
        const TStringBuf name = MimeppString2StringBuf(header.fieldName());

        if(AsciiEqualsIgnoreCase(name, "x-skipped-received") ||
           AsciiEqualsIgnoreCase(name, "message-id") ||
           AsciiHasPrefixIgnoreCase(name, "received") ||
           AsciiHasPrefixIgnoreCase(name, "x-yandex")) {
            const TStringBuf value = MimeppString2StringBuf(header.fieldBody().text());
            gsp.CheckField(name, value, nullptr, true, 0);
        }
    }
}

int report_headers(const TGlobalContext& GlobalContext,
                   const mimepp::Entity* m,
                   CProf& prof,
                   bool first_header,
                   TContext& context,
                   bool bRecheckFRNR,
                   const TSoConfig& config,
                   const TLog& logger,
                   TSessionCache& cache) {
    CProfItemGuard profAll(prof.Prof("all"));
    TString baddr;
    //    string from = "";
    TRengine* gsp = &context;

    if (!gsp)
        return 0;

    mimepp::String sWord;
    TSpHeaderFields spField;
    const CodePage* cp = CodePageByCharset(CODES_KOI8);

    TString charset;
    gsp->InitHeader();
    mimepp::Headers* h = &m->headers();
    TString f;
    //    string suid("dummy");
    context.reset_rs();
    TIpAddr rec_addr;
    bool bFirstRcvd = first_header; // will checkup the very first rcvd, ignore all attaches
    TSet<TString> setIPs;
    TSet<TString>::iterator itIP;
    mimepp::String sFldName;
    mimepp::String attachedFrom;
    mimepp::String attachedMsgId;
    bool bCodingBroken;
    bool bFromFldDone = false;
    bool bMsgidFldDone = false;
    int nHeaderFields = h->numFields();

    TVector<TString> ipsToCheckBl;

    NJson::TJsonValue headersJson(NJson::JSON_ARRAY);
    auto& headers = headersJson.GetArraySafe();


    for (int i = 0; i < nHeaderFields; ++i) {
        mimepp::Field* const fld = &h->fieldAt(i);
#ifdef DO_PROF
        prof gb("header");
#endif
        bool is_mimed = false;
        TString res;

        if (mimepp::Strcasecmp(fld->fieldName(), "subject") == 0) {
            SpCheckField(gsp, "RawSubject",
                         MimeppString2StringBuf(fld->fieldBody().text()),
                         nullptr, 0, 0);
        }

        bCodingBroken = false;
        f = ass_decode_rfc2047(GlobalContext, logger, fld, charset, &is_mimed, CODES_KOI8, &bCodingBroken);

        {
            auto field = ass_decode_rfc2047(GlobalContext, logger, fld, charset, &is_mimed, CODES_UTF8, &bCodingBroken);
            auto &header = headers.emplace_back(NJson::JSON_ARRAY).GetArraySafe();
            header.emplace_back(MimeppString2StringBuf(fld->fieldName()));
            header.emplace_back(std::move(field));
        }

        if (bCodingBroken)
            SpSetRule(gsp, "HDR_NORFC");

        spField = spHFieldUnknown;
        if (mimepp::Strcasecmp(fld->fieldName(), "from") == 0) {
            CProfItemGuard g(prof.Prof("from"));
            spField = spHFieldFrom;

            // also check for corrupt "from", like full-encoded row From: =?utf-8?Q?=22PulsarWorld=22_=3Ccontacts=40pulsarworld=2Enet=3E?=
            if ((fld->fieldBody().text().find('@') == mimepp::String::npos) && (f.find('@') != TString::npos)) {
                const mimepp::String fldName("from");
                const mimepp::String fldBody(f.c_str());
                fld->setFieldBody(fld->createFieldBody(fldName, fldBody, nullptr));
                fld->setFieldName(fldName);
                fld->fieldBody().parse();
                SpSetRule(gsp, "FR_NORFC");
            }

            if (!first_header) // detect attachedFrom for shingle36 check
            {
                mimepp::MailboxList& from = h->from();
                for (int j = 0; j < from.numMailboxes(); ++j) {
                    mimepp::Mailbox& mb = from.mailboxAt(j);
                    attachedFrom.assign(mb.localPart());
                    attachedFrom += "@";
                    attachedFrom += mb.domain();
                    break;
                }
                attachedMsgId = h->messageId().getString();
            } else {
                if (bFromFldDone)
                    SpSetRule(gsp, "NO_RFC5322"); // multiple From
                else
                    bFromFldDone = true;
            }
        } else if (mimepp::Strcasecmp(fld->fieldName(), "to") == 0)
            spField = spHFieldTo;
        else if (mimepp::Strcasecmp(fld->fieldName(), "subject") == 0)
            spField = spHFieldSubject;
        else if (mimepp::Strcasecmp(fld->fieldName(), "cc") == 0)
            spField = spHFieldCc;
        else if (mimepp::Strcasecmp(fld->fieldName(), "message-id") == 0) {
            if (first_header) {
                if (bMsgidFldDone)
                    SpSetRule(gsp, "NO_RFC5322_M"); // miltiple msgID
                else
                    bMsgidFldDone = true;
            }
        }

        if (spField) // report words for name check
        {
            CProfItemGuard g(prof.Prof("checkword"));
            for (auto splitIt : StringSplitter(f).SplitBySet(" ,:;\"'<>/")) {
                const auto tok = splitIt.Token();

                if (tok.Contains('@'))
                    continue;

                const auto lowered = ToLower(TString{tok}, *cp);

                gsp->CheckFieldWord(spField, lowered.c_str(), lowered.size());
            }

            if (spField == spHFieldSubject) {
                SpPrepareSubject(gsp, f.c_str(), f.length());
            }
        }

        if (first_header) // do not check IPs for attached mails
        {
            CProfItemGuard g(prof.Prof("first_header"));
            TString baddr;
            rec_addr.FromString("");

            if (mimepp::Strcasecmp(fld->fieldName(), "x-qip-sender") == 0) {
                CProfItemGuard g(prof.Prof("qip"));
                auto revaddr = fld->fieldBody().text();
                for (int j = revaddr.length() - 1; j >= 0; j--)
                    baddr += revaddr[j];
                rec_addr.FromString(fld->fieldBody().text().c_str());

                if (setIPs.find(rec_addr.ToString()) == setIPs.end())
                    setIPs.insert(rec_addr.ToString());
                else // address checked already
                {
                    baddr.clear();
                    rec_addr.FromString("");
                }
            }

            if (mimepp::Strcasecmp(fld->fieldName(), "received") == 0) {
                CProfItemGuard g(prof.Prof("received"));

                baddr = find_address(f, rec_addr);
                if (baddr.empty() || baddr.length() == 0) {
                    CProfItemGuard g(prof.Prof("baddr.empty"));
                    SpSetRule(gsp, "NO_IP_IN_RCVD");
                    bFirstRcvd = false;
                    if (!context.rs_got_nottrusted()) {
                        fld->setFieldName("X-Skipped-Received");
                        fld->assemble();
                    }
                    res = run_ruleset(GlobalContext, logger, gsp, fld, f, charset, is_mimed);
                    continue;
                }

                if (!context.rs_got_nottrusted()) {
                    CProfItemGuard g(prof.Prof("ip_matcher.match"));
                    if (GlobalContext.ip_matcher.match(rec_addr) || GlobalContext.local_matcher.match(rec_addr)) {
                        // for intranet IPs call SpCheckField just for header logging
                        SpCheckField(gsp, "X-Skipped-Received",
                                     {fld->fieldBody().getString().c_str(),
                                      fld->fieldBody().getString().length()},
                                     charset.c_str(),
                                     true,
                                     is_mimed);
                        bRecheckFRNR = true; // must recheck if any leading rcvds skipped
                        continue;
                    } else {
                        context.rs.got_nottrusted = true;
                    }
                }

                if (bFirstRcvd) {
                    CProfItemGuard g(prof.Prof("bFirstRcvd"));
                    bFirstRcvd = false;
                    if ((fld->fieldBody().text().find("by cso-yandex.ru") == mimepp::String::npos) && config.add_rcvd)
                        logger << TLOG_WARNING << "%sFirst rcvd [%d] is not ours 4: %s" << i << fld->fieldBody().text();
                    else {
                        if (h->hasField("IY-FRNR") && !bRecheckFRNR) // our header valid
                            SpSetRule(gsp, "FRNR");
                        //                    }

                        // must check fake resolv and frnr even for external rcvd
                        // direct resolve must correspond with back
                        //                  {
                        mimepp::String strField = fld->fieldBody().getString();
                        size_t iSquareBracket = strField.rfind('[');
                        size_t iRoundBracket = strField.rfind('(', iSquareBracket);

                        if (iRoundBracket != mimepp::String::npos &&
                            iSquareBracket != mimepp::String::npos &&
                            (iSquareBracket - iRoundBracket > 1)) // both brackets found and host not empty
                        {
                            mimepp::String sHost = strField.substr(iRoundBracket + 1, iSquareBracket - iRoundBracket - 2); // 2 for bracket and space
                            if (sHost.length()) {
                                sHost.trim();

                                TIpAddr rcvdIP, resolvedIP;
                                find_address(strField.c_str(), rcvdIP);

                                // do not check direct resolve if no backward
                                // also do not check IPv6 .... because it mismatches too often :)
                                if ((sHost.compare("unknown") != 0) && rcvdIP.IsValid()) {
                                    CProfItemGuard g(prof.Prof("dns"));
                                    TUdnsContextHolder ctx("def");
                                    if (ctx.Open(logger)) {
                                        TString ctxUnistat;
                                        TVector<TString> resolvedAddrs;
                                        int errorCode{};
                                        if (ctx.ResolveHost(logger, sHost.c_str(), &ctxUnistat, resolvedAddrs, &errorCode)) {
                                            size_t j;

                                            for (j = 0; j < resolvedAddrs.size(); j++) {
                                                resolvedIP.FromString(resolvedAddrs[j].c_str());
                                                if (rcvdIP == resolvedIP)
                                                    break;
                                            }

                                            if (j == resolvedAddrs.size()) {
                                                SpSetRule(gsp, rcvdIP.IsIpv6() ? "FAKE_RESOLV_V6" : "FAKE_RESOLV");
                                                logger << TLOG_WARNING << "%sFAKE_RESOLV '%s' for '%s'" << context.queueID << rcvdIP.ToString() << sHost;
                                            }
                                        } else {
                                            if (bRecheckFRNR && (errorCode == DNS_E_NXDOMAIN)) {
                                                SpSetRule(gsp, "FRNR");
                                                logger << TLOG_NOTICE << "%sFRNR rechecked for '%s'" << sHost;
                                            }
                                        }
                                    }
                                } else
                                    logger << TLOG_WARNING << "%s invalid host or IP at '%s'" << fld->fieldBody().getString();
                            } else
                                logger << TLOG_WARNING << "%sRESOLV for empty host from '%s'" << fld->fieldBody().getString();
                        }
                    }
                }

                if (!GlobalContext.intranet_matcher.match(rec_addr))
                    context.rs.got_notintranet = true;

                context.rs.rec_count++;

                if (setIPs.find(rec_addr.ToString()) == setIPs.end())
                    setIPs.insert(rec_addr.ToString());
                else // ignore IPs already checked
                {
                    baddr.clear();
                    rec_addr.FromString("");
                }
                if (config.dumbMode < 4) {

                    CProfItemGuard g(prof.Prof("dnsbl"));

                    const auto fieldName = MimeppString2StringBuf(fld->fieldName());
                    if (rec_addr.IsValid() && AsciiEqualsIgnoreCase(fieldName, "received") && !GlobalContext.local_matcher.match(rec_addr)) {
                        ipsToCheckBl.emplace_back(rec_addr.ToString());
                    }
                }
            }
        }
        {
            CProfItemGuard g(prof.Prof("ruleset"));
            res = run_ruleset(GlobalContext, logger, gsp, fld, f, charset, is_mimed);
        }
    }
    if(ipsToCheckBl && GlobalContext.Pools && GlobalContext.Pools->RblRequester) {
        SortUnique(ipsToCheckBl);
        for (auto it = ipsToCheckBl.begin(); it != ipsToCheckBl.end();) {
            TMaybe<NFuncClient::TRbl::TResponse> cachedResponse = cache.RblCache.Get(*it, NFuncClient::TRbl::LISTS_ONLY, true);
            if (cachedResponse.Defined()) {
                rbl_stub_ng(*it, *cachedResponse, context, bFirstRcvd);
                it = ipsToCheckBl.erase(it);
            } else {
                ++it;
            }
        }
        if (ipsToCheckBl) {
            auto responseByIpOrOerror = GlobalContext.Pools->RblRequester->Perform(ipsToCheckBl,
                                                                                   config.UnistatPrefix,
                                                                                   NFuncClient::TRbl::LISTS_ONLY,
                                                                                   true);


            Visit(responseByIpOrOerror,
                  [&cache, &context, bFirstRcvd](
                          const THashMap<TString, NFuncClient::TRbl::TResponse> &responseByIp) mutable {
                      for (const auto&[ip, response] : responseByIp) {
                          cache.RblCache.Put(ip, NFuncClient::TRbl::LISTS_ONLY, true, response);
                          rbl_stub_ng(ip, response, context, bFirstRcvd);
                      }
                  },
                  [&logger](const NCurl::TError &error) {
                      logger << (TLOG_ERR) << "RblRequester->Perform:" << error;
                  }
            );
        }
    }

    if (!first_header && (attachedFrom.length() > 2) && attachedMsgId.length()) // attached message, will check for valid bounce
    {
        CProfItemGuard g(prof.Prof("attached"));
        mimepp::String shingleSrc;
        // attachedFrom not just <>
        // checkup user SUID for attached From

        if (config.ClientsConfigs.BBConfig && !attachedFrom.empty()) {
            try {
                CProfItemGuard g(prof.Prof("GetSUID"));
                TString sSUID, sUID;
                GetSUID(GlobalContext, TLogin{attachedFrom.c_str()}, &sSUID, &sUID, nullptr, nullptr, TBBRetrieveInfoMask{}, config, logger);

                if (sUID.length()) {
                    shingleSrc.assign(sUID.c_str());
                    shingleSrc += "_";
                    shingleSrc += attachedMsgId;
                }
            } catch (...) {
                logger << (TLOG_ERR) << __FILE__ << ':' << __LINE__ << ' ' << CurrentExceptionMessageWithBt();
            }
        }

        if (shingleSrc.length()) {
            CProfItemGuard g(prof.Prof("36Src"));
            SpSetShingle36Src(gsp, shingleSrc.c_str());
        }
    }

    gsp->CheckValueAllRules(FD_HEADERS, headersJson);

    if (!context.rs_got_nottrusted() && first_header) {
        SpSetRule(gsp, "ALLTRUSTEDIP");
    }

    if (!context.rs_got_notintranet() && first_header) {
        SpSetRule(gsp, "ALLINTRANETIP");
    }

    return nHeaderFields;
}

static mimepp::String SmartResolve(const TIpAddr& in, const TLog& logger) {
    TUdnsContextHolder ctx("def");

    TString r;
    if (ctx.Open(logger) && ctx.ResolveAddr(logger, in, &r))
        return StringBuf2MimeppString(r);
    else
        return "unknown";
}

mimepp::String FixCSOHeader(const TIpAddr& remoteAddr, mimepp::String sCSO, mimepp::String sRcvd, mimepp::String suffix, const TContext& context, const TLog& logger) {
    size_t fromPos, heloPos, byCSOPos, heloLen;
    mimepp::String helohost, newCSO;

    bool bHTTP = sRcvd.find("with HTTP") != mimepp::String::npos;
    bool bSMTP = sRcvd.find("with asmtp") != mimepp::String::npos;

    sRcvd.convertToLowerCase();

    if (sRcvd.compare(0, 4, "from", 4) == 0)
        fromPos = 0;
    else {
        fromPos = sRcvd.find(" from "); // leading space to ignore text like "(envelope-from "
        if (fromPos == mimepp::String::npos) {
            fromPos = sRcvd.find("\tfrom ");
            if (fromPos == mimepp::String::npos)
                return sCSO;
        }
    }

    heloPos = sRcvd.find("helo", fromPos);
    if (heloPos != mimepp::String::npos) // will take the word after "helo"
    {
        heloPos = sRcvd.find_first_not_of("= ?\"", heloPos + 4);
        heloLen = sRcvd.find_first_of(" ?)\"", heloPos);
        if (heloLen != mimepp::String::npos) {
            heloLen -= heloPos;
            helohost = sRcvd.substr(heloPos, heloLen);
        }
    } else // will take the word after "from"
    {
        heloPos = sRcvd.find_first_not_of(" \"", fromPos + 5);
        heloLen = sRcvd.find_first_of(" ()[\"", heloPos);
        if (heloLen != mimepp::String::npos) {
            heloLen -= heloPos;
            helohost = sRcvd.substr(heloPos, heloLen);
            size_t atPos = helohost.find_first_of("@"); // if helo comes as somebody@domain, take the domain
            if (atPos != mimepp::String::npos)
                helohost = helohost.substr(atPos + 1);
        }
    }

    newCSO = helohost.empty() ? "unknown" : helohost;
    newCSO.append(" (");
    newCSO.append(SmartResolve(remoteAddr, logger));
    newCSO.append(" [");
    newCSO.append(remoteAddr.ToString().c_str());
    newCSO.append("]) by cso-yandex.ru ");
    if (bHTTP)
        newCSO.append("with HTTP ");
    if (bSMTP)
        newCSO.append("with asmtp ");
    newCSO.append(suffix);
    byCSOPos = sCSO.find("cso-yandex.ru;", fromPos);
    if (byCSOPos != mimepp::String::npos) {
        sCSO.erase(5, byCSOPos + 14);
        sCSO.insert(5, newCSO);
    } else
        sCSO.assign(newCSO);

    logger << (TLOG_WARNING) << context.queueID << "detected " << suffix << " fixed header: " << sCSO;

    return sCSO;
}

mimepp::String ExtractDomain(const mimepp::String sFQDN) {
    size_t dotPos, domainPos;

    dotPos = sFQDN.find_last_of(".");
    if (dotPos == mimepp::String::npos)
        return "unknown";

    domainPos = sFQDN.find_last_of(".@ ", dotPos - 1);
    if (domainPos != mimepp::String::npos)
        return sFQDN.substr(domainPos + 1);

    return sFQDN;
}

mimepp::String SenderDomainTheirRcvd(const mimepp::String str) {
    size_t fromPos = str.find("from");
    mimepp::String sDomain;
    size_t spacePos;

    if (fromPos != mimepp::String::npos) {
        spacePos = str.find_first_of(" )>", fromPos + 5);
        sDomain = str.substr(fromPos + 5, spacePos - fromPos - 5);

        if (sDomain.find_first_of("[/") != mimepp::String::npos)
            return "unknown"; // no symbolic domain

        if (mimepp::Strcasecmp(sDomain, "localhost") == 0)
            return sDomain;

        return ExtractDomain(sDomain);
    }

    return "unknown";
}

mimepp::String SenderDomainOurRcvd(const mimepp::String str) {
    size_t fromPos = str.find("from");
    mimepp::String sDomain;
    size_t spacePos;

    if (fromPos != mimepp::String::npos) {
        spacePos = str.find_first_of(" )>", fromPos + 5);
        sDomain = str.substr(fromPos + 5, spacePos - fromPos - 5);

        return ExtractDomain(sDomain);
    }

    return "unknown";
}

bool SkipLocalRelays(const TGlobalContext& GlobalContext,
                     const mimepp::Entity* m,
                     int iStartAt,
                     const char* domain,
                     const char* sLabel,
                     const TContext& context,
                     TIpAddr* remote_addr,
                     const TSoConfig& config,
                     const TLog& logger,
                     int iFakeRcvd = 0) {
    int i;
    TString baddr;
    TIpAddr rec_addr;
    TIpAddr tmp_addr;
    size_t fromPos, labelPos;
    int externalRcvd, dotPos, atPos;
    mimepp::Headers* h = &m->headers();
    TIpAddr maskedIP;
    TIpAddr mask;
    mask.MakeMask(remote_addr->IsIpv6() ? 64 : 16, remote_addr->IsIpv6());
    mimepp::String sDomain = domain;
    bool bMailish = strstr(sLabel, " mailish");

    maskedIP = *remote_addr & mask;

    *remote_addr = TIpAddr();
    for (i = iStartAt; i < h->numFields(); i++) {
        const mimepp::Field& field = h->fieldAt(i);
        const mimepp::String& fieldText = field.fieldBody().text();
        if (mimepp::Strcasecmp(field.fieldName(), "received") != 0)
            continue;

        baddr = find_address(MimeppString2StringBuf(fieldText), rec_addr);
        if (baddr.length() > 0) {
            if (GlobalContext.local_matcher.match(rec_addr))
                continue;

            if ((rec_addr & mask) != maskedIP) {
                if (mimepp::Strcmp(sDomain, "unknown") == 0) // no domain specified, try resolved domain
                {
                    sDomain = ExtractDomain(SmartResolve(rec_addr, logger));
                }

                if (mimepp::Strcmp(sDomain, "unknown") == 0) // no domain specified, cannot resolve => IP rules
                    break;

                fromPos = fieldText.find("from");
                if (fromPos == mimepp::String::npos)
                    continue;
                // record like 'from mx13.rambler.ru'
                dotPos = (int)fieldText.find("." + sDomain, fromPos);
                // OR record like 'from <o-nastya@rambler.ru>'
                atPos = (int)fieldText.find("@" + sDomain, fromPos);

                if ((dotPos == (int)mimepp::String::npos) && (atPos == (int)mimepp::String::npos))
                    break; // no domain found, do not need domain check

                labelPos = std::min((unsigned)dotPos, (unsigned)atPos);

                // skip receiveds where specified domain label comes directly after "from"
                // and break when domain label appeared more distant, like "from blah-blah by <label>"
                if (fieldText.find_first_of(" ", fromPos, 1) !=
                    fieldText.find_last_of(" ", labelPos, 1))
                    break;
            }
        }
    }

    externalRcvd = i;
    if (externalRcvd < h->numFields()) {
        if (bMailish) // use the first external rcvd for mailish, second otherwise
        {
            // detect if there are any more receiveds with IPs
            // if no then just a message from domain
            for (i = externalRcvd; i < h->numFields(); i++) {
                if (mimepp::Strcasecmp(h->fieldAt(i).fieldName(), "received") != 0)
                    continue;

                baddr = find_address(h->fieldAt(i).fieldBody().text().c_str(), tmp_addr);
                if (baddr.length() > 0)
                    break;
            }
        }

        if (i < h->numFields()) {
            if (config.add_rcvd) {
                m->headers().fieldBody("X-Skipped-SO-Forward").setText(h->fieldAt(iFakeRcvd).fieldBody().text());
                m->headers().fieldAt(iFakeRcvd).setString("Received: " + FixCSOHeader(rec_addr,
                                                                                      h->fieldAt(iFakeRcvd).fieldBody().text(),
                                                                                      h->fieldAt(externalRcvd).fieldBody().text(),
                                                                                      sLabel, context, logger));
                m->headers().fieldAt(iFakeRcvd).parse();
            } else {
                mimepp::String fldStr("Received: "); // TODO - fieldAt(0) points to incorrect header if not add_rcvd
                fldStr.append(FixCSOHeader(rec_addr, h->fieldAt(iFakeRcvd).fieldBody().text(), h->fieldAt(externalRcvd).fieldBody().text(), sLabel, context, logger));
                mimepp::Field* fld = new mimepp::Field(fldStr);
                fld->parse();
                m->headers().insertFieldAt(0, fld);
            }
            *remote_addr = rec_addr;
            return true;
        }
    }

    return false;
}

bool MailruSmtp(const TGlobalContext& GlobalContext, const mimepp::Entity* m, TContext& context, TIpAddr* remote_addr, const TSoConfig& config, const TLog& logger) {
    TIpAddr rec_addr;
    TString baddr;
    mimepp::Headers h = m->headers();
    mimepp::Field fld;
    mimepp::String sFakeRcvd, sRemoteHost;
    TVector<int> recvec; // TVector for rcvds to skip
    mimepp::String sFromHost, sByHost;
    int iNativeRcvd = 0;

    if (!h.hasField("user-agent") && !h.hasField("X-Mailer"))
        return false;

    mimepp::String userAgent = h.hasField("user-agent") ? h.fieldBody("user-agent").getString() : h.fieldBody("X-Mailer").getString();
    if (mimepp::Strncasecmp(userAgent, "Mail.Ru Mailer", 14)) // != 0, not mailru agent
        return false;

    for (int i = 0; i < h.numFields(); ++i) {
        fld = h.fieldAt(i);
        if (mimepp::Strcasecmp(fld.fieldName(), "received") == 0) {
            if (sFakeRcvd.empty()) {
                if ((fld.fieldBody().text().find("by cso-yandex.ru") == mimepp::String::npos) && config.add_rcvd) {
                    logger << TLOG_WARNING << "%sFirst received header is not ours 5: %s" << fld.fieldBody().text();
                    return false;
                }
                sFakeRcvd = fld.fieldBody().text();
                continue;
            }

            baddr = find_address(fld.fieldBody().text().c_str(), rec_addr);

            if (baddr.empty() || baddr.length() == 0) // will remove leading rcvds w/o IP
                recvec.insert(recvec.begin(), i);
            else if (GlobalContext.ip_matcher.match(rec_addr) || GlobalContext.local_matcher.match(rec_addr))
                recvec.insert(recvec.begin(), i);
            else {
                iNativeRcvd = i;
                break;
            }
        }
    }

    if (iNativeRcvd == 0 && h.hasField("X-Originating-Ip")) {
        // no real rcvds except cso-yandex.ru
        baddr = find_address(h.fieldBody("X-Originating-Ip").text().c_str(), rec_addr, false);
        logger << TLOG_WARNING << "%sIP subst for '%s' with '%s'" << context.queueID << sFakeRcvd << h.fieldBody("X-Originating-Ip").text();

        m->headers().fieldAt(0).fieldBody().setText(FixCSOHeader(rec_addr,
                                                                 "",      // h.fieldAt(0).fieldBody().text(),
                                                                 "from ", // h.fieldAt(0).fieldBody().text(),
                                                                 "mailru agent;",
                                                                 context,
                                                                 logger));

        *remote_addr = rec_addr;
        SpSetRule(&context, "MAILRU_AUTH");

        return true;
    }

    return false;
}

mimepp::String ExtractFromhostDomain(mimepp::String sRcvd) {
    mimepp::String sHost;

    size_t startPos = sRcvd.find("from");
    if (startPos == mimepp::String::npos)
        return "";

    startPos += 4;
    while (sRcvd[startPos] == ' ')
        startPos++;

    size_t delimPos = sRcvd.find(" ", startPos);
    sHost = sRcvd.substr(startPos, delimPos == mimepp::String::npos ? delimPos : delimPos - startPos);

    size_t dotPos = sHost.find_last_of(".");
    if (dotPos != mimepp::String::npos)
        dotPos = sHost.find_last_of(".", --dotPos, 1);
    if (dotPos == mimepp::String::npos)
        return "";

    return sHost.substr(dotPos);
}

bool YandexPopper(const TGlobalContext& GlobalContext, const mimepp::Entity* m, TContext& context, TIpAddr* remote_addr, const TSoConfig& config, const TLog& logger) {
    TIpAddr rec_addr;
    TString baddr;
    mimepp::Headers h = m->headers();
    mimepp::String sFakeRcvd, sRemoteHost;
    mimepp::Field fld;
    TVector<int> recvec; // TVector for rcvds to skip
    int iNativeRcvd = 0;
    bool bYandexPopImap = false;
    mimepp::String sLastPopServer, sPopServerTail;
    int iLastRcvd = 0;
    int iFakeRcvd = 0;

    if (!h.hasField("x-yandex-pop-server"))
        return false;

    for (int i = 0; i < h.numFields(); ++i) {
        fld = h.fieldAt(i);

        if (mimepp::Strcasecmp(fld.fieldName(), "x-yandex-pop-server") == 0) {
            sLastPopServer = fld.fieldBody().text();
            size_t dotPos = sLastPopServer.find_last_of(".");
            if (dotPos != mimepp::String::npos)
                dotPos = sLastPopServer.find_last_of(".", --dotPos, 1);
            if (dotPos == mimepp::String::npos)
                continue;

            sPopServerTail = sLastPopServer.substr(dotPos); // like .yandex.ru, .gmail.com etc

            bYandexPopImap |= Strcasecmp(sPopServerTail, ".yandex.ru") == 0;

            continue;
        }

        if (mimepp::Strcasecmp(fld.fieldName(), "received") == 0) {
            iLastRcvd = i;
            if (sFakeRcvd.empty()) {
                if ((fld.fieldBody().text().find("by cso-yandex.ru") == mimepp::String::npos) && config.add_rcvd) {
                    logger << TLOG_WARNING << "%sFirst received header is not ours 1: %s" << fld.fieldBody().text();
                    return false;
                }
                sFakeRcvd = fld.fieldBody().text();
                iFakeRcvd = i;
                continue;
            }

            baddr = find_address(fld.fieldBody().text().c_str(), rec_addr);

            if (baddr.empty() || baddr.length() == 0) // will remove leading rcvds w/o IP
                recvec.insert(recvec.begin(), i);
            else {
                if (GlobalContext.local_matcher.match(rec_addr) || GlobalContext.ip_matcher.match(rec_addr)) // always skip our internal relays
                {
                    recvec.insert(recvec.begin(), i);
                    continue;
                }

                if (Strcasecmp(ExtractFromhostDomain(fld.fieldBody().text()), sPopServerTail) == 0) // otherwise skip remote relays internal to last pop server
                {
                    recvec.insert(recvec.begin(), i);
                    continue;
                }

                if (fld.fieldBody().text().find("with IMAP") != mimepp::String::npos) // follow IMAP chain
                {
                    recvec.insert(recvec.begin(), i);
                    logger << TLOG_NOTICE << "%sIMAP chain skip for '%s'" << fld.fieldBody().text();
                    continue;
                }

                iNativeRcvd = i;
                break;
            }
        }
    }

    for (TVector<int>::iterator ndx = recvec.begin(); ndx != recvec.end(); ndx++)
        m->headers().fieldAt(*ndx).setFieldName("X-Skipped-Received");

    if (iNativeRcvd != 0) {
        recvec.insert(recvec.begin(), iNativeRcvd);

        // substitute IP from X-Originating-Ip for yahoo
        if ((Strcasecmp(sPopServerTail, ".yahoo.com") == 0) && h.hasField("X-Originating-IP")) {
            baddr = find_address(h.fieldBody("X-Originating-IP").text().c_str(), rec_addr, false);
            logger << TLOG_WARNING << "%sIP subst for '%s' with '%s'" << sFakeRcvd << h.fieldBody("X-Originating-IP").text();
        }

        m->headers().fieldAt(iFakeRcvd).fieldBody().setText(FixCSOHeader(rec_addr,
                                                                         h.fieldAt(iFakeRcvd).fieldBody().text(),
                                                                         h.fieldAt(iNativeRcvd).fieldBody().text(),
                                                                         "yandex popper;", context, logger));
        *remote_addr = rec_addr;
        if (bYandexPopImap)
            SpSetRule(&context, "__POP3_AUTH");

        logger << TLOG_WARNING << "%sFixed rcvd: '%s'" << m->headers().fieldAt(iFakeRcvd).fieldBody().text();

        return true;
    } else // no external IP
    {
        m->headers().fieldBody("X-Skipped-SO-Forward").setText(h.fieldAt(iFakeRcvd).fieldBody().text());
        TIpAddr inaddr_any;
        if (remote_addr->IsIpv6())
            inaddr_any.FromString("::");
        else
            inaddr_any.FromString("0.0.0.0");

        m->headers().fieldAt(iFakeRcvd).fieldBody().setText(FixCSOHeader(inaddr_any,
                                                                         h.fieldAt(iFakeRcvd).fieldBody().text(),
                                                                         h.fieldAt(iLastRcvd).fieldBody().text(),
                                                                         "yandex popper;", context, logger));
        if (remote_addr->IsIpv6()) {
            remote_addr->FromString("::1");
        } else {
            remote_addr->FromString("127.0.0.1");
        }

        SpSetRule(&context, "POP3_LOCAL");
        return true;
    }

    return false;
}

bool ForwardDetected(const TGlobalContext& GlobalContext, const mimepp::Entity* m, TContext& context, TIpAddr* remote_addr, const TSoConfig& config, const TLog& logger) {
    mimepp::Headers h = m->headers();
    mimepp::String sFakeRecieved, sDomain, sLabel;
    mimepp::Field fld;
    TString baddr;
    TVector<int> recvec;
    int iAlienRcvd = -1;
    TIpAddr rec_addr;
    TIpAddr external_addr;
    int iSPFPos;
    bool bDetected = false;
    bool bMailish = false;
    int iFakeRcvd = 0;

    for (int i = 0; i < h.numFields(); ++i) {
        fld = h.fieldAt(i);
        if (mimepp::Strcasecmp(fld.fieldName(), "received") == 0) {
            if (sFakeRecieved.empty()) {
                if ((fld.fieldBody().text().find("by cso-yandex.ru") == mimepp::String::npos) && config.add_rcvd) {
                    logger << TLOG_WARNING << "%sFirst received header is not ours 2: %s" << h.fieldAt(i).fieldBody().text();
                    return false;
                }
                sFakeRecieved = fld.fieldBody().text();
                iFakeRcvd = i;
                //                continue;
            }

            baddr = find_address(fld.fieldBody().text().c_str(), rec_addr);

            if (baddr.empty() || baddr.length() == 0) // will remove leading rcvds w/o IP
                recvec.insert(recvec.begin(), i);
            else if (GlobalContext.ip_matcher.match(rec_addr) || GlobalContext.local_matcher.match(rec_addr))
                recvec.insert(recvec.begin(), i);
            else {
                iAlienRcvd = i;
                break;
            }
        }
    }

    if (iAlienRcvd >= 0) {
        external_addr = rec_addr;
        sDomain = SenderDomainTheirRcvd(h.fieldAt(iAlienRcvd).fieldBody().text());

        if (h.hasField("X-Yandex-Mailish")) {
            bMailish = true;
            sLabel = sDomain + " mailish;";
            bDetected = SkipLocalRelays(GlobalContext, m, iAlienRcvd, sDomain.c_str(), sLabel.c_str(), context, &rec_addr, config, logger, iFakeRcvd);
            if (bDetected) {
                // must not skip recalculated cso rcvd for mailish
                auto recvec_it = Find(recvec.begin(), recvec.end(), iFakeRcvd);
                if (recvec_it != recvec.end())
                    recvec.erase(recvec_it);
            }
            goto ALMOST_DONE;
        }

        if ((mimepp::Strcasecmp(sDomain, "mail.ru") == 0) &&
            h.hasField("X-MailRu-Forward")) // mail.ru forward
        {
            bDetected = SkipLocalRelays(GlobalContext, m, iAlienRcvd, "mail.ru", "mailru forward;", context, &rec_addr, config, logger);
            if (bDetected)
                SpSetRule(&context, "MAILRU_FWD");
            goto ALMOST_DONE;
        }

        if ((mimepp::Strcasecmp(sDomain, "google.com") == 0) &&
            h.hasField("X-Forwarded-To")) // google.com forward
        {
            bDetected = SkipLocalRelays(GlobalContext, m, iAlienRcvd, "google.com", "googlecom forward;", context, &rec_addr, config, logger);
            if (bDetected)
                SpSetRule(&context, "GMAIL_FWD");
            goto ALMOST_DONE;
        }

        if (mimepp::Strcasecmp(sDomain, "rambler.ru") == 0) {
            bool bRamblerOriginalTo = false;
            mimepp::String sOriginalDomain;

            if (h.hasField("X-Original-To"))
                sOriginalDomain = h.fieldBody("X-Original-To").text();
            if (h.hasField("X-Real-To"))
                sOriginalDomain = h.fieldBody("X-Real-To").text();

            if (!sOriginalDomain.empty()) {
                size_t atPos = sOriginalDomain.rfind("@");
                if (atPos != mimepp::String::npos)
                    bRamblerOriginalTo = mimepp::Strcasecmp(sOriginalDomain.substr(++atPos), "rambler.ru") == 0;
            }

            if (h.hasField("Resent-From") || bRamblerOriginalTo) // rambler.ru forward
            {
                bDetected = SkipLocalRelays(GlobalContext, m, iAlienRcvd, "rambler.ru", "ramblerru forward;", context, &rec_addr, config, logger);
                if (bDetected)
                    SpSetRule(&context, "RAMBLER_FWD");
                goto ALMOST_DONE;
            }
        }

        if (mimepp::Strcasecmp(sDomain, "km.ru") == 0) {
            bDetected = SkipLocalRelays(GlobalContext, m, iAlienRcvd, sDomain.c_str(), sLabel.c_str(), context, &rec_addr, config, logger);
            if (bDetected)
                SpSetRule(&context, "KM_FWD");
            goto ALMOST_DONE;
        }
    }

ALMOST_DONE:
    if (bDetected) // then forward detected
    {
        for (auto ndx = recvec.begin(); ndx != recvec.end(); ndx++) {
            if (bMailish)
                logger << TLOG_DEBUG << "%smailish skip set for [%d] '%s'" << *ndx << m->headers().fieldAt(*ndx).fieldBody().getString();
            m->headers().fieldAt(*ndx).setFieldName("X-Skipped-Received");
        }

        if (h.hasField("Received-SPF")) {
            iSPFPos = h.findFieldPos("Received-SPF");
            if (h.fieldAt(iSPFPos).fieldBody().text().find(".mail.yandex.net:") != mimepp::String::npos)
                m->headers().fieldAt(iSPFPos).setFieldName("X-Skipped-SPF");
        }
    }

    *remote_addr = rec_addr;

    return bDetected;
}

void AddGeoHeader(const TGlobalContext& GlobalContext, const mimepp::Message* m,
                  const char* sHeader,
                  const TIpAddr* host,
                  const NFuncClient::TRbl::TResponse& rblResponse,
                  bool doAsAndTors) {
    if (doAsAndTors) {
        int fldPos = m->headers().findFieldPos("IY-AS");
        if (fldPos >= 0)
            m->headers().removeFieldAt(fldPos);
        fldPos = m->headers().findFieldPos("IY-TOR");
        if (fldPos >= 0)
            m->headers().removeFieldAt(fldPos);
    }

    if (GlobalContext.local_matcher.match(*host)) {
        if (m->headers().hasField(sHeader))
            m->headers().removeFieldAt(m->headers().findFieldPos(sHeader));
        return;
    }

    {
        const TString geoHeader = TStringBuilder{} << rblResponse.GetIsoName() << ' ' << rblResponse.GetCity();
        m->headers().fieldBody(sHeader).setText(StringBuf2MimeppString(geoHeader));
    }

    if (doAsAndTors) // do not add as and tors for FirstGeoZone
    {
        if (rblResponse.IsTor())
            m->headers().fieldBody("IY-TOR").setString("1");

        if (rblResponse.GetLastAsnItem())
            m->headers().fieldBody("IY-AS").setString(StringBuf2MimeppString(*rblResponse.GetLastAsnItem()));
    }
}

void DetectFirstAndLastRemote(const TGlobalContext& GlobalContext, const mimepp::Message* m, TIpAddr* firsthost, TIpAddr* lasthost, TContext& context, bool& bRecheckFRNR,
                              const TSoConfig& config,
                              const TLog& logger,
                              TSessionCache& cache) {
    mimepp::Headers hdrs = m->headers();
    mimepp::Field fld;
    TString baddr;
    TIpAddr rec_addr;
    mimepp::Base64Decoder b64dec;
    mimepp::String sDecodedHint;
    size_t ipfromPos;
    mimepp::String sFakeRcvd;

    if (hdrs.hasField("X-QIP-Sender")) {
        rec_addr.FromString(hdrs.fieldBody("X-QIP-Sender").text().c_str());
        *firsthost = rec_addr;
    } else {
        for (int i = hdrs.numFields() - 1; i >= 0; i--) {
            fld = hdrs.fieldAt(i);
            if (mimepp::Strcasecmp(fld.fieldName(), "received") == 0) {
                baddr = find_address(fld.fieldBody().text().c_str(), rec_addr);
                if (baddr.empty() || baddr.length() == 0) // skip rcvds w/o IP
                    continue;

                if (GlobalContext.local_matcher.match(rec_addr)) // also skip local rcvds
                    continue;

                *firsthost = rec_addr;
                break;
            }
        }
    }

    for (int i = 0; i < hdrs.numFields(); i++) {
        fld = hdrs.fieldAt(i);
        if (mimepp::Strcasecmp(fld.fieldName(), "received") == 0) {
            if (sFakeRcvd.empty()) {
                if ((fld.fieldBody().text().find("by cso-yandex.ru") == mimepp::String::npos) && config.add_rcvd)
                    logger << TLOG_WARNING << "%sFirst received header is not ours 3: %s" << context.queueID << fld.fieldBody().text();
                else
                    sFakeRcvd = fld.fieldBody().text();
            }

            baddr = find_address(fld.fieldBody().text().c_str(), rec_addr);
            if (baddr.empty() || baddr.length() == 0) // skip rcvds w/o IP
                continue;

            if (GlobalContext.local_matcher.match(rec_addr)) // also skip local rcvds
                continue;

            *lasthost = rec_addr;
            break;
        }
    }

    // must ignore external FRNR if unresolved IP in our cso rcvd
    bRecheckFRNR = sFakeRcvd.find("unknown ") != mimepp::String::npos;

    if (config.fWebMail) {
        for (int i = 0; i < hdrs.numFields(); i++) {
            fld = hdrs.fieldAt(i);

            // multiple yandex-hint flds may appear

            if (mimepp::Strcasecmp(fld.fieldName(), "x-yandex-hint") == 0) {
                sDecodedHint = b64dec.decode(fld.fieldBody().text());

                // ipfrom comes for web so-out only, used to discover reak user IP
                ipfromPos = sDecodedHint.find("ipfrom=");
                if (ipfromPos != mimepp::String::npos) {
                    mimepp::String sIP = Parseout(sDecodedHint, "ipfrom=", "\n");
                    sIP = "[" + sIP + "]";
                    baddr = find_address(sIP.c_str(), rec_addr, false);
                    *lasthost = rec_addr;

                    logger << TLOG_WARNING << "%sIP subst for '%s' with '%s'" << context.queueID << sFakeRcvd << baddr;

                    m->headers().fieldAt(0).fieldBody().setText(FixCSOHeader(rec_addr,
                                                                             m->headers().fieldAt(0).fieldBody().text(),
                                                                             "from ",
                                                                             "yandex hint;", context, logger));

                    if(GlobalContext.Pools && GlobalContext.Pools->RblRequester) {
                        TString ip = rec_addr.ToString();
                        TMaybe<NFuncClient::TRbl::TResponse> rblResponse = cache.RblCache.Get(ip, NFuncClient::TRbl::GEO_ONLY, false);
                        if (!rblResponse.Defined()) {
                            auto responseOrError = GlobalContext.Pools->RblRequester->Perform(ip, "ipfrom", NFuncClient::TRbl::GEO_ONLY, false);

                            Visit(responseOrError,
                                  [&cache, &ip, &rblResponse](NFuncClient::TRbl::TResponse&& response) mutable {
                                      cache.RblCache.Put(ip, NFuncClient::TRbl::GEO_ONLY, false, response);
                                      rblResponse = std::move(response);
                                  },
                                  [&logger](const NCurl::TError& error){
                                      logger << TLOG_ERR << "rbl error: " << error;
                                  }
                            );
                        }
                        if(rblResponse) {
                            AddGeoHeader(GlobalContext, m, "IY-Geozone", &rec_addr, *rblResponse, true);
                        }
                    }
                }

                // check for phone
                size_t phonePos = sDecodedHint.find("phone=");
                if (phonePos != mimepp::String::npos) {
                    mimepp::String sPhone = Parseout(sDecodedHint, "phone=", "\n");
                    if (sPhone.length()) {
                        SpSetRule(&context, "WITH_SMS_COPY");
                        logger << TLOG_NOTICE << "WITH_SMS added for hint " << sDecodedHint;
                    }
                }
            }
        }
    }

}

TString BlackboxReply(const char* sAnswer, const char* lbl) {
    TString sResult;
    char *tokBegin, *tokEnd;

    tokBegin = (char*)strstr(sAnswer, lbl);
    if (tokBegin) {
        while (*tokBegin && (*tokBegin != '>'))
            tokBegin++;

        tokEnd = ++tokBegin;
        while (*tokEnd && (*tokEnd != '<'))
            tokEnd++;

        if (tokEnd > tokBegin)
            sResult.append(tokBegin, tokEnd - tokBegin);
    }

    return sResult;
}

TString ToCorpFormat(const TString& src) {
    TString dest(src);

    SubstGlobal(dest, "@", "-at-");
    SubstGlobal(dest, '.', '-');

    return src;
}

TString GetSUID(
    const TGlobalContext& GlobalContext,
    const NFuncClient::CBB::TBbKey& key,
    TString* sCleanSUID,
    TString* sUID,
    TString* login,
    THashSet<TString>* aliases,
    const TBBRetrieveInfoMask& retrieveInfoMask,
    const TSoConfig& config,
    const TLog& logger) {

    if(!GlobalContext.Pools->BBRequester) {
        logger << (TLOG_ERR) << "BB didn't setup";
        return {};
    }

    TString sResult;
    TString sEmailCorp; // special email format for corp, like  errors-dev-at-balance-yandex-ru
    TStringStream streamBuf;

    if ((config.dumbMode >= 4))
        return sResult;

    NBlackbox2::TOptions options;
    {
        NBlackbox2::TDBFields dbFields;

        dbFields << "subscription.suid.2";
        if (retrieveInfoMask.Test(GET_BB_REG_DATE))
            dbFields << "userinfo.reg_date.uid";
        if (retrieveInfoMask.Test(GET_BB_COUNTRY))
            dbFields << "userinfo.country.uid";

        options << dbFields;
    }
    {
        NBlackbox2::TAttributes attributes;
        if (retrieveInfoMask.Test(GET_BB_PHONE_EXISTS))
            attributes << "36";
        options << attributes;
    }

    if (retrieveInfoMask.Test(GET_BB_ALIASES) && aliases) {
        options << NBlackbox2::TOptAliases("8");
    }

    try {
        if (const auto response = GlobalContext.Pools->BBRequester->GetUserInfo(NFuncClient::CBB::MakeRequest(key, options), logger)) {
            if (auto suid = NFuncClient::CBB::GetSuidFromBBResponse(*response)) {
                sResult.append(" id=" + *suid);
                if (sCleanSUID)
                    sCleanSUID->assign(*suid);
            }

            if (login) {
                *login = NBlackbox2::TLoginInfo(response.Get()).Login();
            }

            if (retrieveInfoMask.Test(GET_BB_COUNTRY)) {
                if(auto country = NFuncClient::CBB::GetCountryFromBBResponse(*response))
                    sResult.append(" country=" + *country);
            }

            if (retrieveInfoMask.Test(GET_BB_KARMA)) {
                const NBlackbox2::TKarmaInfo karmaInfo(response.Get());

                sResult.append(" karma=" + karmaInfo.Karma());
                sResult.append(" karma_status=" + karmaInfo.KarmaStatus());
            }

            if (sUID) {
                *sUID = NBlackbox2::TUid(response.Get()).Uid();
                if(*sUID)
                    sResult.append(" uid=" + *sUID);
            }


            if (retrieveInfoMask.Test(GET_BB_PHONE_EXISTS) && NFuncClient::CBB::GetPhoneFromBBResponse(*response)) {
                sResult.append(" tel=1");
            }

            if (retrieveInfoMask.Test(GET_BB_REG_DATE)) {
                if(auto regTime = NFuncClient::CBB::GetRegDateFromBBResponse(*response))
                    sResult.append(" borndate=" + IntToStroka(regTime->TimeT()));
            }

            if (retrieveInfoMask.Test(GET_BB_ALIASES) && aliases) {
                *aliases = NFuncClient::CBB::GetPddAliasesFromBBResponse(*response);
            }
        }
    } catch (...) {
        logger << (TLOG_ERR) << config.ClientsConfigs.BBConfig << "/blackbox?" << CurrentExceptionMessageWithBt();
    }

    return sResult;
}

class TOrgsPool : public NCurl::TPool {
    using NCurl::TPool::TPool;
};

TMaybe<TCheckedMessage> run_assassin(const TSoConfig& config,
                                     TGlobalContext& GlobalContext,
                                     const TAgentDialogArtifact &artifact,
                                     TLog logger,
                                     TIpsListSafe* ipsList) {

    const TIpAddr peer(artifact.RemoteHostData.RemoteIp);
    NProf::Profiler prof;
    AtomicIncrement(GlobalContext.nThreads);
    Y_DEFER {
                const auto cpThreads = AtomicDecrement(GlobalContext.nThreads);

                logger <<TLOG_NOTICE << artifact.qID << " nThreads=" << cpThreads << ", prof: " << prof;
            };

    {
        CProfItemGuard allProf(prof.Prof("all"));

        const float fReject = 1000;
        bool trusted_zone = false;

        if (peer.IsValid()) // then checkup for non-check net(s)
        {
            CProfItemGuard g(prof.Prof("trusted_zone"));
            for (size_t i = 0; i < config.uNet.size(); i++) {
                if (config.uNet[i].IsValid() && config.uMask[i].IsValid() &&
                    ((peer & config.uMask[i]) == config.uNet[i])) {
                    logger << TLOG_WARNING << "non-check-wl outdated, use intranet_zone";
                    return Nothing();
                }
            }
            const bool intranet_helo = GlobalContext.intranet_matcher.match(peer) ||
                                       (peer.ToString() == "0.0.0.0") ||
                                       (peer.ToString() == "::");
            if (intranet_helo) {
                logger <<TLOG_NOTICE << "ip_address inside intranet zone, skipping" << peer.ToString();
                return Nothing();
            }

            trusted_zone = GlobalContext.ip_matcher.match(peer);
        }

        if (artifact.MailConsumer.Empty()) {
            logger << TLOG_ERR <<"empty ci.text";
            return Nothing();
        }

        TString sFrom(artifact.sSender); // collect all fields except SIZE=

        size_t sizePos = sFrom.find("SIZE=");
        while (sizePos != TString::npos) {
            size_t delimPos = sFrom.find_first_of(" \t", sizePos);
            sFrom.erase(sizePos, delimPos == TString::npos ? TString::npos : delimPos - sizePos + 1);

            sizePos = sFrom.find("SIZE=");
        }
        ToLower(sFrom.begin(), sFrom.size());

        const auto splitID = (artifact.MailConsumer.GetFullTextHash() ^ MultiHash(sFrom, artifact.sRecip)) % 1000;

        const auto msgProd = [&artifact, g = CProfItemGuard(prof.Prof("msg.parse")), &logger] {
            const auto text = artifact.MailConsumer.GetLimitedText();
            mimepp::Message msg(StringBuf2MimeppString(text));
            try {
                msg.parse();
            } catch (...) {
                logger << TLOG_ERR << "mimepp exception caught: " << CurrentExceptionMessageWithBt();
            }
            return msg;
        }();

        TString FixedFrom;

        TString FixedRcpto = FixAddress(artifact.sRecip);
        TString FixedOriginalRcpto = FixAddress(artifact.sOriginalRecip);

        const ui64 cacheSum = CreateCacheSum(sFrom, artifact.sRecip, msgProd.headers());
        if (!config.fWebMail && !artifact.OcrText && GlobalContext.Pools->CacheRequester) {
            CProfItemGuard g(prof.Prof("CreateCacheCheck"));

            // SO may have cached answer, do prepare cache key to check
            const TString sCacheCheckRow = CreateCacheCheck(sFrom, artifact.sRecip, true);

            NCacheShingler::TGetRequest getRequest(cacheSum);
            try {
                if(GlobalContext.Pools->CacheRequester->Get(getRequest, logger) && getRequest.GetCacheData().Defined()) {
                    GlobalContext.cacheHitStat->PushSignal(1);

                    logger << TLOG_WARNING << "SOCACHE hit for '%s'" << msgProd.headers().messageId().getString();

                    return std::move(*getRequest.GetCacheData());
                }
            } catch (...) {
                logger << (TLOG_ERR) << __FILE__ << ':' << __LINE__ << ' ' << CurrentExceptionMessageWithBt() << Endl;
            }
        }

        try {
            const TString messageId = [&msgProd]() -> TString {
                if (msgProd.headers().hasField("Message-ID")) {
                    return TStringBuilder{} << MimeppString2StringBuf(msgProd.headers().messageId().localPart())
                                            << '@'
                                            << MimeppString2StringBuf(msgProd.headers().messageId().domain());
                }
                return {};
            }();

            TContext context(GlobalContext.LibspParams,
                             GlobalContext.RulesHolder,
                             GlobalContext.ThreadPool,
                             GlobalContext.Pools,
                             nullptr,
                             GlobalContext.local_matcher,
                             GlobalContext.Models,
                             GlobalContext.TextToVecDssm,
                             GlobalContext.TextDeobfuscator,
                             GlobalContext.DomenFactorsBuilder,
                             GlobalContext.UserWeights,
                             GlobalContext.UidsStats);
            context.InitMessage(messageId, artifact.So2Context, artifact.ActivityInfo, artifact.OcrText, artifact.PrevSpClass, logger);
            GlobalContext.DequeContext();
            Y_DEFER {
                context.EndMessage(GlobalContext.Loggers.DeliveryLog, GlobalContext.Loggers.JsonMlLog);
                GlobalContext.EnqueContext();
            };

            Y_DEFER {
                        if (ipsList && config.ClientsConfigs.RblProducerConfig) {
                            if (context.m_cur->rulesContext.IsRuleWorked("IP_BLACK"))
                                ipsList->AddBlack(peer.ToString());
                            if (context.m_cur->rulesContext.IsRuleWorked("IP_WHITE"))
                                ipsList->AddWhite(peer.ToString());
                            if (context.m_cur->rulesContext.IsRuleWorked("IP_MAL"))
                                ipsList->AddMalic(peer.ToString());
                        }
                    };

            context.CheckRange("check_recip_count", artifact.CheckRecipCount);

            // Whitelist and intranet legality check
            if (msgProd.headers().hasField("From")) {
                mimepp::String srcMailFrom(msgProd.headers().fieldBody("From").text());
                bool frfaNoRFC{}, fromsTruncated{}, bBrokenRFC{};
                mimepp::String tgtMailFrom = CheckAndFixBrokenFrom(srcMailFrom, logger, bBrokenRFC, frfaNoRFC,
                                                                   fromsTruncated);

                if (frfaNoRFC)
                    context.m_cur->rulesContext.SetRule("FRFA_NORFC");
                if (fromsTruncated)
                    context.m_cur->rulesContext.SetRule("FROM_TRUNC");
                if (bBrokenRFC)
                    context.m_cur->rulesContext.SetRule("FR_NORFC");

                if ((tgtMailFrom != srcMailFrom) || bBrokenRFC) {
                    msgProd.headers().fieldBody("From").setString(tgtMailFrom);
                    msgProd.headers().from().parse();
                }

                mimepp::MailboxList &from = msgProd.headers().from();
                if ((from.numMailboxes() > 1) && !msgProd.headers().hasField(
                        "Sender")) // if From contains multiple boxes, Sender must exist by RFC 5322
                    context.m_cur->rulesContext.SetRule("NO_RFC5322");

                if (from.numMailboxes() > 0) {
                    mimepp::Mailbox &mb = from.mailboxAt(from.numMailboxes() - 1);
                    mimepp::String mailFrom = mb.localPart();
                    mailFrom += "@";
                    mailFrom += mb.domain();
                    FixedFrom = FixAddress(MimeppString2StringBuf(mailFrom));
                }
            }

            {
                context.CheckField(FD_IY_SPLIT_ID, IntToStroka(splitID));
                context.CheckRange("split_id", (ui64) splitID);
            }

            context.CheckField(FD_CURRENT_TIME, Now().ToRfc822StringLocal());

            TString foreignZone;
            if (artifact.sRecip && config.bDetectForeignMx) {
                CProfItemGuard g(prof.Prof("ForeignMxFound"));
                if (ForeignMxFound(GlobalContext, logger, FixedRcpto, foreignZone, config.domesticZones, artifact.Cache))
                    context.m_cur->rulesContext.SetRule("FOREIGN_MX");
            }

            context.shinglesCounter = MakeHolder<TShinglesCounter>(GlobalContext, FixedRcpto);
            {
                CProfItemGuard g(prof.Prof("count_shingle"));
                context.shinglesCounter->count_shingle_2(logger, msgProd, "", FixedRcpto, sFrom);
            }

            TShinglesGroupClass fmData;
            bool bIgnoreMXCode = false; // must not add X-Yandex-MXCode for outgoing mails for mfrm == rcpt

            THeadersList headers;

            if (dns_init_res < 0)
                logger << TLOG_NOTICE << "udns init error: " << dns_init_res;

            TString sRcpto = FixedRcpto;
            TString sQID;

            if(artifact.qID) {
                sQID = artifact.qID;
            } else if (msgProd.headers().hasField("X-Yandex-QueueID")) {
                sQID = StripString(
                        MimeppString2StringBuf(msgProd.headers().fieldBody("X-Yandex-QueueID").getString()));
            }
            TString charset;

            context.init_crc();
            context.max_len = context.i_max_len = 0;
            context.att_count = context.i_att_count = 0;
            context.queueID = sQID;

            TRengine *gsp = &context;
            TString compl_url;

            if (msgProd.headers().hasField("X-Yandex-Front")) {
                const char *s = msgProd.headers().fieldBody("X-Yandex-Front").text().c_str();
                size_t n = msgProd.headers().fieldBody("X-Yandex-Front").text().size();
                if (auto res = TRulesHolder::m_pcre->Check("name", TStringBuf(s, n))) {
                    gsp->m_cur->m_sSoFront = res->GetPattern(0);
                }
            }

            TString updated_from = StripString(sFrom);

            TUid sEnvelopeFromUID;
            TString sEnvelopeFromKarmaStatus, sEnvelopeFromBorndate;
            bool bCopyToInbox = false;
            bool bWMISource = false;

            headers.clear();

            {
                auto g = Guard(prof.Prof("decode_headers"));
                mimepp::Base64Decoder b64dec;
                mimepp::Field *fld;
                TStringBuf mailBackend = "";
                TStringBuf queueid = "";
                size_t recv_date_pos;

                for (int i = 0; i < msgProd.headers().numFields(); i++) {
                    fld = &msgProd.headers().fieldAt(i);
                    if (mimepp::Strcasecmp(fld->fieldName(), "x-yandex-hint") == 0) {
                        mimepp::String decodedHint = b64dec.decode(fld->fieldBody().text());
                        decodedHint.convertToLowerCase();
                        if (config.fWebMail) // for so-out must detect self-sent mails, via "copy-to-inbox" hint
                        {
                            bCopyToInbox |= decodedHint.find("copy_to_inbox=1") != mimepp::String::npos;
                            bWMISource |= decodedHint.find("save_to_sent=1") != mimepp::String::npos;
                        } else // extract receive_date for incoming
                        {
                            recv_date_pos = decodedHint.find("received_date=");
                            if (recv_date_pos != mimepp::String::npos) {
                                mimepp::String sRcvdDate = Parseout(decodedHint, "received_date=", "\n");

                                time_t now;
                                time(&now);
                                TString sDaysSinceRcvd = IntToStroka((now - atol(sRcvdDate.c_str())) / 3600 / 24);
                                SpCheckField(gsp, "IY-ReceivedDate", {sDaysSinceRcvd.c_str(), sDaysSinceRcvd.length()},
                                             nullptr, true);
                            }
                        }
                    }
                    if (mimepp::Strcasecmp(fld->fieldName(), "received") == 0) {
                        const char *s = fld->fieldBody().text().c_str();
                        size_t n = fld->fieldBody().text().size();
                        if (auto res = TRulesHolder::m_pcre->Check("mail_backend", TStringBuf(s, n))) {
                            mailBackend = res->GetPattern(1);
                            queueid = res->GetPattern(2);
                            if (!mailBackend.empty() && !queueid.empty() && queueid.equal(sQID)) {
                                gsp->m_cur->m_sMailBackend = mailBackend;
                            }
                        }
                    }
                }
                if (gsp->m_cur->m_sMailBackend.empty())
                    gsp->m_cur->m_sMailBackend = !gsp->m_cur->m_sSoFront.empty() ? gsp->m_cur->m_sSoFront : mailBackend;
                logger <<TLOG_WARNING << " HINT copyToInbox=%d WebSource=%d MailBackend=%s" <<
                       sQID << bCopyToInbox << bWMISource << gsp->m_cur->m_sMailBackend;
            }

            const THashMap<TLogin, NHtmlSanMisc::TUserInfo> *userInfos = gsp->m_cur->So2Context
                                                                         ? &gsp->m_cur->So2Context->GetUserInfos()
                                                                         : nullptr;
            const NHtmlSanMisc::TUserInfo *mailFromInfo = nullptr;
            if (gsp->m_cur->So2Context && gsp->m_cur->So2Context->GetMailFromUserinfo()) {
                mailFromInfo = &*gsp->m_cur->So2Context->GetMailFromUserinfo();
                logger << TLOG_ERR << "mailfrom: " << *mailFromInfo;
            }
            if (userInfos) {
                logger << TLOG_ERR << "userinfos: " << MakeRangeJoiner(",", *userInfos);
            }

            if (updated_from.length() > 2) /*&& (updated_from.find (" tel=") == string::npos))*/ // not just <>
            {
                CProfItemGuard g(prof.Prof("frm"));
                if (sFrom.Contains(" frm=")) {
                    gsp->AddStat(ToString(ST_LOG), "mfrm hamp");
                    SpSetRule(gsp, "FRM_HAMP");
                }

                size_t spacePos = updated_from.find(" ");
                updated_from.assign(updated_from.substr(0, spacePos));

                if (mailFromInfo) {
                    sEnvelopeFromUID = mailFromInfo->Uid;
                    context.m_cur->MailFromAliases.insert(mailFromInfo->PddAliases.cbegin(), mailFromInfo->PddAliases.cend());
                    updated_from = TStringBuilder{} << mailFromInfo->AsLegacy();
                } else if (!gsp->m_cur->So2Context) { // No response from So2
                    try {
                        TString sFromWithBBData = GetSUID(GlobalContext,
                                                          TLogin{updated_from},
                                nullptr,
                                &sEnvelopeFromUID,
                                nullptr,
                                &context.m_cur->MailFromAliases,
                                ~TBBRetrieveInfoMask{},
                                config,
                                logger);
                        if (sFromWithBBData.Contains(" id=")) {
                            updated_from.append(sFromWithBBData);
                            logger << TLOG_WARNING << "%sFRM: suid appended '%s' vs '%s'" <<
                                   updated_from <<
                                   sFrom;
                        }
                    } catch (...) {
                        logger << (TLOG_ERR) << __FILE__ << ':' << __LINE__ << ' ' << CurrentExceptionMessageWithBt();
                    }
                } else {
                    logger << (TLOG_ERR) << "cannot find info for updated_from: " << updated_from;
                }

                if (sEnvelopeFromUID.length())
                    SpSetSenderUID(gsp, sEnvelopeFromUID); // used later to logoff hacked acc
                else
                    logger << TLOG_WARNING << "%s UID problem for '%s'" << sQID << updated_from;

                if (const auto sFromGeoCode = Parseout(updated_from, "country=", " ")) {
                    SpSetSenderGeo(gsp, sFromGeoCode);
                }

                if (gsp->m_cur->So2Context &&
                    !AsciiEqualsIgnoreCase(
                        gsp->m_cur->So2Context->GetMailFromLogin().AsString(),
                        FixedFrom)) // envelope from differs header from:addr && our sender
                {
                    gsp->m_cur->rulesContext.SetRule("FROM_MISSMATCH");
                    logger << TLOG_WARNING << "%s FRM MISM %s != %s" << sQID << gsp->m_cur->So2Context->GetMailFromLogin().AsString() << FixedFrom;
                }
            }

            logger << (TLOG_WARNING) << "envUID: " << sEnvelopeFromUID << " rcpto: " << sRcpto;


            TDeque<TUid> uids;
            TDeque<TSuid> suids;
            TString updated_rcpto, updated_original_rcpto;
            {
                auto g = Guard(prof.Prof("process_rcpttos"));

                for (auto it : StringSplitter(sRcpto).Split(';').SkipEmpty()) {
                    const TString tok(it.Token());
                    const TLogin email(TStringBuf(tok).Before(' '));
                    const NHtmlSanMisc::TUserInfo *info = userInfos ? MapFindPtr(*userInfos, email) : nullptr;

                    if (info)
                        gsp->m_pstat.AddStat(ST_RCP_TO) << *info;

                    size_t delimPos = tok.find_first_of(" @");
                    if ((delimPos != NPOS) && (delimPos > 0)) {
                        if (stBlindLogins->find(tok.substr(0, delimPos)) != stBlindLogins->end())
                            SpIncrementBlindRcpto(gsp);
                    }

                    TUid uid;
                    TSuid suid;
                    TString tmp_rcpto = tok;
                    if (info) {
                        tmp_rcpto.assign(TStringBuilder{} << info->AsLegacy());
                        uid = info->Uid;
                        suid = info->Suid;
                    } else {
                        if (userInfos) {
                            logger << (TLOG_ERR) << "cannot find info for rcpt " << email << " origin: " << tok;
                        }
                        tmp_rcpto = tok;
                        uid.assign(Parseout(tok, "uid=", " "));
                        suid.assign(Parseout(tok, " id=", " "));
                    }

                    if (config.fWebMail && bWMISource && (uid == sEnvelopeFromUID)) {
                        if (!bCopyToInbox) // must ignore fake rcpt copied to so-out from mfrm for WMI
                        {
                            logger << TLOG_WARNING << "%sfake rcpto skipped: '%s'" << sQID << tok;
                            continue;
                        } else
                            bIgnoreMXCode = true; // will ignore MX code when real self-sent message (CopyToInbox flag set)
                    }

                    if (updated_rcpto.length() > 0)
                        updated_rcpto.append(";");

                    updated_rcpto.append(std::move(tmp_rcpto));

                    if (uid) {
                        uids.emplace_back(std::move(uid));
                    }

                    if (suid) {
                        suids.emplace_back(std::move(suid));
                    }
                }

                for (auto it : StringSplitter(FixedOriginalRcpto).Split(';').SkipEmpty()) {
                    const TString tok(it.Token());
                    const TLogin email(TStringBuf(tok).Before(' '));
                    const NHtmlSanMisc::TUserInfo *info = userInfos ? MapFindPtr(*userInfos, email) : nullptr;

                    TString sUID;
                    TString tmp_rcpto;
                    if (info) {
                        tmp_rcpto.assign(TStringBuilder{} << info->AsLegacy());
                        sUID.assign(info->Uid);
                    } else {
                        tmp_rcpto.assign(tok);
                        sUID.assign(Parseout(tmp_rcpto, "uid=", " "));
                    }

                    if (config.fWebMail && bWMISource && (sUID == sEnvelopeFromUID)) {
                        if (!bCopyToInbox) // must ignore fake rcpt copied to so-out from mfrm for WMI
                        {
                            continue;
                        }
                    }

                    if (updated_original_rcpto.length() > 0)
                        updated_original_rcpto.append(";");

                    updated_original_rcpto.append(tmp_rcpto);
                }
            }

            TCheckedMessage checkedMessage(std::move(uids), std::move(suids));

            {
                CProfItemGuard g(prof.Prof("CheckRcpto"));
                SpCheckRcpto(gsp,
                             updated_rcpto.empty() ? sRcpto : updated_rcpto,
                             updated_original_rcpto.empty() ? FixedOriginalRcpto : updated_original_rcpto);
            }

            {
                gsp->ProcessClearText(prof.Sub("html_san"), artifact.MailConsumer.GetFullText());
            }

            if (updated_from.length() > 2) // not just <>
            {
                CProfItemGuard g(prof.Prof("CheckMailFrom"));
                //            printf ("'%s'\n", updated_from.c_str());
                gsp->CheckMailFrom(updated_from);
            }

            {
                TIpAddr last_remote_addr;
                TIpAddr first_remote_addr;
                bool checkFirst = false, checkLast = false;
                TVector<TIpAddr> ipsToCheck{peer};

                bool bNewLastRemote = false;
                {
                    auto g = Guard(prof.Prof("detect_remote"));
                    DetectFirstAndLastRemote(GlobalContext, &msgProd, &first_remote_addr, &last_remote_addr, context, bNewLastRemote,
                                             config, logger, artifact.Cache);
                }
                {
                    auto g = Guard(prof.Prof("intranet_full"));
                    if ((config.fWebMail && MailruSmtp(GlobalContext, &msgProd, context, &last_remote_addr, config, logger)) ||
                        YandexPopper(GlobalContext, &msgProd, context, &last_remote_addr, config, logger) ||
                        ForwardDetected(GlobalContext, &msgProd, context, &last_remote_addr, config, logger)) // then overwrite last_remote
                    {
                        CProfItemGuard intranetMatcherProf(prof.Prof("intranet_matcher"));
                        if (GlobalContext.intranet_matcher.match(last_remote_addr)) {
                            report_rcvds(&msgProd, context);
                            SpSetRule(gsp, "FORCED_HAM");
                            logger <<  TLOG_WARNING << "%sForced HAM for IP=%s" << sQID <<
                                   last_remote_addr.ToString();

                            gsp->CheckMessage(prof.Sub("SpCheckMessage"),
                                                                        MimeppString2StringBuf(msgProd.getString()),
                                                                        checkedMessage);
                            checkedMessage.spClass = TSpClass::HAM;
                            return std::move(checkedMessage);
                        } else // geozone checkup
                        {
                            ipsToCheck.emplace_back(last_remote_addr);
                            checkLast = true;
                        }
                        bNewLastRemote = true;
                    }
                }

                if (first_remote_addr.IsValid()) {
                    ipsToCheck.emplace_back(first_remote_addr);
                    checkFirst = true;
                }

                if (GlobalContext.Pools && GlobalContext.Pools->RblRequester) {
                    TVector<TString> hostsToCheckRbl;
                    for (const auto &ip: ipsToCheck) {
                        hostsToCheckRbl.emplace_back(ip.ToString());
                    }
                    SortUnique(hostsToCheckRbl);
                    THashMap<TString, NFuncClient::TRbl::TResponse> rblResponses;
                    for (auto it = hostsToCheckRbl.begin(); it != hostsToCheckRbl.end();) {
                        TMaybe<NFuncClient::TRbl::TResponse> cachedResponse = artifact.Cache.RblCache.Get(*it, ~NFuncClient::TRbl::TQueryFields{}, false);
                        if (cachedResponse.Defined()) {
                            rblResponses.emplace(*it, std::move(*cachedResponse));
                            it = hostsToCheckRbl.erase(it);
                        } else {
                            ++it;
                        }
                    }
                    try {
                        if (hostsToCheckRbl) {
                            auto rblResponsesOrError = GlobalContext.Pools->RblRequester->Perform(hostsToCheckRbl,
                                                                                         config.UnistatPrefix,
                                                                                         ~NFuncClient::TRbl::TQueryFields{},
                                                                                         false);


                            Visit(rblResponsesOrError,
                                  [&artifact, &rblResponses](THashMap<TString, NFuncClient::TRbl::TResponse>&& responses) mutable {
                                      for (auto& [ip, response] : responses) {
                                          artifact.Cache.RblCache.Put(ip, ~NFuncClient::TRbl::TQueryFields{}, false, response);
                                          rblResponses.emplace(ip, std::move(response));
                                      }
                                  },
                                  [&logger](const NCurl::TError& error){
                                      logger << TLOG_ERR << "rbl error: " << error;
                                  }
                            );
                        }
                        if (auto it = MapFindPtr(rblResponses, artifact.sIP)) {
                            AddGeoHeader(GlobalContext, &msgProd, "IY-Geozone", &peer, *it, true);
                            context.m_cur->MlLogBuilder.Add("remote_ip", artifact.sIP);
                            context.m_pstat.AddStat("remote_ip") << artifact.sIP;
                            TString sFastRaject = IntToString<10>(it->IsInFastReject());
                            context.m_cur->MlLogBuilder.Add("fast_reject", sFastRaject);
                            context.m_pstat.AddStat("fast_reject") << sFastRaject;
                        }
                        if (checkLast) {
                            if (auto it = MapFindPtr(rblResponses, last_remote_addr.ToString())) {
                                AddGeoHeader(GlobalContext, &msgProd, "IY-Geozone", &last_remote_addr, *it, true);
                            }
                        }
                        if (checkFirst) {
                            if (auto it = MapFindPtr(rblResponses, first_remote_addr.ToString())) {
                                gsp->CheckValueAllRules(FD_JRBL, it->GetOriginal());

                                if (first_remote_addr != last_remote_addr) {
                                    CProfItemGuard g(prof.Prof("AddGeoHeader"));
                                    AddGeoHeader(GlobalContext, &msgProd, "IY-FirstGeozone", &first_remote_addr, *it);
                                }
                            }
                        }
                    } catch (...) {
                        logger << (TLOG_ERR) << "gsp->GetRbl:" << CurrentExceptionMessageWithBt();
                    }
                }
                try {
                    report_headers(GlobalContext, &msgProd, prof.Sub("report_headers"), true, context, bNewLastRemote, config, logger, artifact.Cache);
                } catch (...) {
                    logger << TLOG_ERR << "%shdrs check exception: "<< CurrentExceptionMessageWithBt();
                }
            }

            context.init_deep_level();

            try {
                visit_msg(GlobalContext, &msgProd, prof.Sub("visit_msg"), true, context, config, logger, artifact.Cache);
                if (!context.m_cur->BodyTextPresent && context.m_cur->So2Context) {
                    if (const NHtmlSanMisc::TDoc *visiblePart = context.m_cur->So2Context->FindVisiblePart()) {
                        visit_text(GlobalContext, *visiblePart, prof.Sub("visit_so2"), context, logger);
                        context.m_cur->rulesContext.SetRule("USING_SO2_BODYTEXT");
                    }
                }
            } catch (...) {
                logger << (TLOG_ERR) << "body check exception: " << CurrentExceptionMessageWithBt();
            }

            if (TString att_shingle = context.count_crc()) {
                gsp->SetShingle(spFcrc, att_shingle);
            }

            if (context.att_count > 0) {
                SpSetRule(gsp, count_tag.find(context.att_count));
                SpSetRule(gsp, len_tag.find(context.max_len));
            }

            if (context.i_att_count > 0) {
                SpSetRule(gsp, icount_tag.find(context.i_att_count));
                SpSetRule(gsp, ilen_tag.find(context.i_max_len));
            }

            SpSetRule(gsp, words_tag.find(context.shinglesCounter->full_words_count));
            SpSetRule(gsp, cwords_tag.find(context.shinglesCounter->sh_full_words));
            SpSetRule(gsp, cptag.find(context.shinglesCounter->fwords()));

            gsp->SetShingle(spShingle, context.shinglesCounter->last_shingle);

            try {
                gsp->CheckMessage(prof.Sub("SpCheckMessage"), MimeppString2StringBuf(msgProd.getString()), checkedMessage, fmData);
                checkedMessage.ShouldLogForKaspersky = gsp->m_cur->rulesContext.IsRuleWorked("TO_KASP_MALIC");
            } catch (...) {
                logger << (TLOG_ERR) << sQID << "check message exception '" << CurrentExceptionMessageWithBt() << "'";
            }

            checkedMessage.UidsWithPf = gsp->m_cur->PersonalFilterUids;
            checkedMessage.UidsResolutions = gsp->m_cur->UidsResolutions;
            checkedMessage.SoClass = gsp->m_cur->m_sSO_Class;
            checkedMessage.DenyGreyListing = gsp->GetGreyListingStatus();
            checkedMessage.RejectStr = CreateCacheCheck(sFrom, artifact.sRecip, false);
            gsp->m_pstat.VisitRecords([&checkedMessage](const auto& records){
                TDeque<std::pair<TString, TString>> strRecords;
                for (auto &[key, value] : records) {
                    strRecords.emplace_back(key, value.Str());
                }
                checkedMessage.DlvLog.emplace_back(std::move(strRecords));
            });

            for (auto &[login, attrs] : gsp->m_cur->m_rcptattrs) {
                const TUid* uid = attrs.RobustUid();
                if(!uid) {
                    continue;
                }

                if(!checkedMessage.UidsResolutions.contains(*uid)) {
                    checkedMessage.UidsResolutions[*uid] = checkedMessage.spClass;
                }

                if(const TSpClass* mapping = MapFindPtr(config.ResolutionsMapping, *uid)) {
                    checkedMessage.UidsResolutions[*uid] = *mapping;
                }
            }

            {
                const auto printer = checkedMessage.UidsPrinter();
                gsp->m_cur->MlLogBuilder.Add("uids_resolutions", printer.AsJson());
                gsp->m_pstat.AddStat("uids_resolutions") << printer;
            }

            if (!config.corp_mail && config.bDetectForeignMx && !bIgnoreMXCode) {
                if (SpPerformedRule(gsp, "MX_BLACK"))
                    checkedMessage.MXCode = "BLA";
                else if (SpPerformedRule(gsp, "MX_GREY"))
                    checkedMessage.MXCode = "GRE";
                else if (SpPerformedRule(gsp, "MX_WHITE"))
                    checkedMessage.MXCode = "WHI";
            }

            if (!SpPerformedRule(gsp, "DROP_ACTIVITIES")) {
                checkedMessage.UidsActivities = gsp->m_cur->UidsActivities;
            }
            checkedMessage.ForeignZone = foreignZone;

            switch (checkedMessage.spClass) {
                case TSpClass::HAM:
                    GlobalContext.hamStat->PushSignal(1);
                    break;
                case TSpClass::DLVR:
                    GlobalContext.dlvStat->PushSignal(1);
                    break;
                case TSpClass::SPAM:
                    GlobalContext.spamStat->PushSignal(1);
                    break;
                case TSpClass::MALIC:
                    GlobalContext.malicStat->PushSignal(1);
                    break;
                case TSpClass::UNKNOWN:
                    GlobalContext.failStat->PushSignal(1);
                    break;
            }

            CProfItemGuard constrheadersProf(prof.Prof("constrheaders"));

            static const constexpr int FORCEREJ = 11;

            checkedMessage.Reject = (checkedMessage.Report.Hits > fReject) ||
                                    ((int) checkedMessage.spClass == FORCEREJ) ||
                                    (config.smart_reject && (checkedMessage.spClass == TSpClass::MALIC));

            if (trusted_zone && checkedMessage.Reject) {
                checkedMessage.Reject = config.discard_trusted ? Y_DISCARD : 0;
                logger << TLOG_NOTICE << "%sMail from trusted zone IP: %s, will %s"
                        << artifact.qID << peer.ToString() << (checkedMessage.Reject ? "DISCARD" : "ACCEPT");
            }

            if(NeedInvertResolutionFotOutAllYaNotCaptcha(checkedMessage, config, gsp->m_cur->rulesContext)) {
                checkedMessage.SetResHamAndAllNotPfResSpam();
            }

            if (cacheSum && !config.fWebMail && config.ClientsConfigs.CacheConfig) {
                try {
                    NCacheShingler::TPuRequest putRequest(cacheSum, checkedMessage);

                    GlobalContext.Pools->CacheRequester->Put(putRequest, logger);
                } catch (...) {
                    logger << (TLOG_ERR) << __FUNCTION__ << ':' << CurrentExceptionMessageWithBt();
                }
            }

            logger << (TLOG_INFO) << artifact.qID << ' '
                              << artifact.sSender << ' '
                              << checkedMessage.spClass << ' '
                              << checkedMessage.Report.Hits << ' '
                              << peer.ToString() << ' '
                              << artifact.sRecip;
            checkedMessage.Mailish = gsp->m_cur->rulesContext.IsRuleWorked("MAILISH");
            checkedMessage.TempFail = gsp->m_cur->rulesContext.IsRuleWorked("TEMP_FAIL");

            return checkedMessage;
        } catch (...) {
            GlobalContext.failStat->PushSignal(1);
            logger << (TLOG_ERR) << "run_assassin error: " << CurrentExceptionMessageWithBt();
            return Nothing();
        }
    }
}
