#pragma once

// -*- C++ -*-

#ifndef _MSG2TXT_H
#define _MSG2TXT_H 1

#include <map>
#include <library/cpp/charset/codepage.h>
#include "msg2html.h"
#include "chars.h"

template <class FOBJ>
class msg2txt {
public:
    explicit msg2txt(const TGlobalContext& GlobalContext, const mimepp::Message& m) noexcept
    : msg(m)
    , GlobalContext(GlobalContext) {}
    void register_handler(const TString& div, std::reference_wrapper<FOBJ> f) {
        handlers.emplace(div, f);
    }

    bool go(const TLog& logger);

private:
    void visit_msg(const TLog& logger, const mimepp::Entity& m);
    void visit_header(const TLog& logger, const mimepp::Headers& h);
    void visit_text(const TLog& logger, FOBJ& fn, const mimepp::Entity& m);
    THashMap<TString, std::reference_wrapper<FOBJ>> handlers;
    const mimepp::Message& msg;
    const TGlobalContext& GlobalContext;
};

template <class FOBJ>
inline void
msg2txt<FOBJ>::visit_header(const TLog& logger, const mimepp::Headers& h) {
    for (int i = 0; i < h.numFields(); ++i) {
        const mimepp::Field& fl = h.fieldAt(i);

        if (auto handlerPtr = MapFindPtr(handlers, m_lower(fl.fieldName()))) {
            const auto f_val = decode_rfc2047(GlobalContext, logger, fl.fieldBody().text());
            static_cast<FOBJ&>(*handlerPtr)(f_val);
        }
    }
}

template <class FOBJ>
inline void
msg2txt<FOBJ>::visit_msg(const TLog& logger, const mimepp::Entity& m) {
    const auto handlerPtr = MapFindPtr(handlers, "body");

    visit_header(logger, m.headers());
    int type = mimepp::MediaType::TEXT;
    mimepp::String subtype = "plain";

    if (m.headers().hasField("Content-Type")) {
        type = m.headers().contentType().typeAsEnum();
        subtype = m.headers().contentType().subtype();
    }

    if (type == mimepp::MediaType::MULTIPART) {
        if (m.body().numBodyParts() < 1) {
            m.headers().contentType().setType("text");
            m.headers().contentType().setSubtype("plain");
            type = mimepp::MediaType::TEXT;
            subtype = "plain";
        }

        if ((mimepp::Strcasecmp(subtype, "alternative") == 0) && handlerPtr) {
            bool get_text = false;
            for (int i = 0; i < m.body().numBodyParts(); ++i) {
                const mimepp::BodyPart& part = m.body().bodyPartAt(i);
                if (part.headers().hasField("Content-Type")) {
                    if (part.headers().contentType().typeAsEnum() == mimepp::MediaType::TEXT &&
                        mimepp::Strcasecmp(part.headers().contentType().subtype(), "plain") == 0) {
                        visit_text(logger, *handlerPtr, part);
                        get_text = true;
                        break;
                    }
                } else // Default is  text/plain
                {
                    visit_text(logger, *handlerPtr, part);
                    get_text = true;
                    break;
                }
            }

            if (!get_text) {
                for (int i = 0; i < m.body().numBodyParts(); ++i) {
                    mimepp::BodyPart& part = m.body().bodyPartAt(i);
                    if (!part.headers().hasField("Content-Type") ||
                        part.headers().contentType().typeAsEnum() == mimepp::MediaType::TEXT) {
                        visit_text(logger, *handlerPtr, part);
                        break;
                    }
                }
            }
        } else {
            for (int i = 0; i < m.body().numBodyParts(); ++i) {
                visit_msg(logger, m.body().bodyPartAt(i));
            }
        }
    } else // single part
    {
        if (type == mimepp::MediaType::MESSAGE) {
            if(mimepp::Message* msg = m.body().message())
                visit_msg(logger, *msg);
        } else if (handlerPtr && type == mimepp::MediaType::TEXT)
            visit_text(logger, *handlerPtr, m);
    }

}

template <class FOBJ>
inline void
msg2txt<FOBJ>::visit_text(const TLog& logger, FOBJ& fn, const mimepp::Entity& m) {
    TString tout;
    int type = mimepp::MediaType::TEXT;
    mimepp::String subtype = "plain";

    if (m.headers().hasField("Content-Type")) {
        type = m.headers().contentType().typeAsEnum();
        subtype = m.headers().contentType().subtype();
    }
    if (type == mimepp::MediaType::TEXT && (mimepp::Strcasecmp(subtype, "plain") == 0 ||
                                            mimepp::Strcasecmp(subtype, "html") == 0)) {
        if (m.headers().hasField("Content-Transfer-Encoding"))
            tout = decode_transfer(m.body().getString(), m.headers().contentTransferEncoding().type());
        else
            tout.assign (m.body().getString().c_str(), m.body().getString().length());

        if(const auto c = CharsetByName(find_charset_name(m).c_str()); c != CODES_UNKNOWN)
            tout = char_decode(logger, tout, c);

        fn(tout);
    }
}

template <class FOBJ>
inline bool
msg2txt<FOBJ>::go(const TLog& logger) {
    visit_msg(logger, msg);
    return true;
}

#endif
