#include <mail/so/corp/udnscontext.h>

#include <mail/so/spamstop/tools/so-common/tudnsclass.h>

#include <mail/so/libs/protect/protect.h>
#include <mail/so/libs/syslog/so_log.h>

#include <util/datetime/systime.h>

static const auto ShortErrs = MakeTrueConst(
    THashMap<int, TString>{
        {DNS_E_NOERROR, "OK"},
        {DNS_E_TEMPFAIL, "tempfail"},
        {DNS_E_PROTOCOL, "protoerr"},
        {DNS_E_BADQUERY, "queryerr"}
    }
);

bool TUdnsContextHolder::Open(const TLog& logger, TString sSrv, int dns_timeout) {
    int errCode = 0;

    srv_addr = sSrv;
    if (context) {
        if (dns_init(context, 0) < 0)
            logger << TLOG_ERR << "udns dns_init() error: " << dns_strerror(errno);

        char timeoutstr[16];
        sprintf(timeoutstr, "timeout:%d", dns_timeout);
        if (dns_set_opts(context, timeoutstr) != 0)
            logger <<TLOG_WARNING << "udns error setting timeout " << timeoutstr;

        if (dns_set_opts(context, "attempts:1") != 0)
            logger << TLOG_WARNING << "udns error setting attempts";

        if (sSrv.length() > 0) {
            dns_add_serv(context, nullptr);
            errCode = dns_add_serv(context, sSrv.c_str());
            if (errCode < 0)
                logger << TLOG_ERR << "udns add_serv error %d (%s) for %s" <<errCode << dns_strerror(errCode) << sSrv;
        }

        if (dns_open(context) < 0) {
            logger << TLOG_ERR << "udns open error: " << dns_strerror(errno);
            return false;
        }
        bReady = true;
        return true;
    }
    return false;
}

TString TUdnsContextHolder::ErrorWord(int eCode)
{
    auto it = ShortErrs->find(eCode);

    if (it != ShortErrs->end())
        return it->second;

    return {};
}

bool TUdnsContextHolder::ResolveHost(const TLog& logger, const char* sHost, TString *unistatRow, TVector<TString>& pResults, int* errorcode) {
    struct dns_rr_a4* rr_a4 = nullptr;
    struct dns_rr_a6* rr_a6 = nullptr;
    bool result = false;
    char buf[DNS_MAXLABEL];
    timeval start, stop;
    double took = 0;
    TStringStream streamBuf;

    gettimeofday(&start, 0);
    pResults.clear();
    rr_a6 = dns_resolve_a6(context, sHost, 0);
    int err6 = dns_status(context);
    rr_a4 = dns_resolve_a4(context, sHost, 0);
    int err4 = dns_status(context);
    const auto errcode = std::max(err4, err6);
    if(errorcode)
        *errorcode = errcode;
    if (errcode < 0) {
        if (errcode == DNS_E_TEMPFAIL || errcode == DNS_E_PROTOCOL || errcode == DNS_E_BADQUERY)
        {
            logger << TLOG_ERR << "udns_%s error '%s' for %s" << sLabel << dns_strerror(errcode) << sHost;

            if (unistatRow)
            {
                streamBuf << "_udns_" << sLabel << "_err" << ErrorWord(errcode) << "_ammm,";

                gettimeofday(&stop, 0);
                took = ((double)timevalDiff(stop, start)) / 1000000;
            }
        }
    } else
        result = true;

    if (rr_a4) {
        for (int i = 0; i < rr_a4->dnsa4_nrr; i++) {
            memset(buf, '\0', sizeof(buf));
            inet_ntop(AF_INET, (void*)&rr_a4->dnsa4_addr[i], buf, sizeof(buf));
            pResults.push_back(buf);
        }
        free(rr_a4);
    }
    if (rr_a6) {
        for (int i = 0; i < rr_a6->dnsa6_nrr; i++) {
            memset(buf, '\0', sizeof(buf));
            inet_ntop(AF_INET6, (void*)&rr_a6->dnsa6_addr[i], buf, sizeof(buf));
            pResults.push_back(buf);
        }
        free(rr_a6);
    }

    if (result)
    {
        gettimeofday(&stop, 0);
        took = ((double)timevalDiff(stop, start)) / 1000000;
    }

    if (unistatRow)
    {
        if (result)
            streamBuf << "_udns_" << sLabel << "_OK_ammm,";
        streamBuf << "_udns_" << sLabel << "_TOTAL_ammm,";
        streamBuf << "_udns_" << sLabel << "_TIME_TOTAL_ammm:" << (took * 1000) << ',';
        streamBuf << "udns_" << sLabel << "-took-";
        if (took <= 0.05)
            streamBuf << "0,";
        else if (took <= 0.1)
            streamBuf << "50,";
        else if (took <= 0.2)
            streamBuf << "100,";
        else if (took <= 0.5)
            streamBuf << "200,";
        else if (took <= 1)
            streamBuf << "500,";
        else
            streamBuf << "1000,";

        unistatRow->append(streamBuf.Str());
    }

    gettimeofday(&stop, 0);
    logger << (took > 0.1 ? TLOG_WARNING : TLOG_DEBUG) << " (%s) udns_%s took %lf s." << sLabel << sHost << took;

    return result;
}

long long TUdnsContextHolder::timevalDiff(struct timeval& t1, struct timeval& t2) {
    return (long long)(t1.tv_sec - t2.tv_sec) * 1000000 + (t1.tv_usec - t2.tv_usec);
}

bool TUdnsContextHolder::ResolveAddr(const TLog& logger, const TIpAddr& host, TString * sStr) {
    timeval start, stop;
    TStringStream streamBuf;

    gettimeofday(&start, 0);
    struct dns_rr_ptr* rr = nullptr;

    rr = dns_resolve_ptr(context, host);
    auto errcode = dns_status(context);

    if (rr && sStr)
        *sStr = *rr->dnsptr_ptr;

    if (errcode == DNS_E_TEMPFAIL || errcode == DNS_E_PROTOCOL || errcode == DNS_E_BADQUERY) {
        logger << TLOG_ERR << "udns_%s error: %s for %s" << sLabel.c_str() << dns_strerror(errcode) << host.ToString();
    }

    gettimeofday(&stop, 0);
    double took = ((double)timevalDiff(stop, start)) / 1000000;
    logger << (took > 0.1 ? TLOG_WARNING : TLOG_DEBUG) << "(%s) udns took %lf s." << host.ToString() << took;

    if (rr) {
        free(rr);
        return true;
    }
    return false;
}

bool TUdnsContextHolder::ResolveMX(const TLog& logger, const TString& domain, TString *unistatRow, TString * sStr) {
    Y_UNUSED(unistatRow);
    struct dns_rr_mx* rr = nullptr;

    rr = dns_resolve_mx(context, domain.c_str(), 0);
    auto errcode = dns_status(context);

    if (sStr && rr && (rr->dnsmx_nrr > 0))
        *sStr = rr->dnsmx_mx[0].name;

    if (errcode == DNS_E_TEMPFAIL || errcode == DNS_E_PROTOCOL || errcode == DNS_E_BADQUERY) {
        logger << TLOG_ERR << "udns_%s error: %s for %s" << sLabel << dns_strerror(errcode) << domain;
    }

    if (rr) {
        free(rr);
        return true;
    }
    return false;
}

//dns_rr_ptr* dns_resolve_ptr(struct dns_ctx* ctx, const TIpAddr& addr) {
//    if (addr.IsIpv6())
//        return dns_resolve_a6ptr(ctx, addr.GetIn6Addr());
//    else
//        return dns_resolve_a4ptr(ctx, addr.GetInAddr());
//}
//
//dns_rr_a4* dns_resolve_dnsbl(struct dns_ctx* ctx, const TIpAddr& addr, const char* dnsbl) {
//    if (addr.IsIpv6())
//        return dns_resolve_a6dnsbl(ctx, addr.GetIn6Addr(), dnsbl);
//    else
//        return dns_resolve_a4dnsbl(ctx, addr.GetInAddr(), dnsbl);
//}
//
//dns_rr_txt* dns_resolve_dnsbl_txt(struct dns_ctx* ctx, const TIpAddr& addr, const char* dnsbl) {
//    if (addr.IsIpv6())
//        return dns_resolve_a6dnsbl_txt(ctx, addr.GetIn6Addr(), dnsbl);
//    else
//        return dns_resolve_a4dnsbl_txt(ctx, addr.GetInAddr(), dnsbl);
//}
//
//dns_rr_ptr* dns_resolve_ptr(struct dns_ctx* ctx, const struct in_addr* addr) {
//    return dns_resolve_a4ptr(ctx, addr);
//}
//
//dns_rr_ptr* dns_resolve_ptr(struct dns_ctx* ctx, const struct in6_addr* addr) {
//    return dns_resolve_a6ptr(ctx, addr);
//}
//
//dns_rr_a4* dns_resolve_dnsbl(struct dns_ctx* ctx, const struct in_addr* addr, const char* dnsbl) {
//    return dns_resolve_a4dnsbl(ctx, addr, dnsbl);
//}
//
//dns_rr_a4* dns_resolve_dnsbl(struct dns_ctx* ctx, const struct in6_addr* addr, const char* dnsbl) {
//    return dns_resolve_a6dnsbl(ctx, addr, dnsbl);
//}
//
//dns_rr_txt* dns_resolve_dnsbl_txt(struct dns_ctx* ctx, const struct in_addr* addr, const char* dnsbl) {
//    return dns_resolve_a4dnsbl_txt(ctx, addr, dnsbl);
//}
//
//dns_rr_txt* dns_resolve_dnsbl_txt(struct dns_ctx* ctx, const struct in6_addr* addr, const char* dnsbl) {
//    return dns_resolve_a6dnsbl_txt(ctx, addr, dnsbl);
//}

//dns_rr_a4* dns_resolve_a4(struct dns_ctx *ctx, const char *name, int flags);
