package ru.yandex.antifraud;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.util.AbstractMap;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import ru.yandex.antifraud.config.ImmutableAntiFraudConfig;
import ru.yandex.antifraud.currency.CurrencyRateMap;
import ru.yandex.antifraud.lua_context_manager.CurrenciesRatesTuner;
import ru.yandex.antifraud.lua_context_manager.FileListsProvider;
import ru.yandex.antifraud.lua_context_manager.PrototypesManager;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.antifraud.lua_context_manager.UaTraitsTuner;
import ru.yandex.antifraud.lua_context_manager.YasmTuner;
import ru.yandex.antifraud.lua_context_manager.config.ImmutablePrototypesConfig;
import ru.yandex.antifraud.lua_context_manager.config.PrototypesConfigBuilder;
import ru.yandex.antifraud.model_config.ImmutableCatboostModelConfig;
import ru.yandex.antifraud.rbl.RblClient;
import ru.yandex.antifraud.storage.StorageClient;
import ru.yandex.antifraud.util.AsyncClientHostRegistrar;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ExecutorServiceCloser;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.TimerTaskCloser;
import ru.yandex.function.Processable;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.jni.catboost.JniCatboostException;
import ru.yandex.jni.catboost.JniCatboostModel;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.HandlersManager;
import ru.yandex.lua.util.LuaException;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.SoFactorTypesRegistry;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.ThreadPoolStater;

public class AntiFraudHttpServer
        extends HttpProxy<ImmutableAntiFraudConfig>
        implements AsyncClientHostRegistrar {
    private final ThreadPoolExecutor threadPool;
    private final ThreadPoolExecutor uiThreadPool;

    private final Logger deliveryLogger;
    private final AtomicReference<PrototypesManager> prototypesManager;
    private final Map<String, YasmTuner> yasmTuners = new ConcurrentHashMap<>();
    private final Tvm2TicketRenewalTask tvm2RenewalTask;

    private final SoFactorsExtractorsRegistry extractorsRegistry;
    private final LongAdder violationsCounter;

    @Nullable
    private final UaTraitsTuner uaTraitsTuner;
    @Nonnull
    private final CurrencyRateMap currencyRateMap;

    @Nonnull
    private final Map<String, Map.Entry<ImmutableHttpHostConfig, AbstractAsyncClient<?>>> clientsWithConfigs =
            new HashMap<>();

    public AntiFraudHttpServer(final ImmutableAntiFraudConfig config)
            throws
            IOException, ConfigException, JsonException, URISyntaxException,
            HttpException, LuaException,
            TimeRange.IllegalTimeRangeException, FileListsProvider.InvalidFormatException, JniCatboostException {
        super(config);

        extractorsRegistry = new SoFactorsExtractorsRegistry(
                this,
                new SoFactorTypesRegistry());
        closeChain.add(extractorsRegistry);
        violationsCounter = new LongAdder();

        {
            final HandlersManager handlersManager = new HandlersManager();
            deliveryLogger = config.deliveryLog().build(handlersManager);
            registerLoggerForLogrotate(deliveryLogger);
        }
        uiThreadPool = new ThreadPoolExecutor(
                0,
                2,
                1,
                TimeUnit.MINUTES,
                new LifoWaitBlockingQueue<>(config.connections()),
                new NamedThreadFactory(
                        getThreadGroup(),
                        getName() + "-UI-",
                        true),
                new ThreadPoolExecutor.CallerRunsPolicy());
        closeChain.add(new ExecutorServiceCloser(uiThreadPool));
        registerStater(new ThreadPoolStater(uiThreadPool, "ui-thread-pool-"));

        final ThreadPoolExecutor lbQueue =
                new ThreadPoolExecutor(
                        1,
                        1,
                        1,
                        TimeUnit.HOURS,
                        new ArrayBlockingQueue<>(100));

        final ThreadPoolExecutor lbThreadPool = new ThreadPoolExecutor(
                3,
                3,
                1,
                TimeUnit.MINUTES,
                new LifoWaitBlockingQueue<>(config.connections()),
                new NamedThreadFactory(
                        getThreadGroup(),
                        getName() + "-LB-",
                        true),
                new ThreadPoolExecutor.CallerRunsPolicy());
        closeChain.add(new ExecutorServiceCloser(lbThreadPool));
        registerStater(new ThreadPoolStater(lbThreadPool, "lb-thread-pool-"));

        threadPool = new ThreadPoolExecutor(
                config.workers() - 4,
                config.workers() - 4,
                1,
                TimeUnit.MINUTES,
                new LifoWaitBlockingQueue<>(config.connections()),
                new NamedThreadFactory(
                        getThreadGroup(),
                        getName() + "-W-",
                        true),
                new ThreadPoolExecutor.CallerRunsPolicy());
        closeChain.add(new ExecutorServiceCloser(threadPool));
        registerStater(new ThreadPoolStater(threadPool, "thread-pool-"));

        @Nonnull final StorageClient storageClient = new StorageClient("Main", reactor, threadPool,
                config.mainStorageConfigs(), this);

        @Nonnull final StorageClient uiStorageClient = new StorageClient("UI", reactor, uiThreadPool,
                config.uiStorageConfigs(), this);

        {
            final Path path = config().currenciesRatesPath();
            currencyRateMap = CurrencyRateMap.make(path);
        }

        {
            uaTraitsTuner = Optional.ofNullable(config().uatraitsConfig())
                    .map(UaTraitsTuner::new)
                    .orElse(null);
            {
                final PrototypesManager manager = new PrototypesManager(
                        config.prototypesConfig(),
                        yasmTuners,
                        uaTraitsTuner,
                        new CurrenciesRatesTuner(currencyRateMap));
                manager.getChannels().forEach(channelInfo -> {
                    if (channelInfo.deliveryLogger() != null) {
                        registerLoggerForLogrotate(channelInfo.deliveryLogger());
                    }
                    registerStater(channelInfo.yasmTuner());
                });
                prototypesManager = new AtomicReference<>(manager);
            }

            final Scorer scorer, lbScorer;
            final CartesianCybertonicaHandler cartesianCybertonicaHandler;
            {
                final RblClient rblClient =
                        registerClientHost(
                                "RblClient",
                                new RblClient(reactor, config.rblConfig()),
                                config.rblConfig());

                clientHost("BlackboxClient",
                        config.blackboxConfig(),
                        RequestErrorType.ERROR_CLASSIFIER);

                {
                    final ImmutableHttpHostConfig furyConfig = config.furyConfig();
                    if (furyConfig != null) {
                        clientHost("FuryClient",
                                furyConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig passportConfig = config.passportConfig();
                    if (passportConfig != null) {
                        clientHost("PassportClient",
                                passportConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig passportConfig = config.passportSlowConfig();
                    if (passportConfig != null) {
                        clientHost("SlowPassportClient",
                                passportConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig passportConfig = config.deduplicatedPassportConfig();
                    if (passportConfig != null) {
                        clientHost("DeduplicatedPassportClient",
                                passportConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig pharmaConfig = config.pharmaConfig();
                    if (pharmaConfig != null) {
                        clientHost(
                                "PharmaClient",
                                pharmaConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig trustConfig = config.trustConfig();
                    if (trustConfig != null) {
                        clientHost(
                                "TrustClient",
                                trustConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig familyConfig = config.familyConfig();
                    if (familyConfig != null) {
                        clientHost("FamilyClient",
                                familyConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }
                {
                    final ImmutableHttpHostConfig cacheClientConfig =
                            config.cacheClientConfig();
                    if (cacheClientConfig != null) {
                        clientHost("CacheClient",
                                cacheClientConfig,
                                RequestErrorType.ERROR_CLASSIFIER);
                    }
                }

                @Nonnull final Map<String, JniCatboostModel> models;
                {
                    if (!config().models().isEmpty()) {
                        models = new HashMap<>(config.models().size());
                        for (Map.Entry<String, ImmutableCatboostModelConfig> entry :
                                config.models().entrySet()) {
                            models.put(entry.getKey(), new JniCatboostModel(entry.getValue().model(),
                                    entry.getValue().dict()));
                        }
                    } else {
                        models = Collections.emptyMap();
                    }
                }

                scorer = new Scorer(storageClient, this,
                        prototypesManager,
                        rblClient,
                        models,
                        currencyRateMap,
                        clientsWithConfigs,
                        threadPool);

                lbScorer = new Scorer(storageClient, this,
                        prototypesManager,
                        rblClient,
                        models,
                        currencyRateMap,
                        clientsWithConfigs,
                        lbThreadPool);
                cartesianCybertonicaHandler = new CartesianCybertonicaHandler(this, scorer);
            }

            register(
                    new Pattern<>("/cartesian_cybertonica", true),
                    cartesianCybertonicaHandler);

            register(
                    new Pattern<>("/multi_score", true),
                    scorer.makeMultiScoreHandler());

            register(
                    new Pattern<>("/cartesian_cybertonica_test", true),
                    cartesianCybertonicaHandler);
            {
                final HttpAsyncRequestHandler<JsonObject> handler =
                        new DelegatedHttpAsyncRequestHandler<>(scorer.makeScoreHandler(), this, threadPool);
                register(new Pattern<>("/scoring", true), handler);
                register(new Pattern<>("/score", true), handler);
                register(new Pattern<>("/external-score", true), handler);
                register(new Pattern<>("/execute", true), handler);
            }

            register(new Pattern<>("/v2/scoring", true), new So2HttpHandler(this, scorer));

            {
                final HttpAsyncRequestHandler<JsonObject> handler =
                        new DelegatedHttpAsyncRequestHandler<>(scorer.makeSaveHandler(), this, threadPool);
                register(new Pattern<>("/save_transaction", true), handler);
                register(new Pattern<>("/save", true), handler);
            }

            {
                final HttpAsyncRequestHandler<Processable<byte[]>> handler =
                        new DelegatedHttpAsyncRequestHandler<>(new LogBrokerHandler(this, lbScorer), this,
                                lbQueue);
                register(new Pattern<>("/passport-logbroke-it", true), handler);
            }
        }
        register(new Pattern<>("/reload_rules", true), new ReloadRulesHandler());

        register(new Pattern<>("/update_list", true),
                new DelegatedHttpAsyncRequestHandler<>(new UpdateListHandler(this, uiStorageClient,
                        prototypesManager), this, uiThreadPool));

        register(new Pattern<>("/delete_list", true),
                new DelegatedHttpAsyncRequestHandler<>(new DeleteListHandler(this, uiStorageClient,
                        prototypesManager), this, uiThreadPool));

        register(new Pattern<>("/get_list", false), new DelegatedHttpAsyncRequestHandler<>(new GetListHandler(this,
                uiStorageClient,
                prototypesManager), this, uiThreadPool));

        register(new Pattern<>("/get_transactions", true),
                new DelegatedHttpAsyncRequestHandler<>(new GetTransactionHandler(this, uiStorageClient,
                        prototypesManager), this, uiThreadPool));

        register(new Pattern<>("/get_verification_levels", true),
                new DelegatedHttpAsyncRequestHandler<>(new VerificationLevelUiSearchHandler(this, uiStorageClient),
                        this, uiThreadPool));

        register(new Pattern<>("/update_counter", true),
                new DelegatedHttpAsyncRequestHandler<>(new UpdateCounterHandler(this, uiStorageClient,
                        prototypesManager),
                        this, uiThreadPool));

        register(new Pattern<>("/solomon_stat", true),
                new SolomonHandler(() -> prototypesManager.get().metricRegistryRoot()));

        tvm2RenewalTask = new Tvm2TicketRenewalTask(
                logger().addPrefix("tvm2"),
                serviceContextRenewalTask,
                config.tvm2ClientConfig());
        closeChain.add(new TimerTaskCloser(tvm2RenewalTask));
    }

    @Override
    public void start() throws IOException {
        tvm2RenewalTask.start();
        threadPool.prestartAllCoreThreads();
        super.start();
    }

    public Logger deliveryLogger() {
        return deliveryLogger;
    }

    public SoFactorsExtractorsRegistry extractorsRegistry() {
        return extractorsRegistry;
    }

    public LongAdder violationsCounter() {
        return violationsCounter;
    }

    public ThreadPoolExecutor threadPool() {
        return threadPool;
    }

    public ThreadPoolExecutor uiThreadPool() {
        return uiThreadPool;
    }

    @Override
    public AsyncClient clientHost(String name,
                                  ImmutableHttpHostConfig config,
                                  Function<? super Exception, RequestErrorType> errorClassifier) {
        AsyncClient client = new AsyncClient(reactor, config, errorClassifier);
        return registerClientHost(name, client, config);
    }

    @Override
    public <T extends AbstractAsyncClient<T>> T registerClientHost(String name,
                                                                   T client,
                                                                   ImmutableHttpHostConfig config) {
        final T registered = super.registerClient(name, client, config);
        clientsWithConfigs.put(name, new AbstractMap.SimpleImmutableEntry<>(config, registered));
        return registered;
    }

    private class ReloadRulesHandler implements ProxyRequestHandler {
        @Override
        public void handle(ProxySession session) {
            session.logger().info("start reload rules");
            try {
                threadPool.execute(() -> {
                    try {
                        final PrototypesConfigBuilder builder =
                                new PrototypesConfigBuilder(config.rulesConfig(),
                                        config.rulesRoot());
                        final ImmutablePrototypesConfig immutablePrototypesConfig = builder.build();

                        final PrototypesManager manager = new PrototypesManager(
                                immutablePrototypesConfig,
                                yasmTuners,
                                uaTraitsTuner,
                                new CurrenciesRatesTuner(currencyRateMap));
                        manager.getChannels().forEach(channel -> {
                            if (channel.deliveryLogger() != null) {
                                registerLoggerForLogrotate(channel.deliveryLogger());
                            }
                            registerStater(channel.yasmTuner());
                        });
                        prototypesManager.lazySet(manager);
                        session.response(HttpStatus.SC_OK);
                    } catch (RuntimeException | ConfigException | IOException | LuaException |
                             FileListsProvider.InvalidFormatException e) {
                        session.logger().log(Level.WARNING, "fail to reload rules", e);
                        session.response(
                                HttpStatus.SC_INTERNAL_SERVER_ERROR,
                                new NStringEntity(
                                        e.toString(),
                                        ContentType.TEXT_PLAIN));
                    }
                });

            } catch (RuntimeException e) {
                session.logger().log(Level.WARNING, "fail to start reload rules task", e);
                session.response(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        new NStringEntity(
                                e.toString(),
                                ContentType.TEXT_PLAIN));
            }
        }
    }
}

