package ru.yandex.antifraud;

import java.io.IOException;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import javax.annotation.Nonnull;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;

import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FixedMultiFutureCallback;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.lua.util.LuaException;

public class CartesianCybertonicaHandler extends JsonHandler<HttpProxy<?>> {
    private final Scorer scorer;

    public CartesianCybertonicaHandler(@Nonnull HttpProxy<?> server, Scorer scorer) {
        super(server);
        this.scorer = scorer;
    }

    @Override
    public void handle(
            JsonObject jsonObject,
            HttpAsyncExchange httpAsyncExchange,
            HttpContext httpContext)
            throws HttpException {

        final ProxySession session = makeSession(httpAsyncExchange, httpContext);

        final List<String> cards = session.params().getAll("card_id");

        if (cards.isEmpty()) {
            session.logger().log(Level.SEVERE, "zero cards");
            session.response(
                    HttpStatus.SC_BAD_REQUEST,
                    new NStringEntity(
                            "zero cards",
                            ContentType.TEXT_PLAIN));
            return;
        }

        ScoringData baseRequest;
        try {
            baseRequest = new ScoringData(jsonObject.asMap(), scorer.toRubConverter());
        } catch (JsonException e) {
            session.logger().log(Level.SEVERE, "request error ", e);
            session.response(
                    HttpStatus.SC_BAD_REQUEST,
                    new NStringEntity(
                            e.getMessage(),
                            ContentType.TEXT_PLAIN));
            return;
        }

        final FixedMultiFutureCallback<Map.Entry<String, Artefacts>> multiFutureCallback =
                new FixedMultiFutureCallback<>(new Callback(session), cards.size());

        for (int i = 0, len = cards.size(); i < len; i++) {
            final JsonMap rawRequest = baseRequest.asJson().deepCopy();
            final String cardId = cards.get(i);
            rawRequest.put("card_id", new JsonString(cardId));
            rawRequest.put("payment_type", new JsonString("cvv"));
            rawRequest.put("extid", new JsonString(baseRequest.getExternalId() + '-' + cardId));
            final ScoringData request;
            try {
                request = new ScoringData(rawRequest, scorer.toRubConverter());
            } catch (JsonException e) {
                session.logger().log(Level.WARNING, "fail to parse cart request", e);
                throw new BadRequestException(e);
            }
            try {
                scorer.score(
                        request,
                        session.context().getAttribute(HttpServer.SESSION_ID).toString() + '-' + cardId,
                        session.context(),
                        session.listener(),
                        session.logger(),
                        false,
                        new PayloadFutureCallback<>(cardId, multiFutureCallback.callback(i))
                );
            } catch (UnknownChannelException e) {
                throw new BadRequestException(e);
            } catch (IOException | LuaException e) {
                throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e);
            }
        }
    }

    private static class Callback
            extends AbstractProxySessionCallback<List<Map.Entry<String, Artefacts>>> {

        public Callback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(List<Map.Entry<String, Artefacts>> responses) {
            final StringWriter sw = new StringWriter();
            int maxCode = HttpStatus.SC_OK;
            try {
                final JsonWriter writer = new JsonWriter(sw);
                writer.startObject();
                for (final Map.Entry<String, Artefacts> result : responses) {
                    final String cardId = result.getKey();
                    writer.key(cardId);
                    writer.value(result.getValue().getResolution().asJson());
                    maxCode = Integer.max(maxCode, result.getValue().statusCode());
                }
                writer.endObject();
            } catch (IOException e) {
                session.logger().log(Level.SEVERE, "request error ", e);
                session.response(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        new NStringEntity(
                                e.getMessage(),
                                ContentType.TEXT_PLAIN));
                return;
            }

            session.response(
                    maxCode,
                    new NStringEntity(
                            sw.toString(),
                            ContentType.APPLICATION_JSON));
        }
    }
}
