package ru.yandex.antifraud;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.antifraud.data.ListItem;
import ru.yandex.antifraud.lua_context_manager.PrototypesManager;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.storage.ListSearchRequest;
import ru.yandex.antifraud.storage.StorageClient;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonWriter;

public class GetListHandler implements HttpAsyncRequestHandler<JsonObject> {
    private final AntiFraudHttpServer server;
    @Nonnull
    private final StorageClient storageClient;
    @Nonnull
    private final AtomicReference<PrototypesManager> prototypesManager;

    public GetListHandler(@Nonnull AntiFraudHttpServer server,
                          @Nonnull StorageClient storageClient,
                          @Nonnull AtomicReference<PrototypesManager> prototypesManager) {
        this.server = server;
        this.storageClient = storageClient;
        this.prototypesManager = prototypesManager;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
            final HttpRequest request,
            final HttpContext context)
            throws HttpException {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
                ((HttpEntityEnclosingRequest) request).getEntity(),
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
    }

    @Override
    public void handle(
            final JsonObject jsonRequest,
            final HttpAsyncExchange exchange,
            final HttpContext context)
            throws HttpException {
        final ProxySession session = new BasicProxySession(server, exchange, context);

        @Nonnull final ListSearchRequest request;
        try {
            request = new ListSearchRequest(jsonRequest.asMap(), prototypesManager.get());
        } catch (JsonException | UnknownChannelException e) {
            session.logger().log(Level.WARNING, "fail to parse list request", e);
            throw new BadRequestException(e);
        }

        final FutureCallback<List<ListItem>> callback = new Callback(session);

        storageClient.search(
                request,
                session.listener(),
                session.context(),
                new CallbackListProxy(callback, session.logger()));
    }

    private static class Callback extends AbstractProxySessionCallback<List<ListItem>> {
        public Callback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(@Nonnull List<ListItem> response) {
            final StringBuilderWriter sb = new StringBuilderWriter();

            try (JsonWriter writer = new JsonWriter(sb)) {
                writer.startArray();

                for (ListItem item : response) {
                    writer.value(item.asPretty());
                }

                writer.endArray();
            } catch (IOException e) {
                failed(e);
                return;
            }


            session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                            sb.toString(),
                            ContentType.APPLICATION_JSON));
        }
    }

    private static class CallbackListProxy
            extends CallbackFutureBase<List<ListItem>, JsonObject> {
        final Logger logger;

        public CallbackListProxy(FutureCallback<? super List<ListItem>> callback, Logger logger) {
            super(callback);
            this.logger = logger;
        }

        @Override
        @Nonnull
        protected List<ListItem> convertResult(final JsonObject jsonObject) {
            if (jsonObject == null) {
                return Collections.emptyList();
            }
            try {
                return ListItem.parseResponse(jsonObject);
            } catch (JsonException | TimeRange.IllegalTimeRangeException e) {
                logger.log(
                        Level.WARNING,
                        "fail parse aggregates: ",
                        e);
                return Collections.emptyList();
            }
        }
    }
}
