package ru.yandex.antifraud;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.charset.CharacterCodingException;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import javax.annotation.Nonnull;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;

import ru.yandex.antifraud.lua_context_manager.PrototypesManager;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.storage.StorageClient;
import ru.yandex.antifraud.storage.TransactionSearchUIRequest;
import ru.yandex.antifraud.util.ExecutedCallbackFutureBase;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonWriter;

public class GetTransactionHandler extends JsonHandler<AntiFraudHttpServer> {
    @Nonnull
    private final StorageClient storageClient;
    @Nonnull
    private final AtomicReference<PrototypesManager> prototypesManager;

    public GetTransactionHandler(@Nonnull AntiFraudHttpServer server,
                                 @Nonnull StorageClient storageClient,
                                 @Nonnull AtomicReference<PrototypesManager> prototypesManager) {
        super(server);
        this.storageClient = storageClient;
        this.prototypesManager = prototypesManager;
    }

    @Override
    public void handle(
            final JsonObject jsonRequest,
            final HttpAsyncExchange exchange,
            final HttpContext context)
            throws HttpException {
        final ProxySession session = new BasicProxySession(server, exchange, context);

        @Nonnull final List<TransactionSearchUIRequest> requests;
        try {
            requests = TransactionSearchUIRequest.parse(jsonRequest.asMap(), prototypesManager.get());
        } catch (JsonException | UnknownChannelException | CharacterCodingException e) {
            session.logger().log(Level.WARNING, "fail to parse transaction search request", e);
            throw new BadRequestException(e);
        }

        final MultiFutureCallback<JsonObject> callbacks = new MultiFutureCallback<>(new Callback(session));

        try {
            for (TransactionSearchUIRequest request : requests) {
                storageClient.search(
                        request,
                        session.listener(),
                        session.context(),
                        new ExecutedCallbackFutureBase<>(server.uiThreadPool(), callbacks.newCallback()));
            }
        } finally {
            callbacks.done();
        }
    }

    private static class Callback extends AbstractProxySessionCallback<List<JsonObject>> {

        public Callback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(List<JsonObject> responses) {
            final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
            final JsonParser jsonParser = TypesafeValueContentHandler.prepareParser(consumer);
            try {
                final StringBuilderWriter sb = new StringBuilderWriter();

                try (JsonWriter writer = new JsonWriter(sb)) {

                    writer.startObject();

                    writer.key("transactions");
                    writer.startArray();

                    for (JsonObject response : responses) {
                        for (JsonObject item : response.get("hitsArray").asList()) {
                            writer.startObject();
                            for (var entry : item.asMap().entrySet()) {
                                final String key = entry.getKey();
                                JsonObject value = entry.getValue();
                                switch (key) {
                                    case "data":
                                    case "value":
                                    case "user_context":
                                    case "txn_afs_tags":
                                    case "txn_score_context":
                                        if (value.type() == JsonObject.Type.STRING) {
                                            try {
                                                jsonParser.reset();
                                                jsonParser.parse(value.asString());
                                                value = consumer.get();
                                            } catch (JsonException e) {
                                                session.logger().log(Level.WARNING, "cannot parse", e);
                                            }
                                        }
                                        break;
                                    default:
                                        break;
                                }
                                writer.key(key);
                                writer.value(value);
                            }
                            writer.endObject();
                        }
                    }
                    writer.endArray();
                    writer.endObject();
                }

                session.response(
                        HttpStatus.SC_OK,
                        new NStringEntity(sb.toString(), ContentType.APPLICATION_JSON));
            } catch (Exception e) {
                final StringWriter sw = new StringWriter();
                try (PrintWriter pw = new PrintWriter(sw)) {
                    e.printStackTrace(pw);
                }

                session.response(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        new NStringEntity(e.toString() + sw, ContentType.TEXT_PLAIN));
            }
        }
    }
}
