package ru.yandex.antifraud;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.logging.Level;

import javax.annotation.Nonnull;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.MimeTokenStream;

import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.Processable;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.UngzippingByteArrayProcessableAsyncConsumer;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.io.ByteArrayInputStreamFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.lua.util.LuaException;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.tskv.BasicTskvParser;
import ru.yandex.tskv.TskvException;
import ru.yandex.tskv.TskvHandler;
import ru.yandex.tskv.TskvRecord;

public class LogBrokerHandler implements HttpAsyncRequestHandler<Processable<byte[]>> {
    @Nonnull
    private final AntiFraudHttpServer server;
    @Nonnull
    private final Scorer scorer;

    public LogBrokerHandler(@Nonnull AntiFraudHttpServer server,
                            @Nonnull Scorer scorer) {
        this.server = server;
        this.scorer = scorer;
    }

    @SuppressWarnings("UnusedVariable")
    private void parseStream(BasicTskvParser parser, RecordHandler recordHandler, InputStream stream) throws IOException, TskvException {
        final byte[] bytes = stream.readAllBytes();

        final Supplier<InputStreamReader> streamSupplier =
                () -> new InputStreamReader(new ByteArrayInputStream(bytes), Charset.defaultCharset());

        try (BufferedReader reader = new BufferedReader(streamSupplier.get())) {

            final BasicGenericConsumer<JsonObject, JsonException> jsonConsumer =
                    new BasicGenericConsumer<>();

            final JsonParser jsonParser = TypesafeValueContentHandler.prepareParser(jsonConsumer);

            while (reader.ready()) {
                final String line = reader.readLine();
                jsonParser.parse(line);
                recordHandler.onRecord(jsonConsumer.get().asMap());
                jsonParser.reset();
            }

            return;
        } catch (JsonException | UnknownChannelException | LuaException e) {
            server.logger().log(Level.WARNING, "failed to parse json save request " + new String(bytes,
                    Charset.defaultCharset()), e);
        }

        try (Reader reader = streamSupplier.get()) {
            parser.parseWithException(reader);
        }
    }

    @Nonnull
    private static JsonMap tskv2json(@Nonnull TskvRecord record) {
        JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);

        for (Map.Entry<String, String> entry : record.entrySet()) {
            result.put(entry.getKey(), new JsonString(entry.getValue()));
        }

        return result;
    }

    @Override
    public UngzippingByteArrayProcessableAsyncConsumer processRequest(
            final HttpRequest request,
            final HttpContext context) {
        return new UngzippingByteArrayProcessableAsyncConsumer();
    }

    @Override
    public void handle(
            final Processable<byte[]> body,
            final HttpAsyncExchange exchange,
            final HttpContext context)
            throws HttpException {
        final ProxySession session = new BasicProxySession(server, exchange, context);

        final ContentType contentType;
        {
            final String contentTypeHeader =
                    session.headers().getString(HTTP.CONTENT_TYPE, null);
            if (contentTypeHeader != null) {
                contentType = ContentType.parse(contentTypeHeader);
            } else {
                contentType = null;
            }
        }

        final Callback callback = new Callback(session);
        {
            final MultiFutureCallback<Artefacts> multiFutureCallback =
                    new MultiFutureCallback<>(callback);

            final RecordHandler recordHandler = new RecordHandler(session, multiFutureCallback);

            final BasicTskvParser parser = new BasicTskvParser(recordHandler);
            try (final ByteArrayInputStream dataStream = body.processWith(ByteArrayInputStreamFactory.INSTANCE)) {
                if (contentType == null || !contentType.getMimeType().equals("multipart/mixed")) {
                    parseStream(parser, recordHandler, dataStream);
                } else {
                    final MimeTokenStream stream = new MimeTokenStream(
                            DefaultMimeConfig.INSTANCE,
                            null,
                            new Utf8FieldBuilder(),
                            new OverwritingBodyDescriptorBuilder());
                    stream.parseHeadless(dataStream, contentType.toString());

                    for (EntityState state = stream.getState();
                         state != EntityState.T_END_OF_STREAM;
                         state = stream.next()) {
                        if (state == EntityState.T_BODY) {
                            parseStream(parser, recordHandler, stream.getDecodedInputStream());
                        }
                    }
                }
                multiFutureCallback.done();
            } catch (IOException | MimeException | TskvException e) {
                session.logger().log(Level.WARNING, "fail to parse lb request", e);
                multiFutureCallback.failed(e);
            }
        }
    }

    private static class Callback extends AbstractProxySessionCallback<List<Artefacts>> {
        Callback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final List<Artefacts> results) {
            final StringBuilderWriter sb = new StringBuilderWriter();

            int maxCode = HttpStatus.SC_OK;
            try (JsonWriter writer = new JsonWriter(sb)) {
                writer.startArray();
                for (Artefacts artefacts : results) {
                    writer.value(artefacts.getResolution().asJson());
                    maxCode = Integer.max(maxCode, artefacts.statusCode());
                }
                writer.endArray();
            } catch (IOException e) {
                failed(e);
                return;
            }

            session.response(maxCode, new NStringEntity(sb.toString(), ContentType.APPLICATION_JSON));
        }
    }

    class RecordHandler implements TskvHandler<TskvRecord> {
        @Nonnull
        private final ProxySession session;

        @Nonnull
        private final MultiFutureCallback<Artefacts> multiFutureCallback;

        RecordHandler(@Nonnull ProxySession session,
                      @Nonnull MultiFutureCallback<Artefacts> multiFutureCallback) {
            this.session = session;
            this.multiFutureCallback = multiFutureCallback;
        }

        public void onRecord(@Nonnull JsonMap data) throws IOException, UnknownChannelException,
                LuaException, JsonException {
            final ScoringData scoringData = new ScoringData(data, scorer.toRubConverter());
            scorer.score(
                    scoringData,
                    session.context().getAttribute(HttpServer.SESSION_ID).toString() + '-' + scoringData.getExternalId(),
                    session.context(),
                    session.listener(),
                    session.logger(),
                    true,
                    multiFutureCallback.newCallback()
            );
        }

        @Override
        public boolean onRecord(final TskvRecord record) {
            try {

                onRecord(tskv2json(record));
            } catch (Exception e) {
                session.logger().log(Level.WARNING, "fail to process lb request", e);
                throw new RuntimeException(e);
            }

            return true;
        }

        @Override
        public boolean onError(TskvException e) {
            session.logger().log(Level.WARNING, "fail to process tsckv record", e);
            return false;
        }
    }
}
