package ru.yandex.antifraud;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;

import ru.yandex.antifraud.aggregates.Aggregates;
import ru.yandex.antifraud.aggregates.AggregatesBatch;
import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.channel.Channel;
import ru.yandex.antifraud.channel.ChannelWithCrossChannels;
import ru.yandex.antifraud.channel.EntriesDeque;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.currency.ToRubConverter;
import ru.yandex.antifraud.data.AggregatedData;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.lua_context_manager.PrototypesManager;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.rbl.RblClient;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.antifraud.storage.CountersUpdateRequest;
import ru.yandex.antifraud.storage.CreateAggregatesRequest;
import ru.yandex.antifraud.storage.ServiceVerificationLevelUpdateRequest;
import ru.yandex.antifraud.storage.StorageClient;
import ru.yandex.antifraud.storage.TransactionCreateRequest;
import ru.yandex.antifraud.storage.TransactionUpdateRequest;
import ru.yandex.antifraud.util.JsonWriterSb;
import ru.yandex.antifraud.util.Waterfall;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ClosingFutureCallback;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.jni.catboost.JniCatboostModel;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.lua.util.JsonUtils;
import ru.yandex.lua.util.LuaException;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.SoFunctionInputs;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractor;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorContext;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.JsonMapSoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.mail.so.factors.types.SoFactorType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;

public class Scorer implements SoFactorsExtractor {
    private static final List<SoFactorType<?>> INPUTS =
            Arrays.asList(JsonMapSoFactorType.JSON_MAP, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
    private static final List<SoFactorType<?>> OUTPUTS =
            Collections.singletonList(JsonMapSoFactorType.JSON_MAP);

    private static final JsonString APP = new JsonString("app");
    @Nonnull
    private final StorageClient storageClient;
    @Nonnull
    private final AntiFraudHttpServer server;
    @Nonnull
    private final AtomicReference<PrototypesManager> prototypesManager;
    @Nonnull
    private final Map<String, JniCatboostModel> models;
    @Nonnull
    private final SourcesCollector sourcesCollector;
    @Nonnull
    private final ToRubConverter toRubConverter;
    @Nonnull
    private final Map<String, Map.Entry<ImmutableHttpHostConfig, AbstractAsyncClient<?>>> clients;
    @Nonnull
    private final Executor executor;

    public Scorer(@Nonnull StorageClient storageClient,
                  @Nonnull AntiFraudHttpServer server,
                  @Nonnull AtomicReference<PrototypesManager> prototypesManager,
                  @Nonnull RblClient rblClient,
                  @Nonnull Map<String, JniCatboostModel> models,
                  @Nonnull ToRubConverter toRubConverter,
                  @Nonnull Map<String, Map.Entry<ImmutableHttpHostConfig, AbstractAsyncClient<?>>> clients,
                  @Nonnull Executor executor) {
        this.storageClient = storageClient;
        this.server = server;
        this.prototypesManager = prototypesManager;
        this.models = models;
        this.toRubConverter = toRubConverter;
        this.clients = clients;
        this.executor = executor;

        sourcesCollector = new SourcesCollector(
                storageClient,
                rblClient);
    }

    @Nonnull
    public ToRubConverter toRubConverter() {
        return toRubConverter;
    }

    /**
     * hotfix for <a href="https://st.yandex-team.ru/GMSP-472">GMSP-472</a>
     * todo - remove this asap
     */
    @Nonnull
    static JsonMap removeCardInfoForPays(@Nonnull JsonMap src) throws JsonException {
        if ("taxi".equals(src.getString("channel", null)) ||
                "eda_lavka".equals(src.getString("sub_channel", null))) {
            final JsonObject cardId = src.get("card_id");

            if (cardId == null || cardId.type() != JsonObject.Type.STRING) {
                for (String fieldToRemove : Arrays.asList(
                        "src_id",
                        "card_bin",
                        "card_brand",
                        "card_isoa2",
                        "card_isocountry",
                        "card_level",
                        "card_type"

                )) {
                    src.remove(fieldToRemove);
                }
            }
        }
        return src;
    }

    public void score(final ScoringData request,
                      @Nonnull final String sessionId,
                      @Nonnull final HttpContext httpContext,
                      @Nonnull final RequestsListener requestsListener,
                      @Nonnull final Logger logger,
                      final boolean isSave,
                      FutureCallback<Artefacts> resolutionCallback) throws IOException, LuaException,
            UnknownChannelException {
        @Nonnull final PrototypesManager prototypesManager = this.prototypesManager.get();

        final GenericAutoCloseableChain<RuntimeException> closeableChain = new GenericAutoCloseableChain<>();

        @Nonnull final ChannelWithCrossChannels channelWithCrossChannels = prototypesManager.getChannel(request);
        @Nonnull final Channel channel = channelWithCrossChannels.getChannel();
        @Nonnull final ImmutableChannelConfig channelConfig = channel.getConfig();

        @Nullable final EntriesDeque.Entry entry =
                channelWithCrossChannels.getChannel().entries().popEntryOrCreate(isSave ?
                        EntriesDeque.AppRoot.SAVE : EntriesDeque.AppRoot.SCORE);

        @Nonnull final Logger deliveryLogger;
        {
            @Nullable final Logger channelDeliveryLogger = channel.deliveryLogger();
            if (channelDeliveryLogger != null) {
                deliveryLogger = channelDeliveryLogger;
            } else {
                deliveryLogger = server.deliveryLogger();
            }
        }

        @Nonnull final EntriesDeque.Context context = new EntriesDeque.Context(
                channel,
                prototypesManager,
                request,
                logger);
        {
            context.updateContext(prototypesManager.getListsProvider(), prototypesManager);
            context.updateContext(models);
        }

        {
            if (!isSave) {
                context.artefacts().updateRequests().add(new CountersUpdateRequest(request, channelConfig));
            } else {
                context.artefacts().updateRequests().add(new TransactionUpdateRequest(request, channelConfig));
            }

            {
                @Nullable final ServiceVerificationLevelUpdateRequest serviceWideStatsRequest =
                        ServiceVerificationLevelUpdateRequest.make(request, channelConfig.storageLoginIdService());
                if (serviceWideStatsRequest != null) {
                    context.artefacts().updateRequests().add(serviceWideStatsRequest);
                }
            }

            context.artefacts().updateRequests().add(new TransactionCreateRequest(
                    request,
                    storageClient.fieldsToFetch().get(channelConfig.storageService()),
                    context.artefacts(),
                    channelConfig,
                    isSave ? RequestType.SAVE : RequestType.MAIN));
        }

        if (entry != null) {
            closeableChain.add(entry);
        }

        final JsonWriterSb logData = request.makeBaseLogData(channelConfig.fieldsToExclude());

        closeableChain.add(() -> {
            try (logData) {
                logData.key("request");
                logData.value(isSave ? "save" : "score");

                logData.key("channel_uri");
                logData.value(context.channel().getConfig().channelUri());

                logData.key("storage_service");
                logData.value(context.channel().getConfig().storageService());

                logData.key("session_id");
                logData.value(sessionId);

                {
                    final LuaTable log = context.artefacts().rootLog();
                    for (final LuaValue key : log.keys()) {
                        logData.key(key.tojstring());
                        logData.value(JsonUtils.luaAsJson(log.get(key)));
                    }
                }

                {
                    final String queues = context.artefacts().getQueues().toString();
                    if (!queues.isBlank()) {
                        logData.key("queues");
                        logData.value(queues);
                    }
                }

                logData.key("lua_resolution");
                logData.value(context.artefacts().getResolution().asJson());

                logData.key("comments");
                logData.value(context.artefacts().getComments());

                channel.writePaymentLog(request);

                logger.severe(request.shortInfo());
            } catch (IOException e) {
                logger.log(Level.WARNING, "fail to close dlv logger ", e);
            }

            deliveryLogger.fine(logData.toString());

            channelWithCrossChannels.getChannel().yasmTuner().addOurResolution(context.artefacts().getResolution()
                    .getResolutionCode());
            channelWithCrossChannels.getChannel().solomonTuner().pushResolution(context.artefacts().getResolution()
                    .getResolutionCode());
        });

        Waterfall.<Void,
                SourcesCollector.Sources,
                Artefacts,
                Artefacts,
                Void,
                String>waterfall(
                (cb) -> new Loop(
                        entry,
                        context,
                        httpContext,
                        requestsListener,
                        logger,
                        clients,
                        executor,
                        cb).execute(),
                (cb, unused) -> {
                    if (entry != null && !context.artefacts().interrupted()) {
                        sourcesCollector.execute(
                                context.scoringData(),
                                channelWithCrossChannels,
                                context.artefacts(),
                                requestsListener,
                                httpContext,
                                isSave,
                                cb,
                                logger
                        );
                    } else {
                        cb.completed(SourcesCollector.Sources.EMPTY);
                    }
                },
                (cb, sources) -> {
                    if (entry == null || context.artefacts().interrupted()) {
                        cb.completed(context.artefacts());
                        return;
                    }

                    final List<Map.Entry<ImmutableChannelConfig, AggregatesBatch>> aggregatedData =
                            sources.aggregates();

                    for (Map.Entry<ImmutableChannelConfig, AggregatesBatch> kv : aggregatedData) {
                        context.updateContext(kv.getValue(), prototypesManager);
                    }

                    {
                        @Nullable final RblData rblData = sources.rblData();
                        final Aggregates aggregates = AggregatedData.calcAggregates(
                                context.channel().getConfig(),
                                aggregatedData,
                                context.scoringData(),
                                rblData,
                                context.channel().solomonTuner());
                        logData.key("aggregates");
                        logData.value((JsonValue) aggregates.asShortJson());
                        logData.key("structured_aggregates");
                        logData.value(new JsonUtils.LuaAsJson(aggregates.structuredFull()));
                        logData.key("aggrs_are_valid");
                        logData.value(aggregates.isValid());
                        logData.key("counters_are_valid");
                        logData.value(aggregates.areCountersValid());

                        if (rblData != null) {
                            logData.key("rbl");
                            logData.value(rblData.toJson());
                        }

                        if (!isSave) {
                            context.artefacts().updateRequests().add(
                                    new CreateAggregatesRequest(
                                            context.scoringData(),
                                            aggregates,
                                            context.artefacts(),
                                            context.channel().getConfig()));
                        }

                        context.updateContext(
                                aggregates,
                                rblData);
                    }
                    for (Map.Entry<ImmutableChannelConfig, AggregatesBatch> configAggregatesBatchEntry :
                            aggregatedData) {
                        context.updateContext(configAggregatesBatchEntry.getValue().getLists());
                        context.updateContext(configAggregatesBatchEntry.getValue().getCounters());
                    }
                    entry.runMain(context);
                    cb.completed(context.artefacts());
                },
                (cb, artefacts) -> {
                    if (entry != null) {
                        entry.runPostAction(context);
                    }
                    cb.completed(artefacts);
                },
                (cb, artefacts) -> {
                    resolutionCallback.completed(artefacts);
                    executor.execute(() -> new Loop(
                            entry,
                            context,
                            httpContext,
                            requestsListener,
                            logger,
                            clients,
                            executor,
                            cb).execute());
                },
                (cb, unused) -> {
                    if (channelConfig.storeInDb()) {
                        storageClient.save(
                                request.getExternalId(),
                                context.artefacts().updateRequests(),
                                requestsListener,
                                httpContext,
                                cb);
                    }
                },
                new ClosingFutureCallback<>(
                        EmptyFutureCallback.instance(),
                        closeableChain));
    }

    @Override
    public void extract(SoFactorsExtractorContext context, SoFunctionInputs inputs, FutureCallback<?
            super List<SoFactor<?>>> callback) {
        try {
            final JsonMap src = inputs.get(0, JsonMapSoFactorType.JSON_MAP);
            final SmtpEnvelopeHolder envelope =
                    inputs.get(1, SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
            score(new ScoringData(src, toRubConverter),
                    envelope.envelope().getConnectInfo().getSessionId(),
                    context.httpContext(),
                    context.requestsListener(),
                    context.logger(),
                    false,
                    new AbstractFilterFutureCallback<Artefacts, List<SoFactor<?>>>(callback) {
                        @Override
                        public void completed(Artefacts artefacts) {
                            callback.completed(
                                    Collections.singletonList(
                                            JsonMapSoFactorType.JSON_MAP.createFactor(
                                                    artefacts.getResolution().asJson(
                                                            BasicContainerFactory.INSTANCE))));
                        }
                    });
        } catch (JsonException | IOException | UnknownChannelException | LuaException e) {
            callback.failed(e);
        }
    }

    @Override
    public List<SoFactorType<?>> inputs() {
        return INPUTS;
    }

    @Override
    public List<SoFactorType<?>> outputs() {
        return OUTPUTS;
    }

    @Override
    public void registerInternals(SoFactorsExtractorsRegistry typesRegistry) throws ConfigException {
        typesRegistry.typesRegistry().registerSoFactorType(JsonMapSoFactorType.JSON_MAP);
        typesRegistry.typesRegistry().registerSoFactorType(SmtpEnvelopeSoFactorType.SMTP_ENVELOPE);
    }

    @Override
    public void close() {

    }

    public Handler makeScoreHandler() {
        return new Handler(false, ScoreCallback::new);
    }

    public Handler makeSaveHandler() {
        return new Handler(true, SaveCallback::new);
    }

    public MultiHandler makeMultiScoreHandler() {
        return new MultiHandler();
    }

    public class Handler extends JsonHandler<HttpProxy<?>> {
        private final boolean isSave;
        private final Function<ProxySession, FutureCallback<Artefacts>> callbackSupplier;

        public Handler(boolean isSave,
                       Function<ProxySession, FutureCallback<Artefacts>> callbackSupplier) {
            super(Scorer.this.server);
            this.isSave = isSave;
            this.callbackSupplier = callbackSupplier;
        }

        @Override
        public void handle(
                final JsonObject jsonRequest,
                final HttpAsyncExchange exchange,
                final HttpContext context)
                throws HttpException {
            final ProxySession session = makeSession(exchange, context);
            final ScoringData request;
            try {
                final JsonMap preparedData = removeCardInfoForPays(jsonRequest.asMap());
                {
                    final String app = session.params().getOrNull("app");
                    if (app != null) {
                        preparedData.put("request", APP);
                        preparedData.put("channel", APP);
                        preparedData.put("sub_channel", new JsonString(app));
                    }
                }
                request = new ScoringData(preparedData, toRubConverter);
            } catch (JsonException e) {
                session.logger().log(Level.WARNING, "fail to parse scoring data", e);
                throw new BadRequestException(e);
            }

            try {
                score(
                        request,
                        session.context().getAttribute(HttpServer.SESSION_ID).toString(),
                        session.context(),
                        session.listener(),
                        session.logger(),
                        isSave,
                        callbackSupplier.apply(session));
            } catch (UnknownChannelException e) {
                throw new BadRequestException(e);
            } catch (RuntimeException | IOException | LuaException e) {
                throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e);
            }
        }
    }

    public class MultiHandler extends JsonHandler<HttpProxy<?>> {
        public MultiHandler() {
            super(Scorer.this.server);
        }

        @Override
        public void handle(
                final JsonObject jsonRequest,
                final HttpAsyncExchange exchange,
                final HttpContext context)
                throws HttpException {
            final JsonMap base;
            final JsonList patches;
            try {
                base = jsonRequest.get("base").asMap();
                patches = jsonRequest.get("patches").asList();
            } catch (JsonBadCastException e) {
                throw new BadRequestException(e);
            }

            final ProxySession session = makeSession(exchange, context);

            session.params().getString("consumer");

            final MultiFutureCallback<Map.Entry<Integer, Artefacts>> multiFutureCallback =
                    new MultiFutureCallback<>(new MultiScoreCallback(session));
            try {

                for (int i = 0; i < patches.size(); i++) {
                    final JsonMap patch = patches.get(i).asMap();
                    for (var entry : base.entrySet()) {
                        patch.merge(entry.getKey(), entry.getValue(),
                                (oldValue, newValue) -> oldValue);
                    }

                    final JsonMap preparedPatch = removeCardInfoForPays(patch);

                    final ScoringData request = new ScoringData(preparedPatch, toRubConverter);
                    score(
                            request,
                            session.context().getAttribute(HttpServer.SESSION_ID).toString(),
                            session.context(),
                            session.listener(),
                            session.logger(),
                            false,
                            new PayloadFutureCallback<>(i, multiFutureCallback.newCallback()));
                }
                multiFutureCallback.done();

            } catch (JsonException | UnknownChannelException e) {
                multiFutureCallback.failed(e);
                throw new BadRequestException(e);
            } catch (RuntimeException | IOException | LuaException e) {
                multiFutureCallback.failed(e);
                throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e);
            }
        }
    }

    static class MultiScoreCallback
            extends AbstractProxySessionCallback<List<Map.Entry<Integer, Artefacts>>> {

        public MultiScoreCallback(@Nonnull ProxySession session) {
            super(session);
        }

        @Override
        public void completed(List<Map.Entry<Integer, Artefacts>> responses) {
            responses.sort(Comparator.comparingInt(Map.Entry::getKey));

            final StringBuilderWriter sb = new StringBuilderWriter();
            try (JsonWriterBase writer = new JsonWriter(sb)) {
                writer.startObject();
                {
                    writer.key("responses");
                    writer.startArray();
                    for (var entry : responses) {
                        writer.value(entry.getValue().getResolution().asJson());
                    }
                    writer.endArray();
                }
                writer.endObject();
            } catch (IOException e) {
                session.handleException(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e));
                return;
            }

            session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(sb.toString(),
                            ContentType.APPLICATION_JSON.withCharset(session.acceptedCharset())));
        }
    }
}
