package ru.yandex.antifraud;

import java.io.IOException;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;

import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.factors.ScoringDataFactorType;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.so.factors.BasicSoFunctionInputs;
import ru.yandex.mail.so.factors.LoggingFactorsAccessViolationHandler;
import ru.yandex.mail.so.factors.SoFactor;
import ru.yandex.mail.so.factors.http.HttpSoFactorsExtractorContext;
import ru.yandex.parser.mail.errors.ErrorsLogger;

public class So2HttpHandler extends JsonHandler<AntiFraudHttpServer> {
    private final Scorer scorer;

    public So2HttpHandler(final AntiFraudHttpServer server, Scorer scorer) {
        super(server);
        this.scorer = scorer;
    }


    @Override
    public void handle(
            final JsonObject jsonRequest,
            final HttpAsyncExchange exchange,
            final HttpContext context) throws HttpException {

        final ProxySession session =
                new BasicProxySession(server, exchange, context);
        final String extractorName =
                session.params().getString("extractor-name", "main");

        final ScoringData scoringData;
        try {
            scoringData = new ScoringData(jsonRequest.asMap(), scorer.toRubConverter());
        } catch (JsonException e) {
            session.logger().log(Level.WARNING, "fail to make parse scoring data", e);
            session.handleException(new BadRequestException(e));
            return;
        }

        final ErrorsLogger errorsLogger = new ErrorsLogger(session.logger());

        final LoggingFactorsAccessViolationHandler accessViolationHandler =
                new LoggingFactorsAccessViolationHandler(
                    server.violationsCounter(),
                    session.logger());


        final HttpSoFactorsExtractorContext extractorContext =
                new HttpSoFactorsExtractorContext(
                        session,
                        accessViolationHandler,
                        errorsLogger,
                        server.threadPool());

        server.extractorsRegistry().getExtractor(extractorName).extract(
                extractorContext,
                new BasicSoFunctionInputs(
                        accessViolationHandler,
                        ScoringDataFactorType.SCORING_DATA.createFactor(scoringData)),
                new Callback(session, JsonType.NORMAL));
    }

    private static class Callback
            extends AbstractProxySessionCallback<List<SoFactor<?>>> {
        private final JsonType jsonType;

        Callback(final ProxySession session,
                 final JsonType jsonType) {
            super(session);
            this.jsonType = jsonType;
        }

        @Override
        public void completed(final List<SoFactor<?>> result) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = jsonType.create(sbw)) {
                Object value = null;
                if (result.size() == 1) {
                    SoFactor<?> factor = result.get(0);
                    if (factor != null) {
                        value = factor.value();
                    }
                }
                writer.value(value);
            } catch (IOException e) {
                failed(e);
                return;
            }
            session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                            sbw.toString(),
                            ContentType.APPLICATION_JSON));
        }
    }
}

