package ru.yandex.antifraud;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.protocol.HttpContext;

import ru.yandex.antifraud.aggregates.AggregatesBatch;
import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.channel.ChannelWithCrossChannels;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.rbl.RblClient;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.antifraud.storage.StorageClient;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.nio.client.RequestsListener;

public class SourcesCollector {
    @Nonnull
    private final StorageClient storageClient;
    @Nonnull
    private final RblClient rblClient;

    public SourcesCollector(@Nonnull StorageClient storageClient,
                            @Nonnull RblClient rblClient) {
        this.storageClient = storageClient;
        this.rblClient = rblClient;
    }

    void execute(@Nonnull ScoringData scoringData,
                 @Nonnull ChannelWithCrossChannels channelWithCrossChannels,
                 @Nonnull Artefacts artefacts,
                 @Nonnull RequestsListener listener,
                 @Nonnull HttpContext context,
                 final boolean isSave,
                 @Nonnull FutureCallback<? super Sources> sourcesCallback,
                 @Nonnull Logger logger) {

        final DoubleFutureCallback<List<
                Map.Entry<ImmutableChannelConfig, AggregatesBatch>>, RblData> aggregatesAndTailCallback =
                new DoubleFutureCallback<>(new SourcesCallback(sourcesCallback));

        {
            final FutureCallback<List<
                    Map.Entry<ImmutableChannelConfig, AggregatesBatch>>> aggregationsCallback =
                    aggregatesAndTailCallback.first();

            storageClient.search(
                    channelWithCrossChannels,
                    scoringData,
                    artefacts.getLists(),
                    artefacts.getCounters(),
                    listener,
                    context,
                    isSave,
                    aggregationsCallback,
                    logger
            );

            {
                final FutureCallback<RblData> rblCallback = aggregatesAndTailCallback.second();
                final String ip = scoringData.getIp();
                if (ip != null) {
                    final RblClient client = rblClient.adjust(context);
                    client.getIpInfo(
                            ip,
                            listener.createContextGeneratorFor(client),
                            new ErrorSuppressingFutureCallback<>(rblCallback, (RblData) null),
                            logger);
                } else {
                    rblCallback.completed(null);
                }
            }
        }
    }

    static class Sources {
        public static final Sources EMPTY = new Sources(Collections.emptyList(), null);

        @Nonnull
        private final List<Map.Entry<ImmutableChannelConfig, AggregatesBatch>> aggregates;

        @Nullable
        private final RblData rblData;

        Sources(@Nonnull List<Map.Entry<ImmutableChannelConfig, AggregatesBatch>> aggregates,
                @Nullable RblData rblData) {
            this.aggregates = aggregates;
            this.rblData = rblData;
        }

        @Nonnull
        public List<Map.Entry<ImmutableChannelConfig, AggregatesBatch>> aggregates() {
            return aggregates;
        }

        @Nullable
        public RblData rblData() {
            return rblData;
        }
    }

    static class SourcesCallback extends CallbackFutureBase<Sources, Map.Entry<List<Map.Entry<ImmutableChannelConfig,
            AggregatesBatch>>, RblData>> {
        protected SourcesCallback(FutureCallback<? super Sources> callback) {
            super(callback);
        }

        @Override
        protected Sources convertResult(Map.Entry<List<Map.Entry<ImmutableChannelConfig,
                AggregatesBatch>>, RblData> result) {
            return new Sources(
                    result.getKey(),
                    result.getValue()
            );
        }
    }
}
