package ru.yandex.antifraud.config;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.lua_context_manager.config.PrototypesConfigBuilder;
import ru.yandex.antifraud.model_config.CatboostModelConfig;
import ru.yandex.antifraud.model_config.CatboostModelConfigBuilder;
import ru.yandex.antifraud.storage.config.StorageConfig;
import ru.yandex.antifraud.storage.config.StorageConfigBuilder;
import ru.yandex.antifraud.storage.config.StorageConfigDefaults;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.LoggerConfigBuilder;
import ru.yandex.logger.MinimalLoggerConfigDefaults;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractAntiFraudConfigBuilder
        <T extends AbstractAntiFraudConfigBuilder<T>>
        extends AbstractHttpProxyConfigBuilder<T>
        implements AntiFraudConfig {
    private PrototypesConfigBuilder prototypesConfig;
    private Path rulesConfig;
    private Path rulesRoot;
    private Path currenciesRatesPath;
    private LoggerConfigBuilder deliveryLog;
    private HttpHostConfigBuilder blackboxConfig;
    private HttpHostConfigBuilder furyConfig;
    private HttpHostConfigBuilder passportConfig;
    private HttpHostConfigBuilder passportSlowConfig;
    private HttpHostConfigBuilder deduplicatedPassportConfig;
    private HttpHostConfigBuilder pharmaConfig;
    private HttpHostConfigBuilder trustConfig;
    private HttpHostConfigBuilder familyConfig;
    private HttpHostConfigBuilder cacheClientConfig;
    private Map<String, StorageConfigBuilder> mainStorageConfigs;
    private Map<String, StorageConfigBuilder> uiStorageConfigs;
    private HttpHostConfigBuilder rblConfig;
    private Tvm2ClientConfigBuilder tvm2ClientConfig;
    private Map<String, CatboostModelConfigBuilder> models;

    @Nullable
    private Path uatraitsConfig;

    protected AbstractAntiFraudConfigBuilder(final AntiFraudConfig config) {
        super(config);
        deliveryLog = new LoggerConfigBuilder(config.deliveryLog());

        prototypesConfig = new PrototypesConfigBuilder(config.prototypesConfig());
        blackboxConfig = new HttpHostConfigBuilder(config.blackboxConfig());

        if (config.furyConfig() != null) {
            furyConfig = new HttpHostConfigBuilder(config.furyConfig());
        } else {
            furyConfig = null;
        }

        if (config.passportConfig() != null) {
            passportConfig = new HttpHostConfigBuilder(config.passportConfig());
        } else {
            passportConfig = null;
        }

        if (config.passportSlowConfig() != null) {
            passportSlowConfig = new HttpHostConfigBuilder(config.passportSlowConfig());
        } else {
            passportSlowConfig = null;
        }

        if (config.deduplicatedPassportConfig() != null) {
            deduplicatedPassportConfig = new HttpHostConfigBuilder(config.deduplicatedPassportConfig());
        } else {
            deduplicatedPassportConfig = null;
        }

        if (config.pharmaConfig() != null) {
            pharmaConfig = new HttpHostConfigBuilder(config.pharmaConfig());
        } else {
            pharmaConfig = null;
        }

        if (config.familyConfig() != null) {
            familyConfig = new HttpHostConfigBuilder(config.familyConfig());
        } else {
            familyConfig = null;
        }

        if (config.cacheClientConfig() != null) {
            cacheClientConfig = new HttpHostConfigBuilder(config.cacheClientConfig());
        } else {
            cacheClientConfig = null;
        }

        trustConfig = Optional.ofNullable(config.trustConfig()).map(HttpHostConfigBuilder::new).orElse(null);

        mainStorageConfigs = new HashMap<>();
        for (final Map.Entry<String, ? extends StorageConfig> kv : config.mainStorageConfigs().entrySet()) {
            mainStorageConfigs.put(kv.getKey(), new StorageConfigBuilder(kv.getValue()));
        }

        uiStorageConfigs = new HashMap<>();
        for (final Map.Entry<String, ? extends StorageConfig> kv : config.uiStorageConfigs().entrySet()) {
            uiStorageConfigs.put(kv.getKey(), new StorageConfigBuilder(kv.getValue()));
        }

        rblConfig = new HttpHostConfigBuilder(config.rblConfig());
        tvm2ClientConfig = new Tvm2ClientConfigBuilder(config.tvm2ClientConfig());

        if (!config.models().isEmpty()) {
            models = new HashMap<>(config.models().size());
            for (Map.Entry<String, ? extends CatboostModelConfig> entry : config.models().entrySet()) {
                models.put(entry.getKey(), new CatboostModelConfigBuilder(entry.getValue()));
            }
        } else {
            models = Collections.emptyMap();
        }

        uatraitsConfig = config.uatraitsConfig();

        rulesConfig = config.rulesConfig();
        rulesRoot = config.rulesRoot();
        currenciesRatesPath = config.currenciesRatesPath();
    }

    protected AbstractAntiFraudConfigBuilder(
            final IniConfig config,
            final AntiFraudConfig defaults)
            throws ConfigException, IOException, JsonException {
        super(config, defaults);
        deliveryLog =
                new LoggerConfigBuilder(
                        config.section("delivery-log"),
                        MinimalLoggerConfigDefaults.INSTANCE);
        {
            final IniConfig section = config.section("blackbox");
            blackboxConfig = new HttpHostConfigBuilder(section, defaults.blackboxConfig());
        }
        {
            final IniConfig section = config.sectionOrNull("fury");
            if (section != null) {
                furyConfig = new HttpHostConfigBuilder(section, defaults.furyConfig());
            } else {
                furyConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("passport");
            if (section != null) {
                passportConfig = new HttpHostConfigBuilder(section, defaults.passportConfig());
            } else {
                passportConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("slow-passport");
            if (section != null) {
                passportSlowConfig = new HttpHostConfigBuilder(section, defaults.passportSlowConfig());
            } else {
                passportSlowConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("deduplicated-passport");
            if (section != null) {
                deduplicatedPassportConfig = new HttpHostConfigBuilder(section, defaults.deduplicatedPassportConfig());
            } else {
                deduplicatedPassportConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("pharma");
            if (section != null) {
                pharmaConfig = new HttpHostConfigBuilder(section, defaults.pharmaConfig());
            } else {
                pharmaConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("trust");
            if (section != null) {
                trustConfig = new HttpHostConfigBuilder(section, defaults.trustConfig());
            } else {
                trustConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("family");
            if (section != null) {
                familyConfig = new HttpHostConfigBuilder(section, defaults.familyConfig());
            } else {
                familyConfig = null;
            }
        }
        {
            final IniConfig section = config.sectionOrNull("cache");
            if (section != null) {
                cacheClientConfig = new HttpHostConfigBuilder(section, defaults.cacheClientConfig());
            } else {
                cacheClientConfig = null;
            }
        }
        {
            final IniConfig section = config.section("main").section("storages");
            mainStorageConfigs = new HashMap<>();
            for (final Map.Entry<String, IniConfig> kv : section.sections().entrySet()) {
                mainStorageConfigs.put(kv.getKey(), new StorageConfigBuilder(kv.getValue(),
                        StorageConfigDefaults.INSTANCE));
            }
        }
        {
            final IniConfig section = config.section("ui").section("storages");
            uiStorageConfigs = new HashMap<>();
            for (final Map.Entry<String, IniConfig> kv : section.sections().entrySet()) {
                uiStorageConfigs.put(kv.getKey(), new StorageConfigBuilder(kv.getValue(),
                        StorageConfigDefaults.INSTANCE));
            }
        }
        {
            final IniConfig section = config.section("rbl");
            rblConfig = new HttpHostConfigBuilder(section, defaults.rblConfig());
        }
        {
            final IniConfig rulesSection = config.section("rules");
            rulesConfig = rulesSection.getInputFile("config").toPath();
            rulesRoot = rulesSection.getExistingDir("root", rulesConfig.getParent().toFile()).toPath();
            prototypesConfig = new PrototypesConfigBuilder(rulesConfig, rulesRoot);
        }
        {
            tvm2ClientConfig = new Tvm2ClientConfigBuilder(
                    config.section("tvm2"),
                    defaults.tvm2ClientConfig());
        }
        {
            final IniConfig section = config.sectionOrNull("models");
            if (section != null) {
                final Map<String, IniConfig> sections = section.sections();
                models = new HashMap<>(sections.size());

                for (Map.Entry<String, IniConfig> entry : sections.entrySet()) {
                    models.put(entry.getKey(), new CatboostModelConfigBuilder(entry.getValue()));
                }
            } else {
                models = Collections.emptyMap();
            }
        }
        {
            final IniConfig section = config.sectionOrNull("uatraits");
            if (section != null) {
                uatraitsConfig = section.getInputFile("path").toPath();
            } else {
                uatraitsConfig = defaults.uatraitsConfig();
            }
        }
        {
            final IniConfig section = config.sectionOrNull("currencies");
            currenciesRatesPath = section.getInputFile("rates-path").toPath();
        }
    }

    @Override
    public LoggerConfigBuilder deliveryLog() {
        return deliveryLog;
    }

    public void deliveryLog(LoggerConfigBuilder deliveryLog) {
        this.deliveryLog = deliveryLog;
    }

    @Nonnull
    @Override
    public Path rulesConfig() {
        return rulesConfig;
    }

    public void rulesConfig(@Nonnull Path rulesConfig) {
        this.rulesConfig = rulesConfig;
    }

    @Nonnull
    @Override
    public Path rulesRoot() {
        return rulesRoot;
    }

    public void rulesRoot(@Nonnull Path rulesRoot) {
        this.rulesRoot = rulesRoot;
    }

    @Nonnull
    @Override
    public Path currenciesRatesPath() {
        return currenciesRatesPath;
    }

    public void currenciesRatesPath(@Nonnull Path currenciesRatesPath) {
        this.currenciesRatesPath = currenciesRatesPath;
    }

    @Nonnull
    @Override
    public PrototypesConfigBuilder prototypesConfig() {
        return prototypesConfig;
    }

    public void prototypesConfig(PrototypesConfigBuilder prototypesConfig) {
        this.prototypesConfig = prototypesConfig;
    }

    @Nonnull
    @Override
    public Map<String, StorageConfigBuilder> mainStorageConfigs() {
        return mainStorageConfigs;
    }

    public void mainStorageConfigs(Map<String, StorageConfigBuilder> mainStorageConfigs) {
        this.mainStorageConfigs = mainStorageConfigs;
    }

    @Nonnull
    @Override
    public Map<String, StorageConfigBuilder> uiStorageConfigs() {
        return uiStorageConfigs;
    }

    public void uiStorageConfigs(Map<String, StorageConfigBuilder> uiStorageConfigs) {
        this.uiStorageConfigs = uiStorageConfigs;
    }

    @Nonnull
    @Override
    public HttpHostConfigBuilder blackboxConfig() {
        return blackboxConfig;
    }

    public void blackboxConfig(HttpHostConfigBuilder blackboxConfig) {
        this.blackboxConfig = blackboxConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder furyConfig() {
        return furyConfig;
    }

    public void furyConfig(HttpHostConfigBuilder furyConfig) {
        this.furyConfig = furyConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder passportConfig() {
        return passportConfig;
    }

    public void passportConfig(HttpHostConfigBuilder passportConfig) {
        this.passportConfig = passportConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder passportSlowConfig() {
        return passportSlowConfig;
    }

    public void passportSlowConfig(HttpHostConfigBuilder passportSlowConfig) {
        this.passportSlowConfig = passportSlowConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder deduplicatedPassportConfig() {
        return deduplicatedPassportConfig;
    }

    public void deduplicatedPassportConfig(HttpHostConfigBuilder deduplicatedPassportConfig) {
        this.deduplicatedPassportConfig = deduplicatedPassportConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder pharmaConfig() {
        return pharmaConfig;
    }

    public void pharmaConfig(HttpHostConfigBuilder pharmaConfig) {
        this.pharmaConfig = pharmaConfig;
    }

    @Nonnull
    @Override
    public HttpHostConfigBuilder rblConfig() {
        return rblConfig;
    }

    public void rblConfig(HttpHostConfigBuilder rblConfig) {
        this.rblConfig = rblConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder trustConfig() {
        return trustConfig;
    }

    public void trustConfig(HttpHostConfigBuilder trustConfig) {
        this.trustConfig = trustConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder familyConfig() {
        return familyConfig;
    }

    public void familyConfig(HttpHostConfigBuilder familyConfig) {
        this.familyConfig = familyConfig;
    }

    @Nullable
    @Override
    public HttpHostConfigBuilder cacheClientConfig() {
        return cacheClientConfig;
    }

    public void cacheClientConfig(HttpHostConfigBuilder cacheClientConfig) {
        this.cacheClientConfig = cacheClientConfig;
    }


    @Nonnull
    @Override
    public Tvm2ClientConfigBuilder tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public void tvm2ClientConfig(Tvm2ClientConfigBuilder tvm2ClientConfig) {
        this.tvm2ClientConfig = tvm2ClientConfig;
    }

    @Nonnull
    @Override
    public Map<String, CatboostModelConfigBuilder> models() {
        return models;
    }

    public T models(Map<String, ? extends CatboostModelConfig> models) {
        this.models = new HashMap<>(models.size());

        for (Map.Entry<String, ? extends CatboostModelConfig> entry : models.entrySet()) {
            this.models.put(entry.getKey(), new CatboostModelConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Nullable
    @Override
    public Path uatraitsConfig() {
        return uatraitsConfig;
    }

    public T uatraitsConfig(final Path uatraitsConfig) {
        this.uatraitsConfig = uatraitsConfig;
        return self();
    }
}

