from datetime import timedelta

from django import forms
from django.forms import widgets
from django.utils.datetime_safe import datetime

from mail.so.daemons.antifraud.antifraud_django.sofraud.library import utils

form_control_attrs = {'class': 'form-control'}
file_form_control_attrs = {'class': 'form-control-file'}


class ChannelForm(forms.Form):
    channel = forms.ChoiceField()

    def __init__(self, *args, **kwargs):
        self.channels = kwargs.pop("channels")
        super(ChannelForm, self).__init__(*args, **kwargs)
        channels_choices = [(choice, choice) for choice in self.channels]
        self.fields['channel'] = forms.ChoiceField(choices=channels_choices,
                                                   widget=widgets.Select(
                                                       attrs={'class': 'form-control',
                                                              'size': '11'}))


class ChannelFormWithAny(forms.Form):
    channel = forms.MultipleChoiceField(required=False)

    def __init__(self, *args, **kwargs):
        self.channels = kwargs.pop("channels")
        super(ChannelFormWithAny, self).__init__(*args, **kwargs)
        channels_choices = [(choice, choice) for choice in self.channels]
        self.fields['channel'] = forms.MultipleChoiceField(choices=channels_choices,
                                                           required=False,
                                                           widget=widgets.SelectMultiple(
                                                               attrs={'class': 'form-control',
                                                                      'size': '11'}))


list_name_attrs = {"list": "lists-names"}
list_name_attrs.update(form_control_attrs)


class NewListForm(ChannelForm):
    list_name = forms.CharField(max_length=200, widget=widgets.TextInput(attrs=list_name_attrs))
    days = forms.DurationField(widget=widgets.TextInput(attrs=form_control_attrs))
    reason = forms.CharField(max_length=200, widget=widgets.TextInput(attrs=form_control_attrs))

    text_items = forms.CharField(widget=widgets.Textarea(attrs=form_control_attrs), required=False,
                                 label="'\\s+' separated values")
    file_items = forms.FileField(widget=widgets.FileInput(attrs=file_form_control_attrs),
                                 required=False,
                                 label="File with '\\s+' separated values")


class ListsRequestForm(ChannelForm):
    list_name = forms.CharField(max_length=200, widget=widgets.TextInput(attrs=list_name_attrs))

    value = forms.CharField(max_length=200,
                            widget=widgets.TextInput(attrs=form_control_attrs),
                            required=False)

    limit = forms.IntegerField(
        widget=widgets.TextInput(
            attrs=form_control_attrs),
        required=True,
        initial=100)


class TransactionsRequestForm(ChannelFormWithAny):
    query_attrs = {"list": "search-fields"}
    query_attrs.update(form_control_attrs)
    query = forms.CharField(widget=widgets.TextInput(attrs=query_attrs), required=False)
    limit = forms.IntegerField(
        widget=widgets.TextInput(
            attrs=form_control_attrs
        ),
        required=False,
        initial=100
    )

    since = forms.CharField(
        initial=lambda: utils.datetime_to_str(datetime.now() - timedelta(minutes=5)),
        widget=widgets.TextInput(attrs=form_control_attrs))

    until = forms.CharField(
        required=False,
        widget=widgets.TextInput(attrs=form_control_attrs))


class VerificationLevelRequestForm(forms.Form):
    query = forms.CharField(widget=widgets.TextInput(attrs=form_control_attrs), required=False)
    limit = forms.IntegerField(
        widget=widgets.TextInput(
            attrs=form_control_attrs
        ),
        required=False,
        initial=100
    )


class OffChallengeForm(forms.Form):
    uid = forms.IntegerField(widget=widgets.TextInput(attrs=form_control_attrs), required=True)
    reason = forms.CharField(max_length=200,
                             widget=widgets.TextInput(attrs=form_control_attrs),
                             required=True)


class RefundForm(forms.Form):
    purchase_token = forms.CharField(widget=widgets.TextInput(attrs=form_control_attrs), required=True)
    uid = forms.IntegerField(widget=widgets.TextInput(attrs=form_control_attrs), required=True)
    reason = forms.CharField(max_length=200,
                             widget=widgets.TextInput(attrs=form_control_attrs),
                             required=True)
