import json
import logging
import re
from datetime import timedelta
from functools import wraps
from urllib.parse import urlencode, unquote_plus

from asgiref.sync import async_to_sync
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied
from django.core.serializers.json import DjangoJSONEncoder
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404, render
from django.urls import reverse
from django.utils.datetime_safe import datetime
from django.utils.decorators import method_decorator
from django.views.generic import FormView
from django.views.generic.base import (
    TemplateView
)
from django_yauth.decorators import yalogin_required

from mail.so.daemons.antifraud.antifraud_django.sofraud.forms.forms import (
    NewListForm,
    ListsRequestForm,
    TransactionsRequestForm,
    VerificationLevelRequestForm,
    OffChallengeForm,
    RefundForm,
)
from mail.so.daemons.antifraud.antifraud_django.sofraud.library import utils, payments
from mail.so.daemons.antifraud.antifraud_django.sofraud.models.log import Log
from mail.so.daemons.antifraud.antifraud_django.sofraud.models.refund_log import RefundLog

logger = logging.getLogger(__name__)

class_based_login_required = method_decorator(yalogin_required, name='dispatch')


def access_decorator(pattern: str):
    def decorator(func):
        @wraps(func)
        def inner(*args, **kwargs):
            request = kwargs.get('request') or args[0]

            if utils.login_has_rights(request.yauser.login, pattern):
                return func(*args, **kwargs)
            raise PermissionDenied()

        return inner

    return decorator


channels_access_decorator = method_decorator(
    [
        yalogin_required,
        access_decorator(r"af\.set\..+")
    ],
    name='dispatch')
support_access_decorator = method_decorator(
    [
        yalogin_required,
        access_decorator(r"af\.group\.support")
    ],
    name='dispatch')
refunder_access_decorator = method_decorator(
    [
        yalogin_required,
        access_decorator(r"af\.group\.refunder")
    ],
    name='dispatch')
trust_access_decorator = method_decorator(
    [
        yalogin_required,
        access_decorator(r"af\.set\.trust")
    ],
    name='dispatch')


@class_based_login_required
class IndexView(TemplateView):
    template_name = "index.html"


class InitialFormView(FormView):
    def get_initial(self):
        return {k: unquote_plus(value) for k, value in self.request.GET.items() if value}

    def get(self, request, *args, **kwargs):
        return self.render_to_response(self.get_context_data(**kwargs))


@class_based_login_required
class TestView(TemplateView):
    template_name = "test.html"

    def get(self, request, *args, **kwargs):
        logger.info(request.yauser.login)
        user = get_object_or_404(get_user_model(), username=request.yauser.login)

        kwargs['roles'] = utils.get_user_channels(user)
        kwargs['user'] = user
        return super(TestView, self).get(request, *args, **kwargs)


class StaffMemberFormView(InitialFormView):
    template_name = 'boostrap_form.html'

    def get_initial(self):
        initial = super(StaffMemberFormView, self).get_initial()

        channel = initial.get('channel')
        try:
            if channel is not None:
                initial['channel'] = json.loads(channel)
        except Exception:
            logger.exception(f"cannot parse json from {channel}")

        return initial

    def get_form_kwargs(self):
        kwargs = super(StaffMemberFormView, self).get_form_kwargs()
        kwargs['channels'] = utils.get_user_channels(
            get_object_or_404(get_user_model(), username=self.request.yauser.login)
        )
        return kwargs


MS_PER_DAY = timedelta(days=1).total_seconds() * 1000
MS_PER_HOUR = timedelta(hours=1).total_seconds() * 1000


@class_based_login_required
@channels_access_decorator
class NewListView(StaffMemberFormView):
    form_class = NewListForm
    template_name = 'lists_update_template.html'

    re_space = re.compile("\\s+")

    @staticmethod
    def gen_items_batches(form, batch_size):
        batch = []

        srcs = [
            form.cleaned_data["text_items"],
        ]

        file = form.cleaned_data.get("file_items")
        if file:
            srcs.append(file.read().decode('utf-8'))

        for src in srcs:
            for item in NewListView.re_space.split(src):
                if item:
                    batch.append(item)
                    if len(batch) >= batch_size:
                        yield batch
                        batch = []

        if len(batch) > 0:
            yield batch

    @async_to_sync
    async def form_valid(self, form):
        now = utils.now_milliseconds()
        saved_count = 0

        channel, sub_channel = utils.split_channel_uri(form.data["channel"])
        async with utils.backend_session(self.kwargs['project']) as session:
            for batch in self.gen_items_batches(form, 1000):
                await utils.update_list(self.kwargs['project'],
                                        session,
                                        channel,
                                        sub_channel,
                                        form.data["list_name"],
                                        self.request.yauser.login,
                                        now,
                                        now + int(form.data["days"]) * MS_PER_DAY,
                                        batch,
                                        form.data["reason"])
                saved_count += len(batch)

        context = {
            "message": f"Successfully saved {saved_count} items"
        }

        return render(self.request, 'update_template.html', context)


@class_based_login_required
@channels_access_decorator
class ListsSearchView(StaffMemberFormView):
    form_class = ListsRequestForm
    template_name = 'lists_template.html'

    @async_to_sync
    async def form_valid(self, form):
        action = self.request.POST['action'].lower()

        if action == "search":
            reversed_url = reverse(
                'mail.so.daemons.antifraud.antifraud_django.sofraud:lists_search',
                kwargs=self.kwargs)

            query_params = {
                'channel': form.cleaned_data['channel'],
                'list_name': form.cleaned_data['list_name'],
                'limit': form.cleaned_data['limit'],
            }

            value = form.cleaned_data.get('value')
            if value:
                query_params['value'] = value

            query_params = urlencode(query_params)

            return HttpResponseRedirect(reversed_url + '?' + query_params)

        if action == "delete":
            channel, sub_channel = utils.split_channel_uri(form.cleaned_data["channel"])
            value = form.cleaned_data["value"]
            async with utils.backend_session(self.kwargs['project']) as session:
                await utils.delete_list(self.kwargs['project'],
                                        session,
                                        channel,
                                        sub_channel,
                                        form.cleaned_data["list_name"],
                                        [value] if value else None)

            context = {
                "message": "Successfully deleted"
            }

            return render(self.request, 'update_template.html', context)

    def get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: unquote_plus(value) for k, value in self.request.GET.items()}

        list_name = request_get_data.get("list_name")

        if list_name:
            context['items'] = utils.get_lists_sync(
                self.kwargs['project'],
                request_get_data["channel"],
                request_get_data["list_name"],
                request_get_data.get("value"),
                request_get_data["limit"],
            )

        context['now'] = datetime.now()

        return self.render_to_response(context)


@class_based_login_required
@channels_access_decorator
class TransactionsSearchView(StaffMemberFormView):
    template_name = 'transactions_template.html'
    form_class = TransactionsRequestForm

    @async_to_sync
    async def form_valid(self, form):
        reversed_url = reverse(
            'mail.so.daemons.antifraud.antifraud_django.sofraud:transactions_search',
            kwargs=self.kwargs)

        since = form.cleaned_data.get('since')
        try:
            utils.datetime_from_str(since)
        except Exception:
            logger.exception(f"error while parsing 'since'={since}")
            since = utils.datetime_to_str(datetime.now() - timedelta(minutes=5))

        query_params = {
            'channel': json.dumps(form.cleaned_data.get('channel')),
            'query': form.cleaned_data.get('query'),
            'limit': form.cleaned_data.get('limit'),
            'since': since
        }

        until = form.cleaned_data.get('until')
        try:
            utils.datetime_from_str(until)
            query_params["until"] = until
        except Exception:
            logger.exception("error while parsing 'until'")

        query_params = urlencode(query_params)

        return HttpResponseRedirect(reversed_url + '?' + query_params)

    def get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: unquote_plus(value) for k, value in request.GET.items()}

        query = request_get_data.get("query")

        if query is not None:
            if query:
                query = "(" + query + ")"

            since = request_get_data.get('since')
            until = request_get_data.get('until', utils.datetime_to_str(datetime.now()))
            if since and until:
                if query:
                    query += " AND "

                query += utils.make_timestamp_query(
                    utils.datetime_from_str(since),
                    utils.datetime_from_str(until))

            channel = json.loads(request_get_data["channel"])
            response, errors = utils.get_transactions_by_channels_sync(
                self.kwargs['project'],
                tuple(channel if channel else self.get_form().channels),
                "MAIN",
                request_get_data["limit"],
                query)

            response_save, errors_save = utils.get_transactions_by_channels_sync(
                self.kwargs['project'],
                tuple(channel if channel else self.get_form().channels),
                "SAVE",
                request_get_data["limit"],
                query)

            items = response['transactions'] + response_save['transactions']

            context['items'] = items
            context['items_str'] = json.dumps(items,
                                              cls=DjangoJSONEncoder)
            context['errors'] = errors + errors_save

        return self.render_to_response(context)


@class_based_login_required
@channels_access_decorator
class TransactionView(TemplateView):
    template_name = 'transaction_template.html'

    @async_to_sync
    async def update_with_aggr(self, tx_id, channel_uri, context):
        is_save = "SAVE" in tx_id
        async with utils.backend_session(self.kwargs['project']) as session:
            transactions = await utils.get_transactions(self.kwargs['project'],
                                                        session,
                                                        channel_uri,
                                                        is_save and "SAVE" or "MAIN", 1,
                                                        'id:"txn_' + tx_id + '"')

            if len(transactions) > 0:
                context['transaction'] = transactions[0]
                txn_extid = transactions[0].get("txn_extid")
                if txn_extid:
                    try:
                        logger.info(f"trying found {txn_extid} payment")
                        context['payment'] = utils.remove_nones(payments.get_payments_by_id(txn_extid))
                        logger.info(f"found {context['payment']} payment")
                    except:
                        logger.exception("payment error")

            if not is_save:
                transactions = await utils.get_transactions(self.kwargs['project'],
                                                            session,
                                                            channel_uri,
                                                            "AGGRS", 1,
                                                            'id:"txn_aggrs_' + tx_id + '"',
                                                            prefix=0)

                if transactions:
                    context['aggregates'] = transactions[0]

    def get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: unquote_plus(value) for k, value in self.request.GET.items()}

        tx_id = request_get_data['id']

        tx_id = tx_id.removeprefix("txn_aggrs_").removeprefix("txn_")

        self.update_with_aggr(tx_id, request_get_data["channel"], context)

        return self.render_to_response(context)


@class_based_login_required
@channels_access_decorator
class VerificationLevelsSearchView(InitialFormView):
    template_name = 'verification_levels_template.html'
    form_class = VerificationLevelRequestForm

    def form_valid(self, form):
        reversed_url = reverse(
            'mail.so.daemons.antifraud.antifraud_django.sofraud:verification_levels',
            kwargs=self.kwargs)

        query_params = {
            'query': form.cleaned_data.get('query'),
            'limit': form.cleaned_data.get('limit'),
        }

        query_params = urlencode(query_params)

        return HttpResponseRedirect(reversed_url + '?' + query_params)

    def get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: unquote_plus(value) for k, value in request.GET.items()}

        query = request_get_data.get("query")

        if query is not None:
            try:
                response = utils.get_verification_levels_sync(
                    self.kwargs['project'],
                    query,
                    request_get_data["limit"])

                context['items'] = response['levels']
            except Exception as e:
                logger.exception(f"{locals()}", exc_info=e)
                context['errors'] = [e]

        return self.render_to_response(context)


@class_based_login_required
@support_access_decorator
class OffChallengeView(InitialFormView):
    template_name = 'off_challenge.html'
    form_class = OffChallengeForm

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)

        context["logs"] = Log.objects.order_by('-timestamp')[:100]

        return context

    def form_valid(self, form):
        uid = form.cleaned_data["uid"]
        reason = form.cleaned_data["reason"]
        user = get_object_or_404(get_user_model(), username=self.request.yauser.login)
        utils.update_counter_sync(self.kwargs['project'],
                                  "auth",
                                  "login",
                                  "uid_off_challenge",
                                  uid)

        Log(user=user,
            data={
                "channel": "auth",
                "sub_channel": "login",
                "counter": "uid_off_challenge",
                "uid": uid,
                "reason": reason
            }).save()

        return render(self.request, 'update_template.html', {
            "message": "Successfully off"
        })


@class_based_login_required
@trust_access_decorator
class PredictorView(TemplateView):
    template_name = "predictor.html"

    def get(self, request, *args, **kwargs):
        logger.info(request.yauser.login)
        user = get_object_or_404(get_user_model(), username=request.yauser.login)

        kwargs['roles'] = utils.get_user_channels(user)
        kwargs['user'] = user
        return super(PredictorView, self).get(request, *args, **kwargs)


@class_based_login_required
@refunder_access_decorator
class RefundView(InitialFormView):
    template_name = 'refund.html'
    form_class = RefundForm

    def form_valid(self, form):
        uid = form.cleaned_data["uid"]
        purchase_token = form.cleaned_data["purchase_token"]
        reason = form.cleaned_data["reason"]

        logger.info(f"refund:{locals()}")

        user = get_object_or_404(get_user_model(), username=self.request.yauser.login)

        log_data = {
            "uid": uid,
            "purchase_token": purchase_token,
            "reason": reason,
        }
        try:
            summary = utils.make_refund_and_start(purchase_token, uid, reason)
            log_data["summary"] = summary

            return render(self.request, 'update_template.html', {
                "message": summary
            })
        except Exception as e:
            log_data["error"] = str(e)

            return render(self.request, 'error_template.html', {
                "error": e
            })
        finally:
            RefundLog(user=user, data=log_data).save()
