package ru.yandex.antifraud.aggregates;

import java.time.Instant;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaTable;

import ru.yandex.antifraud.aggregates_v2.AggregatedTimeRange;
import ru.yandex.antifraud.aggregates_v2.AggregatorConsumer;
import ru.yandex.antifraud.aggregates_v2.ConglomerateAggregatorConsumer;
import ru.yandex.antifraud.aggregates_v2.aggregates_config.ImmutableAggregatorsConfig;
import ru.yandex.antifraud.data.CollapsedAggregates;
import ru.yandex.antifraud.data.Counters;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.data.StorageData;
import ru.yandex.antifraud.lua_context_manager.SolomonTuner;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.lua.util.JsonUtils;

public class AggregatesBuilder {
    @Nonnull
    private final Counters counters;
    @Nullable
    private final RblData rblData;
    @Nonnull
    private final Instant now;

    @Nonnull
    private final ConglomerateAggregatorConsumer aggregatorConsumer;

    private final boolean hasCollapsedAggrs;
    private final boolean hasAggrs;
    private final boolean hasCounters;

    public AggregatesBuilder(
            @Nonnull ScoringData request,
            @Nonnull Counters counters,
            @Nullable RblData rblData,
            @Nonnull ImmutableAggregatorsConfig aggregatorsConfig,
            boolean hasCollapsedAggrs,
            boolean hasAggrs,
            boolean hasCounters) {

        this.counters = counters;
        this.rblData = rblData;

        aggregatorConsumer = aggregatorsConfig.materialize(new AggregatesScoringContext(request,
                Optional.ofNullable(rblData).map(RblData::getIsoCountry).orElse(null)));
        now = request.getTimestamp();
        this.hasCollapsedAggrs = hasCollapsedAggrs;
        this.hasAggrs = hasAggrs;
        this.hasCounters = hasCounters;
    }

    public void consume(@Nonnull StorageData data) {
        aggregatorConsumer.accept(new AggregatesScoringContext(data.getRequest(),
                rblData != null ? rblData.getIsoCountry() : null,
                data.getTransactionStatusAccordingToUpdateTimestamp(now),
                data.getResolutionCode(),
                data.isAuthed(),
                data.getQueues()));
    }

    public void consume(@Nonnull CollapsedAggregates data) throws StatsMergeException {
        aggregatorConsumer.accept(data);
    }

    private static final Set<FieldAcceptor> STATED_ACCEPTORS = Stream.of(
            FieldAcceptor.UID,
            FieldAcceptor.CARD_ID,
            FieldAcceptor.IP,
            FieldAcceptor.DEVICE_ID
    ).collect(Collectors.toSet());

    private static final Set<AggregatedTimeRange> STATED_RANGES = Stream.of(
            AggregatedTimeRange.HOUR,
            AggregatedTimeRange.WEEK
    ).collect(Collectors.toSet());

    public void stat(@Nonnull SolomonTuner solomonTuner) {
        for (AggregatorConsumer consumer : aggregatorConsumer.getAggregatorConsumers()) {
            final Aggregator.AggregatorInstance aggregatorInstance = consumer.getAggregatorInstance();
            if (aggregatorInstance == null ||
                    !STATED_ACCEPTORS.contains(consumer.getFieldAcceptorInstance().ideal()) ||
                    !STATED_RANGES.contains(consumer.getAggregatedTimeRange().ideal()) ||
                    aggregatorInstance.getStat().isEmpty()
            ) {
                continue;
            }
            solomonTuner.subRegistry(
                            consumer.getAggregatedTimeRange().ideal().getName(),
                            consumer.getFieldAcceptorInstance().ideal().getName())
                    .rate(aggregatorInstance.ideal().getName());
        }
    }

    @Nonnull
    public Aggregates build() throws JsonBadCastException {
        final int expectedSize = aggregatorConsumer.getAggregatorConsumers().size();
        final LuaTable full = new LuaTable(0, expectedSize);
        final JsonMap shortJson = new JsonMap(BasicContainerFactory.INSTANCE, expectedSize);
        final LuaTable structuredFull = new LuaTable(0, expectedSize);

        for (AggregatorConsumer consumer : aggregatorConsumer.getAggregatorConsumers()) {
            consumer.fillStats(full, shortJson, structuredFull);
        }

        shortJson.put("counters", JsonUtils.luaToJson(counters.toLua(), shortJson.containerFactory()));
        full.set("counters", counters.toLua());

        return new Aggregates(full, shortJson, structuredFull, hasAggrs && hasCollapsedAggrs, hasCounters);
    }
}
