package ru.yandex.antifraud.aggregates;

import java.util.Collection;
import java.util.Collections;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.antifraud.invariants.TransactionStatus;
import ru.yandex.antifraud.invariants.TransactionType;

public class AggregatesScoringContext {
    @Nonnull
    private final ScoringData data;
    @Nullable
    private final String ipIsoCountry;
    @Nullable
    private final TransactionStatus transactionStatus;

    @Nullable
    private final ResolutionCode resolutionCode;
    @Nullable
    private final Boolean authed;
    @Nonnull
    private final Collection<String> queues;
    private final boolean isForeignIp;
    private final boolean isForeignCard;
    private final boolean isYandexPlusWithdraw;
    private final boolean isYandexPlusTopup;
    private final boolean isBinding;
    private final boolean failed;
    private final boolean success;
    private final boolean simpleTransaction;
    private final boolean allow;
    private final boolean deny;

    public AggregatesScoringContext(@Nonnull ScoringData data,
                                    @Nullable String ipIsoCountry) {
        this(data, ipIsoCountry, null, null, null, Collections.emptyList());
    }

    public AggregatesScoringContext(@Nonnull ScoringData data,
                                    @Nullable String ipIsoCountry,
                                    @Nullable TransactionStatus transactionStatus,
                                    @Nullable ResolutionCode resolutionCode,
                                    @Nullable Boolean isAuthed,
                                    @Nonnull Collection<String> queues) {
        this.data = data;
        this.ipIsoCountry = ipIsoCountry;
        this.transactionStatus = transactionStatus;
        this.resolutionCode = resolutionCode;
        this.authed = isAuthed;
        this.queues = queues;

        isForeignIp = ipIsoCountry != null && !"RU".equalsIgnoreCase(ipIsoCountry);
        isForeignCard = data.getIsoCountry() != null && !"RU".equalsIgnoreCase(data.getIsoCountry());

        {
            @Nullable final String paymentMethod = data.getValue(Field.PAYMENT_METHOD).value();
            isYandexPlusWithdraw = "yandex_account_withdraw".equals(paymentMethod);
            isYandexPlusTopup = "yandex_account_topup".equals(paymentMethod);
        }

        isBinding = data.getTransactionType() == TransactionType.BINDING;

        failed = transactionStatus == TransactionStatus.FAILED;
        success = transactionStatus == TransactionStatus.OK;

        simpleTransaction = !isYandexPlusTopup && !isYandexPlusWithdraw;

        allow = ResolutionCode.ALLOW == resolutionCode;
        deny = ResolutionCode.DENY == resolutionCode;
    }

    @Nonnull
    public ScoringData getData() {
        return data;
    }

    @Nullable
    public String getIpIsoCountry() {
        return ipIsoCountry;
    }

    @Nullable
    public TransactionStatus getTransactionStatus() {
        return transactionStatus;
    }

    @Nullable
    public ResolutionCode getResolutionCode() {
        return resolutionCode;
    }

    @Nullable
    public Boolean isAuthed() {
        return authed;
    }

    @Nonnull
    public Collection<String> getQueues() {
        return queues;
    }

    public boolean isForeignIp() {
        return isForeignIp;
    }

    public boolean isForeignCard() {
        return isForeignCard;
    }

    public boolean isYandexPlusWithdraw() {
        return isYandexPlusWithdraw;
    }

    public boolean isYandexPlusTopup() {
        return isYandexPlusTopup;
    }

    public boolean isBinding() {
        return isBinding;
    }

    public boolean isFailed() {
        return failed;
    }

    public boolean isSimpleTransaction() {
        return simpleTransaction;
    }

    public boolean isSuccess() {
        return success;
    }

    public boolean isAllow() {
        return allow;
    }

    public boolean isDeny() {
        return deny;
    }
}
