package ru.yandex.antifraud.aggregates;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.data.Field;

public enum Aggregator {
    TNX_COUNT("tnx_cnt"),
    AMOUNT("amount_sum"),
    AMOUNT_LEGACY("amnt"),
    FAILED_COUNT("failed_tnx_cnt"),
    FAILED_COUNT_LEGACY("failed_cnt"),
    SUCCESS_AMOUNT("succ_amnts_sum"),
    SUCCESS_TNX_COUNT("succ_tnx_cnt"),
    BIND_TNX_COUNT("bind_tnx_cnt"),
    FAILED_BIND_TNX_COUNT("failed_bind_tnx_cnt"),
    PLUS_WITHDRAW_AMOUNT("plus_withdraw_amount"),
    PLUS_WITHDRAW_TNX_COUNT("plus_withdraw_tnx_count"),
    PLUS_TOPUP_AMOUNT("plus_topup_amount"),
    PLUS_TOPUP_TNX_COUNT("plus_topup_tnx_count"),
    CARDS("cards", Field.CARD_ID),
    SUCCESS_CARDS("succ_cards", Field.CARD_ID),
    BIND_CARDS("bind_cards", Field.CARD_ID),
    FAILED_BIND_CARDS("failed_bind_cards", Field.CARD_ID),
    FOREIGN_CARDS("foreign_cards", Field.CARD_ID),
    IPS("ips", Field.IP),
    FOREIGN_IPS("foreign_ips", Field.IP),
    FOREIGN_IPS_TNX_COUNT("foreign_ip_tnx_cnt", Field.IP),
    UIDS("uids", Field.UID),
    SUCCESS_UIDS("succ_uids", Field.UID),
    FOREIGN_UIDS("foreign_uids", Field.UID),
    FOREIGN_CARD_UIDS("foreign_card_uids", Field.UID),
    ORDERS("orders", Field.ORDER),
    BIN_COUNTRIES("bin_countries", Field.CARD_COUNTRY),
    DRIVER_LICENSES("taxi_driver_licenses", Field.TAXI_DRIVER_LICENSE),
    CURRENCIES("currencies", Field.CURRENCY),
    REGIONS("regions", Field.ORDER_REGION_NAME),
    TAXI_CAR_NUMBERS("taxi_car_numbers", Field.TAXI_CAR_NUMBER),
    QUEUES("queues"),
    COUNTRIES("countries"),
    FAILED_COUNTRIES("failed_countries"),
    SUCCESS_COUNTRIES("succ_countries"),
    SUCCESS_AUTHED_3DS_TNX_COUNT("succ_3ds_tnx_cnt"),
    AUTHED_3DS_TNX_COUNT("3ds_tnx_cnt"),
    SUCCESS_AUTHED_YIDS("succ_3ds_yids", Field.YID),
    DEVICE_IDS("device_ids", Field.DEVICE_ID),
    SUCCESS_DEVICE_IDS("succ_device_ids", Field.DEVICE_ID),
    FAILED_DEVICE_IDS("failed_device_ids", Field.DEVICE_ID),
    SUCCESS_AUTHED_DEVICE_IDS("succ_3ds_device_ids", Field.DEVICE_ID),
    SUCCESS_TRIAL_TNX_COUNT("succ_trial_tnx_cnt"),
    TARIFS("tarifs", Field.ORDER_TARIF),
    ;

    static final Map<String, Aggregator> NAME_INDEX;

    static {
        NAME_INDEX = new HashMap<>(Aggregator.values().length);
        for (Aggregator aggregator : Aggregator.values()) {
            NAME_INDEX.put(aggregator.getName(), aggregator);
        }
        assert NAME_INDEX.size() == Aggregator.values().length;
    }

    @Nonnull
    private final String name;
    @Nullable
    private final Field dependence;

    Aggregator(@Nonnull String name) {
        this(name, null);
    }

    Aggregator(@Nonnull String name, @Nullable Field dependence) {
        this.name = name;
        this.dependence = dependence;
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nullable
    public Field dependence() {
        return dependence;
    }

    @Nonnull
    public static Aggregator parse(@Nonnull String src) {
        src = src.trim();
        Aggregator aggregator = NAME_INDEX.get(src);
        if (aggregator != null) {
            return aggregator;
        }
        return Aggregator.valueOf(src);
    }

    public AggregatorInstance materialize(@Nonnull AggregatesScoringContext context) {
        switch (this) {
            case TNX_COUNT:
                return new TnxCount();
            case AMOUNT:
            case AMOUNT_LEGACY:
                return new Amount();
            case FAILED_COUNT:
            case FAILED_COUNT_LEGACY:
                return new FailedTnxCnt();
            case SUCCESS_AMOUNT:
                return new SuccessAmount();
            case SUCCESS_TNX_COUNT:
                return new SuccessTnxCnt();
            case BIND_TNX_COUNT:
                return new BindTnxCount();
            case FAILED_BIND_TNX_COUNT:
                return new BindFailedTnxCount();
            case PLUS_WITHDRAW_AMOUNT:
                return new PlusWithDrawAmounts();
            case PLUS_WITHDRAW_TNX_COUNT:
                return new PlusWithDrawTnxCount();
            case PLUS_TOPUP_AMOUNT:
                return new PlusTopupAmounts();
            case PLUS_TOPUP_TNX_COUNT:
                return new PlusTopupTnxCount();
            case CARDS:
                return new Cards(context);
            case SUCCESS_CARDS:
                return new SuccessCards(context);
            case BIND_CARDS:
                return new BindCards(context);
            case FAILED_BIND_CARDS:
                return new BindFailedCards(context);
            case FOREIGN_CARDS:
                return new ForeignCards(context);
            case IPS:
                return new Ips(context);
            case FOREIGN_IPS:
                return new ForeignIps(context);
            case FOREIGN_IPS_TNX_COUNT:
                return new ForeignIpsTnxCnt();
            case UIDS:
                return new Uids(context);
            case SUCCESS_UIDS:
                return new SuccessUids(context);
            case FOREIGN_UIDS:
                return new ForeignIpUids(context);
            case FOREIGN_CARD_UIDS:
                return new ForeignCardUids(context);
            case ORDERS:
                return new Orders(context);
            case BIN_COUNTRIES:
                return new BinCountries(context);
            case DRIVER_LICENSES:
                return new DriverLicenses(context);
            case CURRENCIES:
                return new Currencies(context);
            case REGIONS:
                return new Regions(context);
            case TAXI_CAR_NUMBERS:
                return new TaxiCarNumbers(context);
            case QUEUES:
                return new Queues(context);
            case COUNTRIES:
                return new Countries(context);
            case FAILED_COUNTRIES:
                return new FailedCountries(context);
            case SUCCESS_COUNTRIES:
                return new SuccessCountries(context);
            case SUCCESS_AUTHED_3DS_TNX_COUNT:
                return new SuccessAuthedTnxCnt();
            case AUTHED_3DS_TNX_COUNT:
                return new AuthedTnxCnt();
            case SUCCESS_AUTHED_YIDS:
                return new SuccessAuthedYids(context);
            case DEVICE_IDS:
                return new DeviceIds(context);
            case SUCCESS_DEVICE_IDS:
                return new SuccessDeviceIds(context);
            case FAILED_DEVICE_IDS:
                return new FailedDeviceIds(context);
            case SUCCESS_AUTHED_DEVICE_IDS:
                return new SuccessAuthedDeviceIds(context);
            case SUCCESS_TRIAL_TNX_COUNT:
                return new SuccessTrialTnxCnt();
            case TARIFS:
                return new Tarifs(context);
        }

        throw new RuntimeException("impossible");
    }

    public abstract class AggregatorInstance implements Predicate<AggregatesScoringContext>,
            Consumer<AggregatesScoringContext> {
        @Nonnull
        public abstract Stat getStat();

        @Nonnull
        public Aggregator ideal() {
            return Aggregator.this;
        }

        @Override
        public boolean equals(Object another) {
            if (another instanceof AggregatorInstance) {
                return Aggregator.this == ((AggregatorInstance) another).ideal();
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            return Aggregator.this.hashCode();
        }
    }

    abstract class LongAggregatorInstance extends AggregatorInstance {
        @Nonnull
        protected final LongStat stat = new LongStat();

        @Override
        @Nonnull
        public Stat getStat() {
            return stat;
        }

        public long value(@Nonnull AggregatesScoringContext context) {
            return 1;
        }

        @Override
        public void accept(@Nonnull AggregatesScoringContext context) {
            stat.add(value(context));
        }
    }

    abstract class SetAggregatorInstance extends AggregatorInstance {
        @Nonnull
        protected final SetStringStat stat;

        protected SetAggregatorInstance(@Nullable String reference) {
            this.stat = new SetStringStat(reference);
        }

        protected SetAggregatorInstance(@Nonnull Collection<String> values) {
            this.stat = new SetStringStat(values);
        }

        @Override
        @Nonnull
        public Stat getStat() {
            return stat;
        }

        public String value(@Nonnull AggregatesScoringContext context) {
            return null;
        }

        public Collection<String> values(@Nonnull AggregatesScoringContext context) {
            return Collections.emptyList();
        }

        @Override
        public void accept(@Nonnull AggregatesScoringContext context) {
            stat.add(value(context));
            stat.addAll(values(context));
        }
    }


    public class PlusWithDrawTnxCount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isYandexPlusWithdraw();
        }
    }

    public class PlusWithDrawAmounts extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isYandexPlusWithdraw();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getRubAmount();
        }
    }

    public class PlusTopupTnxCount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isYandexPlusTopup();
        }
    }

    public class PlusTopupAmounts extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isYandexPlusTopup();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getRubAmount();
        }
    }

    public class BindCards extends SetAggregatorInstance {
        protected BindCards(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getCardId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isBinding();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCardId();
        }
    }

    public class BindTnxCount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isBinding();
        }
    }

    public class BindFailedCards extends SetAggregatorInstance {
        protected BindFailedCards(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getCardId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isBinding() && context.isFailed();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCardId();
        }
    }

    public class BindFailedTnxCount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isBinding() && context.isFailed();
        }
    }

    public class TnxCount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }
    }

    public class Amount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getRubAmount();
        }
    }

    public class Cards extends SetAggregatorInstance {
        protected Cards(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getCardId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCardId();
        }
    }

    public class Orders extends SetAggregatorInstance {
        protected Orders(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getOrderId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getOrderId();
        }
    }

    public class ForeignCards extends SetAggregatorInstance {
        protected ForeignCards(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getCardId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isForeignCard();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCardId();
        }
    }

    public class ForeignCardUids extends SetAggregatorInstance {
        protected ForeignCardUids(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getUid());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isForeignCard();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getUid();
        }
    }

    public class ForeignIps extends SetAggregatorInstance {
        protected ForeignIps(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getIpOrHbfId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isForeignIp();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getIpOrHbfId();
        }
    }

    public class ForeignIpsTnxCnt extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isForeignIp();
        }
    }

    public class ForeignIpUids extends SetAggregatorInstance {
        protected ForeignIpUids(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getUid());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isForeignIp();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getUid();
        }
    }

    public class AuthedTnxCnt extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && Boolean.TRUE.equals(context.isAuthed());
        }
    }

    public class Queues extends SetAggregatorInstance {
        protected Queues(@Nonnull AggregatesScoringContext context) {
            super(context.getQueues());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public Collection<String> values(@Nonnull AggregatesScoringContext context) {
            return context.getQueues();
        }
    }

    public class SuccessAmount extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getRubAmount();
        }
    }

    public class SuccessTnxCnt extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }
    }

    public class SuccessCards extends SetAggregatorInstance {
        protected SuccessCards(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getCardId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCardId();
        }
    }

    public class SuccessUids extends SetAggregatorInstance {
        protected SuccessUids(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getUid());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getUid();
        }
    }

    public class SuccessAuthedTnxCnt extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess() && Boolean.TRUE.equals(context.isAuthed());
        }
    }

    public class SuccessCountries extends SetAggregatorInstance {
        protected SuccessCountries(@Nonnull AggregatesScoringContext context) {
            super(context.getIpIsoCountry());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getIpIsoCountry();
        }
    }

    public class SuccessDeviceIds extends SetAggregatorInstance {
        protected SuccessDeviceIds(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getDeviceId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getDeviceId();
        }
    }

    public class SuccessTrialTnxCnt extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            final String paymentMethod = context.getData().getPaymentMethod();
            final String servicePaymentMethod = context.getData().getValue(Field.SERVICE_PAYMENT_TYPE).value();
            return context.isSimpleTransaction() &&
                    context.isSuccess() &&
                    ("trial_payment".equals(paymentMethod) || "trial".equals(servicePaymentMethod));
        }
    }

    public class SuccessAuthedYids extends SetAggregatorInstance {
        protected SuccessAuthedYids(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getYid());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() &&
                    context.isSuccess() &&
                    Boolean.TRUE.equals(context.isAuthed());
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getYid();
        }
    }

    public class SuccessAuthedDeviceIds extends SetAggregatorInstance {
        protected SuccessAuthedDeviceIds(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getDeviceId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess() && Boolean.TRUE.equals(context.isAuthed());
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getDeviceId();
        }
    }

    public class FailedTnxCnt extends LongAggregatorInstance {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isFailed();
        }
    }

    public class Countries extends SetAggregatorInstance {
        protected Countries(@Nonnull AggregatesScoringContext context) {
            super(context.getIpIsoCountry());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getIpIsoCountry();
        }
    }

    public class FailedCountries extends SetAggregatorInstance {
        protected FailedCountries(@Nonnull AggregatesScoringContext context) {
            super(context.getIpIsoCountry());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isFailed();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getIpIsoCountry();
        }
    }

    public class FailedDeviceIds extends SetAggregatorInstance {
        protected FailedDeviceIds(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getDeviceId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isFailed();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getDeviceId();
        }
    }

    public class BinCountries extends SetAggregatorInstance {
        protected BinCountries(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getIsoCountry());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getIsoCountry();
        }
    }

    public class DriverLicenses extends SetAggregatorInstance {
        protected DriverLicenses(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getTaxiDriverLicense());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getTaxiDriverLicense();
        }
    }

    public class Currencies extends SetAggregatorInstance {
        protected Currencies(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getCurrency());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCurrency();
        }
    }

    public class Regions extends SetAggregatorInstance {
        protected Regions(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getRegionName());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getRegionName();
        }
    }

    public class TaxiCarNumbers extends SetAggregatorInstance {
        protected TaxiCarNumbers(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getTaxiCarNumber());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getTaxiCarNumber();
        }
    }

    public class Uids extends SetAggregatorInstance {
        protected Uids(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getUid());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getUid();
        }
    }

    public class DeviceIds extends SetAggregatorInstance {
        protected DeviceIds(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getDeviceId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getDeviceId();
        }
    }

    public class Ips extends SetAggregatorInstance {
        protected Ips(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getIpOrHbfId());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getIpOrHbfId();
        }
    }

    public class Tarifs extends SetAggregatorInstance {
        protected Tarifs(@Nonnull AggregatesScoringContext context) {
            super(context.getData().getOrderTarif());
        }

        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getOrderTarif();
        }
    }
}
