package ru.yandex.antifraud.aggregates;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaString;

import ru.yandex.antifraud.data.CollapsedAggregates;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.data.Value;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public enum FieldAcceptor implements JsonValue, Function<ScoringData, FieldAcceptor.FieldAcceptorInstance> {
    UID(Field.UID.singleMulti()),
    CARD_ID(Field.CARD_ID.singleMulti()),
    UID_CARD_ID(new Field.Multi(Field.UID, Field.CARD_ID)),
    MAIL(Field.MAIL.singleMulti()),
    IP(Field.IP.singleMulti()),
    UID_PAYED_BY(new Field.Multi(Field.UID, Field.PAYED_BY)),
    UID_APPLE_PAY(new Field.Multi(Field.UID, Field.APPLE_PAY_PAYED_BY)),
    UID_GOOGLE_PAY(new Field.Multi(Field.UID, Field.GOOGLE_PAY_PAYED_BY)),
    DEVICE_ID(Field.DEVICE_ID.singleMulti()),
    USER_PHONE(Field.USER_PHONE.singleMulti()),
    ;

    static final Map<String, FieldAcceptor> NAME_INDEX;

    static {
        NAME_INDEX = new HashMap<>(FieldAcceptor.values().length);
        for (FieldAcceptor value : FieldAcceptor.values()) {
            NAME_INDEX.put(value.getName(), value);
        }
        assert NAME_INDEX.size() == FieldAcceptor.values().length;
    }

    @Nonnull
    private final Field.Multi fieldMulti;
    @Nonnull
    private final LuaString luaName;

    FieldAcceptor(@Nonnull Field.Multi fieldMulti) {
        this.fieldMulti = fieldMulti;
        luaName = LuaString.valueOf(fieldMulti.toString());
    }

    @Override
    public FieldAcceptorInstance apply(ScoringData data) {
        return new FieldAcceptorInstance(data.getMultiValue(fieldMulti));
    }

    @Nonnull
    public String getName() {
        return fieldMulti.toString();
    }

    @Nonnull
    public LuaString getLuaName() {
        return luaName;
    }

    @Nonnull
    public Field.Multi getDeps() {
        return fieldMulti;
    }

    public boolean isEmpty(@Nonnull ScoringData scoringData) {
        for (Field field : fieldMulti.fields()) {
            if (scoringData.getValue(field).isEmpty()) {
                return true;
            }
        }
        return false;
    }

    @Nonnull
    public FieldAcceptorInstance materialize(@Nonnull ScoringData data) {
        return apply(data);
    }

    @Nonnull
    public static FieldAcceptor parse(@Nonnull String src) {
        src = src.trim();
        FieldAcceptor fieldAcceptor = NAME_INDEX.get(src);
        if (fieldAcceptor != null) {
            return fieldAcceptor;
        }
        return FieldAcceptor.valueOf(src);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.value(fieldMulti.toString());
    }

    public class FieldAcceptorInstance {
        @Nonnull
        private final Value.Multi multiValue;
        @Nonnull
        private final String keyValue;

        private FieldAcceptorInstance(@Nonnull Value.Multi multiValue) {
            this.multiValue = multiValue;
            keyValue = fieldMulti.toString() + '_' + multiValue.value();
        }

        public boolean test(AggregatesScoringContext context) {
            final Value.Multi requestValues = context.getData().getMultiValue(multiValue.fields());
            return !requestValues.isEmpty() && multiValue.equals(requestValues);
        }

        public boolean test(CollapsedAggregates collapsedAggregates) {
            return collapsedAggregates.fieldPrefix().equals(fieldMulti.toString());
        }

        public boolean isEmpty() {
            return multiValue.isEmpty();
        }

        @Nonnull
        public String key() {
            return multiValue.fields().toString();
        }

        @Nonnull
        public String keyValue() {
            return keyValue;
        }

        @Nonnull
        public FieldAcceptor ideal() {
            return FieldAcceptor.this;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof FieldAcceptorInstance)) {
                return false;
            }
            FieldAcceptorInstance that = (FieldAcceptorInstance) o;
            return FieldAcceptor.this == that.ideal() &&
                    multiValue.equals(that.multiValue);
        }

        @Override
        public int hashCode() {
            return Objects.hash(ideal(), multiValue);
        }

        @Override
        public String toString() {
            return ideal() + "(" + keyValue + ")";
        }
    }
}
