package ru.yandex.antifraud.aggregates;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;

import ru.yandex.antifraud.util.BiConsumerSafe;
import ru.yandex.antifraud.util.ExplicitJsonValue;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;

public interface MultiStructuredStat extends StructuredStat, ExplicitJsonValue {
    @Nonnull
    Map<String, ? extends StructuredStat> stats();

    void put(@Nonnull String key, @Nonnull StructuredStat value) throws StatsMergeException;

    @Override
    @Nonnull
    default JsonObject serialize(@Nonnull ContainerFactory containerFactory) {
        final var stats = stats();
        final JsonMap map = new JsonMap(containerFactory, stats.size());

        visit((k, v) -> {
            map.put(k, v.serialize(containerFactory));
        });

        return map;
    }

    @Override
    @Nonnull
    default LuaValue serialize() {
        final LuaTable map = new LuaTable();

        visit((k, v) -> {
            map.set(k, v.serialize());
        });

        return map;
    }

    @Override
    default void jsonify(@Nonnull final JsonWriterBase writer)
        throws IOException
    {
        writer.startObject();
        visit((k, v) -> {
            writer.key(k);
            v.jsonify(writer);
        });
        writer.endObject();
    }

    private <E extends Throwable> void visit(@Nonnull BiConsumerSafe<String, StructuredStat, E> consumer) throws E {
        for (var entry : stats().entrySet()) {
            final StructuredStat stat = entry.getValue();
            if (!stat.isEmpty()) {
                consumer.accept(entry.getKey(), entry.getValue());
            }
        }
    }

    @Override
    default boolean isEmpty() {
        final Map<String, ? extends StructuredStat> stats = stats();
        return stats.isEmpty() || stats.values().stream().allMatch(StructuredStat::isEmpty);
    }

    @Override
    default void structuredMerge(@Nonnull StructuredStat another) throws StatsMergeException {
        if (another instanceof MultiStructuredStat) {
            final Map<String, ? extends StructuredStat> rightStats = ((MultiStructuredStat) another).stats();
            final Map<String, ? extends StructuredStat> leftStats = stats();

            for (var entry : rightStats.entrySet()) {
                final StructuredStat left = leftStats.get(entry.getKey());
                if (left == null) {
                    put(entry.getKey(), entry.getValue().copy());
                } else {
                    left.structuredMerge(entry.getValue());
                }
            }
        } else {
            throw new StatsMergeException(another.getClass().getName() + " is not MultiStructuredStat");
        }
    }

    @Override
    default StructuredStat copy() {
        final Map<String, ? extends StructuredStat> stats = stats();
        final Map<String, StructuredStat> copy = new HashMap<>(stats.size());
        for (var entry : stats.entrySet()) {
            copy.put(entry.getKey(), entry.getValue().copy());
        }
        return new Instance(copy);
    }

    class Instance implements MultiStructuredStat {
        @Nonnull
        private final Map<String, StructuredStat> stats;

        public Instance() {
            stats = new HashMap<>();
        }

        public Instance(@Nonnull Map<String, StructuredStat> stats) {
            this.stats = stats;
        }

        @Nonnull
        @Override
        public Map<String, ? extends StructuredStat> stats() {
            return stats;
        }

        @Override
        public void put(@Nonnull String key, @Nonnull StructuredStat value) {
            stats.put(key, value);
        }
    }

    @Nonnull
    static StructuredStat parse(@Nonnull JsonObject src) throws JsonException {
        switch (src.type()) {
            case NULL:
            case BOOLEAN:
            case STRING:
            case LIST:
                throw new JsonException("unsupported type");
            case LONG:
                return new LongStat(src.asLong());
            case DOUBLE:
                return new LongStat((long) src.asDouble());
            case MAP: {
                final JsonMap map = src.asMap();
                final Map<String, StructuredStat> stats = new HashMap<>(map.size());

                for (var entry : map.entrySet()) {
                    stats.put(entry.getKey(), parse(entry.getValue()));
                }

                return new Instance(stats);
            }
        }
        throw new JsonException("never thrown");
    }
}
