package ru.yandex.antifraud.aggregates;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaString;

import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.antifraud.invariants.TransactionStatus;

public enum StructuredAggregator {
    AMOUNTS_BY_CURRENCY("amount"),
    SUCCESS_AMOUNTS_BY_CURRENCY(TransactionStatus.OK + "_amount"),
    FAILED_AMOUNTS_BY_CURRENCY(TransactionStatus.FAILED + "_amount"),
    ALLOW_AMOUNTS_BY_CURRENCY(ResolutionCode.ALLOW + "_amount"),
    DENY_AMOUNTS_BY_CURRENCY(ResolutionCode.DENY + "_amount"),
    ;

    static final Map<String, StructuredAggregator> NAME_INDEX;

    static {
        NAME_INDEX = new HashMap<>(StructuredAggregator.values().length);
        for (StructuredAggregator aggregator : StructuredAggregator.values()) {
            NAME_INDEX.put(aggregator.getName(), aggregator);
        }
        assert NAME_INDEX.size() == StructuredAggregator.values().length;
    }

    @Nonnull
    private final String name;

    @Nonnull
    private final LuaString luaName;

    StructuredAggregator(@Nonnull String name) {
        this.name = name;
        this.luaName = LuaString.valueOf(name);
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nonnull
    public LuaString getLuaName() {
        return luaName;
    }

    public StructuredAggregatorInstance materialize() {
        switch (this) {
            case AMOUNTS_BY_CURRENCY:
                return new AmountsByCurrency();
            case SUCCESS_AMOUNTS_BY_CURRENCY:
                return new SuccessAmountsByCurrency();
            case FAILED_AMOUNTS_BY_CURRENCY:
                return new FailedAmountsByCurrency();
            case ALLOW_AMOUNTS_BY_CURRENCY:
                return new AllowedAmountsByCurrency();
            case DENY_AMOUNTS_BY_CURRENCY:
                return new DeniedAmountsByCurrency();
        }

        throw new RuntimeException("impossible");
    }


    @Nonnull
    public static StructuredAggregator parse(@Nonnull String src) {
        src = src.trim();
        StructuredAggregator aggregator = NAME_INDEX.get(src);
        if (aggregator != null) {
            return aggregator;
        }
        return StructuredAggregator.valueOf(src);
    }


    public abstract class StructuredAggregatorInstance implements Predicate<AggregatesScoringContext>,
            Consumer<AggregatesScoringContext> {
        @Nonnull
        public StructuredAggregator ideal() {
            return StructuredAggregator.this;
        }

        @Nonnull
        public abstract StructuredStat getStat();
    }

    public abstract class StructuredLongStat extends StructuredAggregatorInstance {
        @Nonnull
        protected final MultiLongStat stat = new MultiLongStat();

        @Override
        @Nonnull
        public StructuredStat getStat() {
            return stat;
        }

        public String key(@Nonnull AggregatesScoringContext context) {
            return null;
        }

        public long value(@Nonnull AggregatesScoringContext context) {
            return 0;
        }

        @Override
        public void accept(@Nonnull AggregatesScoringContext context) {
            stat.add(key(context), value(context));
        }
    }

    public class SuccessAmountsByCurrency extends StructuredLongStat {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isSuccess();
        }

        @Override
        public String key(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCurrency();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getAmount();
        }
    }

    public class FailedAmountsByCurrency extends StructuredLongStat {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isFailed();
        }

        @Override
        public String key(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCurrency();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getAmount();
        }
    }

    public class AllowedAmountsByCurrency extends StructuredLongStat {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isAllow();
        }

        @Override
        public String key(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCurrency();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getAmount();
        }
    }

    public class DeniedAmountsByCurrency extends StructuredLongStat {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction() && context.isDeny();
        }

        @Override
        public String key(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCurrency();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getAmount();
        }
    }

    public class AmountsByCurrency extends StructuredLongStat {
        @Override
        public boolean test(@Nonnull AggregatesScoringContext context) {
            return context.isSimpleTransaction();
        }

        @Override
        public String key(@Nonnull AggregatesScoringContext context) {
            return context.getData().getCurrency();
        }

        @Override
        public long value(@Nonnull AggregatesScoringContext context) {
            return context.getData().getAmount();
        }
    }
}
