package ru.yandex.antifraud.aggregates_v2;

import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaString;

import ru.yandex.antifraud.util.Range;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public enum AggregatedTimeRange implements JsonValue {
    ANY("any"),
    HOUR("h", Duration.ofHours(-1)),
    DAY("d", Duration.ofDays(-1)),
    WEEK("w", Duration.ofDays(-7)),
    MONTH("m", Duration.ofDays(-28)),
    HALF_YEAR("m6", Duration.ofDays(-187), MONTH.since),
    ;

    @Nonnull
    private final String name;

    @Nonnull
    private final LuaString luaName;
    @Nullable
    private final Duration since, until;

    @Nonnull
    public String getName() {
        return name;
    }

    @Nonnull
    public LuaString getLuaName() {
        return luaName;
    }

    @Nonnull
    public AggregatedTimeRangeInstance materialize(@Nonnull Instant now) {
        return new AggregatedTimeRangeInstance(now);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.value(name);
    }

    public static Range<Instant> rangesToDays(@Nonnull Set<AggregatedTimeRange> ranges,
                                             @Nonnull Instant now,
                                             @Nonnull Instant min,
                                             @Nonnull Instant max) {
        final Range<Instant> union = new Range<>(Instant::compareTo);
        for (AggregatedTimeRange timeRange : ranges) {
            final AggregatedTimeRangeInstance instance = timeRange.materialize(now);
            final Instant b = max(min, instance.since);
            final Instant e = min(max, instance.until);
            if(b.isBefore(e)) {
                union.unionWith(b, e);
            }
        }
        return union;
    }

    private static Instant max(Instant lhs, Instant rhs) {
        return lhs.isBefore(rhs) ? rhs : lhs;
    }

    private static Instant min(Instant lhs, Instant rhs) {
        return lhs.isBefore(rhs) ? lhs : rhs;
    }

    public class AggregatedTimeRangeInstance implements Predicate<Instant> {
        @Nonnull
        private final Instant since, until;

        AggregatedTimeRangeInstance(@Nonnull Instant now) {
            this.since = AggregatedTimeRange.this.since != null ? now.plus(AggregatedTimeRange.this.since) :
                    Instant.MIN;
            this.until = AggregatedTimeRange.this.until != null ? now.plus(AggregatedTimeRange.this.until) :
                    Instant.MAX;
        }

        @Override
        public boolean test(Instant instant) {
            return instant.isAfter(since) && instant.isBefore(until);
        }

        public AggregatedTimeRange ideal() {
            return AggregatedTimeRange.this;
        }
    }

    AggregatedTimeRange(@Nonnull String name) {
        this(name, null, null);
    }

    AggregatedTimeRange(@Nonnull String name, @Nonnull Duration since) {
        this(name, since, null);
    }

    AggregatedTimeRange(@Nonnull String name, @Nullable Duration since, @Nullable Duration until) {
        this.name = name;
        this.since = since;
        this.until = until;
        this.luaName = LuaString.valueOf(name);
    }

    @Nonnull
    public static AggregatedTimeRange parse(@Nonnull String src) {
        src = src.trim();

        AggregatedTimeRange range = NAME_INDEX.get(src);
        if (range != null) {
            return range;
        }
        return AggregatedTimeRange.valueOf(src);
    }

    private static final Map<String, AggregatedTimeRange> NAME_INDEX;

    static {
        NAME_INDEX = new HashMap<>(AggregatedTimeRange.values().length);
        for (AggregatedTimeRange range : AggregatedTimeRange.values()) {
            NAME_INDEX.put(range.name, range);
        }
        assert NAME_INDEX.size() == AggregatedTimeRange.values().length;
    }
}
