package ru.yandex.antifraud.aggregates_v2;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaString;
import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;

import ru.yandex.antifraud.aggregates.AggregatesScoringContext;
import ru.yandex.antifraud.aggregates.Aggregator;
import ru.yandex.antifraud.aggregates.FieldAcceptor;
import ru.yandex.antifraud.aggregates.Stat;
import ru.yandex.antifraud.aggregates.StatsMergeException;
import ru.yandex.antifraud.aggregates.StructuredAggregator;
import ru.yandex.antifraud.aggregates.StructuredStat;
import ru.yandex.antifraud.data.CollapsedAggregates;
import ru.yandex.antifraud.util.AggregatesNamesCache;
import ru.yandex.json.dom.JsonMap;

public class AggregatorConsumer {
    @Nonnull
    private final AggregatedTimeRange.AggregatedTimeRangeInstance aggregatedTimeRange;
    @Nullable
    private final Aggregator.AggregatorInstance aggregatorInstance;
    @Nullable
    private final StructuredAggregator.StructuredAggregatorInstance structuredAggregatorInstance;
    @Nonnull
    private final FieldAcceptor.FieldAcceptorInstance fieldAcceptorInstance;

    private final boolean fullHists;

    public AggregatorConsumer(@Nonnull AggregatedTimeRange.AggregatedTimeRangeInstance aggregatedTimeRange,
                              @Nullable Aggregator.AggregatorInstance aggregatorInstance,
                              @Nullable StructuredAggregator.StructuredAggregatorInstance structuredAggregatorInstance,
                              @Nonnull FieldAcceptor.FieldAcceptorInstance fieldAcceptorInstance,
                              boolean fullHists) {
        this.aggregatedTimeRange = aggregatedTimeRange;
        this.aggregatorInstance = aggregatorInstance;
        this.structuredAggregatorInstance = structuredAggregatorInstance;
        this.fieldAcceptorInstance = fieldAcceptorInstance;
        this.fullHists = fullHists;
    }

    public boolean test(AggregatesScoringContext aggregatesScoringContext) {
        return aggregatorInstance != null && aggregatorInstance.test(aggregatesScoringContext) ||
                structuredAggregatorInstance != null && structuredAggregatorInstance.test(aggregatesScoringContext);
    }

    public boolean test(CollapsedAggregates collapsedAggregates) {
        return fieldAcceptorInstance.test(collapsedAggregates);
    }

    public void accept(@Nonnull AggregatesScoringContext context) {
        if (aggregatorInstance != null) {
            aggregatorInstance.accept(context);
        }
        if (structuredAggregatorInstance != null) {
            structuredAggregatorInstance.accept(context);
        }
    }

    public void accept(@Nonnull CollapsedAggregates collapsedAggregates) throws StatsMergeException {
        if (aggregatorInstance != null) {
            final Stat stat = collapsedAggregates.stats().getStat(aggregatorInstance.ideal());
            if (stat != null) {
                aggregatorInstance.getStat().mergeWith(stat);
            }
        }
        if (structuredAggregatorInstance != null) {
            final StructuredStat stat =
                    collapsedAggregates.structuredStats().get(structuredAggregatorInstance.ideal());
            if (stat != null) {
                structuredAggregatorInstance.getStat().structuredMerge(stat);
            }
        }
    }

    public void fillStats(@Nonnull LuaTable full, @Nonnull JsonMap shortJson, @Nonnull LuaTable structuredFull) {
        if (aggregatorInstance != null) {
            final Stat stat = aggregatorInstance.getStat();

            if (!stat.isEmpty()) {
                final AggregatesNamesCache.Names names = AggregatesNamesCache.INSTANCE.names(
                        fieldAcceptorInstance.ideal(),
                        aggregatorInstance.ideal(),
                        aggregatedTimeRange.ideal());
                if (fullHists) {
                    stat.fully(full, names);
                } else {
                    stat.brief(full, names);
                }
                stat.brief(shortJson, names);
            }
        }
        if (structuredAggregatorInstance != null) {
            final StructuredStat stat = structuredAggregatorInstance.getStat();
            if (!stat.isEmpty()) {
                LuaValue dst = structuredFull;

                dst = goDeeper(dst, fieldAcceptorInstance.ideal().getLuaName());
                dst = goDeeper(dst, aggregatedTimeRange.ideal().getLuaName());

                dst.set(structuredAggregatorInstance.ideal().getLuaName(), stat.serialize());
            }
        }
    }

    @Nonnull
    private static LuaValue goDeeper(LuaValue src, LuaString key) {
        LuaValue next = src.get(key);
        if (!next.istable()) {
            next = new LuaTable();
            src.set(key, next);
        }
        return next;
    }

    @Nonnull
    public AggregatedTimeRange.AggregatedTimeRangeInstance getAggregatedTimeRange() {
        return aggregatedTimeRange;
    }

    @Nullable
    public Aggregator.AggregatorInstance getAggregatorInstance() {
        return aggregatorInstance;
    }

    @Nullable
    public StructuredAggregator.StructuredAggregatorInstance getStructuredAggregatorInstance() {
        return structuredAggregatorInstance;
    }

    @Nonnull
    public FieldAcceptor.FieldAcceptorInstance getFieldAcceptorInstance() {
        return fieldAcceptorInstance;
    }
}
