package ru.yandex.antifraud.aggregates_v2.aggregate_config;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.aggregates.Aggregator;
import ru.yandex.antifraud.aggregates.FieldAcceptor;
import ru.yandex.antifraud.aggregates.StructuredAggregator;
import ru.yandex.antifraud.aggregates_v2.AggregatedTimeRange;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.config.ParameterConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.CollectionValidator;
import ru.yandex.parser.string.NonEmptyCollectionValidator;

public abstract class AbstractAggregatorConfigBuilder
        <T extends AbstractAggregatorConfigBuilder<T>>
        implements AggregatorConfig {

    private static final ParameterConfig<Set<AggregatedTimeRange>> TIME_RANGES = new ParameterConfig<>(
            "aggregated-time-ranges",
            new CollectionParser<>(AggregatedTimeRange::parse, LinkedHashSet::new),
            new CollectionValidator<AggregatedTimeRange, Set<AggregatedTimeRange>>(r -> r)
                    .andThen(NonEmptyCollectionValidator.instance())
                    .andThen(LinkedHashSet::new)
    );

    private static final ParameterConfig<Set<Aggregator>> AGGREGATORS = new ParameterConfig<>(
            "aggregators",
            new CollectionParser<>(Aggregator::parse, LinkedHashSet::new),
            new CollectionValidator<Aggregator, Set<Aggregator>>(r -> r)
                    .andThen(NonEmptyCollectionValidator.instance())
                    .andThen(LinkedHashSet::new)
    );

    private static final ParameterConfig<Set<StructuredAggregator>> STRUCTURED_AGGREGATORS = new ParameterConfig<>(
            "structured-aggregators",
            new CollectionParser<>(StructuredAggregator::parse, LinkedHashSet::new),
            new CollectionValidator<StructuredAggregator, Set<StructuredAggregator>>(r -> r)
                    .andThen(NonEmptyCollectionValidator.instance())
                    .andThen(LinkedHashSet::new)
    );

    private static final ParameterConfig<Set<FieldAcceptor>> FIELD_ACCEPTORS = new ParameterConfig<>(
            "field-acceptors",
            new CollectionParser<>(FieldAcceptor::parse, LinkedHashSet::new),
            new CollectionValidator<FieldAcceptor, Set<FieldAcceptor>>(r -> r)
                    .andThen(NonEmptyCollectionValidator.instance())
                    .andThen(LinkedHashSet::new)
    );

    private static final String FULL_HISTS = "full-hists";

    private Set<AggregatedTimeRange> timeRanges;

    private Set<Aggregator> aggregators;

    private Set<StructuredAggregator> structuredAggregators;

    private Set<FieldAcceptor> fieldAcceptors;

    private boolean fullHists;

    protected AbstractAggregatorConfigBuilder(final AggregatorConfig config) {
        timeRanges = config.timeRanges();
        aggregators = config.aggregators();
        structuredAggregators = config.structuredAggregators();
        fieldAcceptors = config.fieldAcceptors();
        fullHists = config.fullHists();
    }

    protected AbstractAggregatorConfigBuilder(
            @Nonnull IniConfig config,
            @Nonnull AggregatorConfig defaults)
            throws ConfigException {
        timeRanges = new LinkedHashSet<>(TIME_RANGES.extract(config, defaults.timeRanges()));
        aggregators = new LinkedHashSet<>(AGGREGATORS.extract(config, defaults.aggregators()));
        structuredAggregators = new LinkedHashSet<>(STRUCTURED_AGGREGATORS.extract(config,
                defaults.structuredAggregators()));
        fieldAcceptors = new LinkedHashSet<>(FIELD_ACCEPTORS.extract(config, defaults.fieldAcceptors()));
        fullHists = config.getBoolean(FULL_HISTS, defaults.fullHists());
    }


    @Override
    public Set<AggregatedTimeRange> timeRanges() {
        return timeRanges;
    }

    public void timeRanges(@Nonnull Collection<AggregatedTimeRange> timeRanges) {
        this.timeRanges = new LinkedHashSet<>(timeRanges);
    }

    @Override
    public Set<Aggregator> aggregators() {
        return aggregators;
    }

    public void aggregators(@Nonnull Collection<Aggregator> aggregators) {
        this.aggregators = new LinkedHashSet<>(aggregators);
    }

    @Override
    public Set<StructuredAggregator> structuredAggregators() {
        return structuredAggregators;
    }

    public void structuredAggregators(@Nonnull Set<StructuredAggregator> structuredAggregators) {
        this.structuredAggregators = structuredAggregators;
    }

    @Override
    public Set<FieldAcceptor> fieldAcceptors() {
        return fieldAcceptors;
    }

    public void fieldAcceptors(@Nonnull Set<FieldAcceptor> fieldAcceptors) {
        this.fieldAcceptors = fieldAcceptors;
    }

    @Override
    public boolean fullHists() {
        return fullHists;
    }

    public void fullHists(boolean fullHists) {
        this.fullHists = fullHists;
    }
}

