package ru.yandex.antifraud.aggregates_v2.aggregate_config;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.aggregates.AggregatesScoringContext;
import ru.yandex.antifraud.aggregates.Aggregator;
import ru.yandex.antifraud.aggregates.FieldAcceptor;
import ru.yandex.antifraud.aggregates.StructuredAggregator;
import ru.yandex.antifraud.aggregates_v2.AggregatedTimeRange;
import ru.yandex.antifraud.aggregates_v2.AggregatorConsumer;

public class ImmutableAggregatorConfig implements AggregatorConfig {
    private final Set<AggregatedTimeRange> timeRanges;

    private final Set<Aggregator> aggregators;

    private final Set<StructuredAggregator> structuredAggregators;

    private final Set<FieldAcceptor> fieldAcceptors;

    private final Map<FieldAcceptor, List<Aggregator>> availableAggregatorsByFields;

    private final boolean fullHists;

    public ImmutableAggregatorConfig(final AggregatorConfig config) {
        timeRanges = config.timeRanges();
        aggregators = config.aggregators();
        structuredAggregators = config.structuredAggregators();
        fieldAcceptors = config.fieldAcceptors();
        fullHists = config.fullHists();

        availableAggregatorsByFields = new EnumMap<>(FieldAcceptor.class);

        for (FieldAcceptor acceptor : FieldAcceptor.values()) {
            final List<Aggregator> availableAggregators = new ArrayList<>(Aggregator.values().length);

            for (Aggregator aggregator : aggregators) {
                if (!acceptor.getDeps().contains(aggregator.dependence())) {
                    availableAggregators.add(aggregator);
                }
            }

            availableAggregatorsByFields.put(acceptor, availableAggregators);
        }
    }

    @Override
    public Set<AggregatedTimeRange> timeRanges() {
        return timeRanges;
    }


    @Override
    public Set<Aggregator> aggregators() {
        return aggregators;
    }

    @Override
    public Set<StructuredAggregator> structuredAggregators() {
        return structuredAggregators;
    }

    @Override
    public Set<FieldAcceptor> fieldAcceptors() {
        return fieldAcceptors;
    }

    @Override
    public boolean fullHists() {
        return fullHists;
    }

    public void materialize(@Nonnull AggregatesScoringContext context,
                            Consumer<AggregatorConsumer> aggregatorConsumerConsumer) {
        final List<AggregatedTimeRange.AggregatedTimeRangeInstance> timeRangeAggregatedTimeRangeInstances = timeRanges
                .stream()
                .map(range -> range.materialize(context.getData().getTimestamp()))
                .collect(Collectors.toList());

        for (var entry : availableAggregatorsByFields.entrySet()) {
            final FieldAcceptor fieldAcceptor = entry.getKey();
            if (fieldAcceptor.isEmpty(context.getData())) {
                continue;
            }

            final FieldAcceptor.FieldAcceptorInstance fieldAcceptorInstance =
                    fieldAcceptor.materialize(context.getData());

            for (Aggregator aggregator : entry.getValue()) {
                for (AggregatedTimeRange.AggregatedTimeRangeInstance timeRangeAggregatedTimeRangeInstance :
                        timeRangeAggregatedTimeRangeInstances) {
                    aggregatorConsumerConsumer.accept(new AggregatorConsumer(timeRangeAggregatedTimeRangeInstance,
                            aggregator.materialize(context),
                            null,
                            fieldAcceptorInstance,
                            fullHists));
                }
            }

            for (StructuredAggregator aggregator : structuredAggregators) {
                for (AggregatedTimeRange.AggregatedTimeRangeInstance timeRangeAggregatedTimeRangeInstance :
                        timeRangeAggregatedTimeRangeInstances) {
                    aggregatorConsumerConsumer.accept(new AggregatorConsumer(timeRangeAggregatedTimeRangeInstance,
                            null,
                            aggregator.materialize(),
                            fieldAcceptorInstance,
                            fullHists));
                }
            }
        }
    }
}

