package ru.yandex.antifraud.artefacts;

import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaFunction;
import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.antifraud.lua_context_manager.LuaBinding;
import ru.yandex.antifraud.util.Request;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.writer.JsonType;
import ru.yandex.lua.util.JsonUtils;
import ru.yandex.parser.uri.QueryConstructor;

public class LuaClient {
    @Nonnull
    private List<LuaRequest> requests = new ArrayList<>();

    @LuaBinding
    public void post(@Nonnull String clientName,
                     @Nonnull QueryConstructor queryConstructor,
                     @Nonnull Map.Entry<String, ContentType> data,
                     @Nullable LuaTable headers,
                     @Nullable LuaFunction callback) {
        requests.add(new LuaRequest(
                clientName,
                queryConstructor,
                data,
                headers,
                callback));
    }

    @LuaBinding
    public void post(@Nonnull String clientName,
                     @Nonnull QueryConstructor queryConstructor,
                     @Nonnull Map.Entry<String, ContentType> data,
                     @Nullable LuaTable headers) {
        post(
                clientName,
                queryConstructor,
                data,
                headers,
                null);
    }

    @LuaBinding
    public void get(@Nonnull String clientName,
                    @Nonnull QueryConstructor queryConstructor,
                    @Nullable LuaTable headers,
                    @Nonnull LuaFunction callback) {
        requests.add(new LuaRequest(
                clientName,
                queryConstructor,
                headers,
                callback));
    }

    @LuaBinding
    public void post(@Nonnull String clientName,
                     @Nonnull QueryConstructor queryConstructor,
                     @Nonnull Map.Entry<String, ContentType> data,
                     @Nonnull LuaFunction callback) {
        post(
                clientName,
                queryConstructor,
                data,
                null,
                callback);
    }

    @LuaBinding
    public void post(@Nonnull String clientName,
                     @Nonnull QueryConstructor queryConstructor,
                     @Nonnull Map.Entry<String, ContentType> data) {
        post(
                clientName,
                queryConstructor,
                data,
                null,
                null);
    }

    @LuaBinding
    public void get(@Nonnull String clientName,
                    @Nonnull QueryConstructor queryConstructor,
                    @Nonnull LuaFunction callback) {
        get(
                clientName,
                queryConstructor,
                null,
                callback);
    }

    @LuaBinding
    @Nonnull
    public static Map.Entry<String, ContentType> json(LuaTable data) throws BadRequestException, IOException {
        return new AbstractMap.SimpleImmutableEntry<>(
                JsonType.NORMAL.toString(new JsonUtils.LuaAsJson(data)),
                ContentType.APPLICATION_JSON);
    }

    @LuaBinding
    @Nonnull
    public static Map.Entry<String, ContentType> dollar_json(LuaTable data) throws BadRequestException, IOException {
        return new AbstractMap.SimpleImmutableEntry<>(
                JsonType.DOLLAR.toString(new JsonUtils.LuaAsJson(data)),
                ContentType.APPLICATION_JSON);
    }

    @LuaBinding
    @Nonnull
    public static Map.Entry<String, ContentType> url_encoded(LuaTable data) throws IOException {
        final StringBuilder sb = new StringBuilder();

        if (data != null) {
            boolean first = true;
            for (LuaValue key : data.keys()) {
                if (!first) {
                    sb.append('&');
                }
                first = false;
                sb.append(URLEncoder.encode(key.checkjstring(), Charset.defaultCharset())).append('=');
                final LuaValue value = data.get(key);
                switch (value.type()) {
                    case LuaValue.TNONE:
                    case LuaValue.TNIL:
                        break;
                    case LuaValue.TNUMBER:
                        if (value.islong()) {
                            sb.append(value.tolong());
                        } else {
                            sb.append(value.todouble());
                        }
                        break;
                    case LuaValue.TBOOLEAN:
                    case LuaValue.TFUNCTION:
                    case LuaValue.TLIGHTUSERDATA:
                    case LuaValue.TUSERDATA:
                    case LuaValue.TTHREAD:
                    case LuaValue.TSTRING:
                    case LuaValue.TVALUE:
                    case LuaValue.TTABLE:
                        sb.append(URLEncoder.encode(value.tojstring(), Charset.defaultCharset()));
                        break;
                }
            }
        }

        return new AbstractMap.SimpleImmutableEntry<>(
                sb.toString(),
                ContentType.APPLICATION_FORM_URLENCODED);
    }

    @LuaBinding
    @Nonnull
    public static QueryConstructor query(@Nonnull String path, LuaTable args) throws BadRequestException {
        if (args != null && !path.endsWith("?")) {
            path += '?';
        }
        final QueryConstructor queryConstructor = new QueryConstructor(path, false);
        if (args != null) {
            for (LuaValue key : args.keys()) {
                final LuaValue value = args.get(key);
                switch (value.type()) {
                    case LuaValue.TNONE:
                    case LuaValue.TNIL:
                        continue;
                    case LuaValue.TNUMBER:
                        if (value.islong()) {
                            queryConstructor.append(key.checkjstring(), value.tolong());
                        } else {
                            queryConstructor.append(key.checkjstring(), value.tojstring());
                        }
                        break;
                    case LuaValue.TBOOLEAN:
                    case LuaValue.TFUNCTION:
                    case LuaValue.TLIGHTUSERDATA:
                    case LuaValue.TUSERDATA:
                    case LuaValue.TTHREAD:
                    case LuaValue.TSTRING:
                    case LuaValue.TVALUE:
                    case LuaValue.TTABLE:
                        queryConstructor.append(key.checkjstring(), value.tojstring());
                        break;
                }
            }
        }
        return queryConstructor;
    }

    @Nonnull
    public final List<LuaRequest> moveRequests() {
        final List<LuaRequest> movedRequests = requests;
        requests = new ArrayList<>();
        return movedRequests;
    }

    public boolean isEmpty() {
        return requests.isEmpty();
    }

    public static class LuaRequest implements Request {
        @Nonnull
        private final String clientName;
        @Nonnull
        private final BasicAsyncRequestProducerGenerator requestProducerGenerator;
        @Nullable
        private final LuaFunction function;

        public LuaRequest(@Nonnull String clientName,
                          @Nonnull QueryConstructor queryConstructor,
                          @Nonnull Map.Entry<String, ContentType> data,
                          @Nullable LuaTable headers,
                          @Nullable LuaFunction function) {
            this(clientName,
                    new BasicAsyncRequestProducerGenerator(
                            queryConstructor.toString(),
                            data.getKey(),
                            data.getValue()),
                    headers,
                    function);
        }

        public LuaRequest(@Nonnull String clientName,
                          @Nonnull QueryConstructor queryConstructor,
                          @Nullable LuaTable headers,
                          @Nullable LuaFunction function) {
            this(clientName,
                    new BasicAsyncRequestProducerGenerator(queryConstructor.toString()),
                    headers,
                    function);
        }


        private LuaRequest(@Nonnull String clientName,
                           @Nonnull BasicAsyncRequestProducerGenerator producerGenerator,
                           @Nullable LuaTable headers,
                           @Nullable LuaFunction function) {
            this.clientName = clientName;
            this.function = function;
            this.requestProducerGenerator = producerGenerator;
            requestProducerGenerator.addHeader("Accept-Encoding", "gzip");
            if (headers != null) {
                for (LuaValue key : headers.keys()) {
                    requestProducerGenerator.addHeader(key.checkjstring(), headers.get(key).checkjstring());
                }
            }
        }

        @Nonnull
        @Override
        public BasicAsyncRequestProducerGenerator makeRequest() {
            return requestProducerGenerator;
        }

        public boolean isEmpty() {
            return function == null;
        }

        @Nullable
        public FutureCallback<LuaValue> makeCallback(@Nonnull FutureCallback<Runnable> callback) {
            if (function != null) {
                return new Callback(callback, function);
            } else {
                return null;
            }
        }

        @Nonnull
        public String clientName() {
            return clientName;
        }
    }
//
//    private static byte[] compressData(@Nonnull String data) throws IOException {
//        final ByteArrayOutputStream compressed = new ByteArrayOutputStream();
//        try (GzipOutputStream stream = new GzipOutputStream(compressed)) {
//            stream.write(data.getBytes(Charset.defaultCharset()));
//        }
//        return compressed.toByteArray();
//    }

    private static String throwableMessage(Throwable e) {
        if (e instanceof BadResponseException) {
            final BadResponseException badResponseException = (BadResponseException) e;
            return badResponseException.statusCode() + ": " + badResponseException.getResponseBody();
        } else {
            return e.getLocalizedMessage();
        }
    }

    @Nonnull
    public static LuaTable throwableToLuaTable(Throwable e) {
        final LuaTable error = new LuaTable();
        error.set("type", e.getClass().getCanonicalName());

        error.set("message", throwableMessage(e));
        if (e.getCause() != null) {
            error.set("cause", throwableToLuaTable(e.getCause()));
        }
        return error;
    }

    public static class Callback implements FutureCallback<LuaValue> {
        @Nonnull
        private final LuaFunction function;
        @Nonnull
        private final FutureCallback<? super Runnable> callback;

        protected Callback(@Nonnull FutureCallback<? super Runnable> callback, @Nonnull LuaFunction function) {
            this.callback = callback;
            this.function = function;
        }

        @Override
        public void completed(LuaValue result) {
            final Runnable task = () -> function.call(LuaValue.NIL, result);
            callback.completed(task);
        }

        @Override
        public void failed(Exception e) {
            final Runnable task = () -> function.call(throwableToLuaTable(e));
            callback.completed(task);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }
}

