package ru.yandex.antifraud.artefacts;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaNil;
import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;

import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.lua_context_manager.LuaBinding;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.storage.CountersSingleUpdateRequest;
import ru.yandex.antifraud.storage.UpdateRequest;

import static ru.yandex.antifraud.storage.CountersUpdateRequest.makeCountersId;

public class PreparedCounters {
    private final Map<String, List<CounterToCheck>> countersByService = new HashMap<>();

    @Nonnull
    private final Instant now;
    @Nonnull
    private final ImmutableChannelConfig config;
    @Nonnull
    private final ChannelResolver channelResolver;
    @Nonnull
    private final Consumer<UpdateRequest> updateRequestConsumer;

    public PreparedCounters(@Nonnull Instant now,
                            @Nonnull ImmutableChannelConfig config,
                            @Nonnull ChannelResolver channelResolver,
                            @Nonnull Consumer<UpdateRequest> updateRequestConsumer) {
        this.now = now;
        this.config = config;
        this.channelResolver = channelResolver;
        this.updateRequestConsumer = updateRequestConsumer;
    }

    private CounterBinding counterToCheck(@Nonnull ImmutableChannelConfig config,
                                          @Nonnull String key,
                                          @Nonnull String value) {
        final CounterBinding counterToCheck =
                new CounterBinding(
                        now,
                        key,
                        value,
                        null,
                        config);
        countersByService.computeIfAbsent(config.storageService(), ignored -> new ArrayList<>()).add(counterToCheck);
        return counterToCheck;
    }

    @LuaBinding
    public CounterBinding counterToCheck(@Nonnull String channel,
                                         @Nonnull String subChannel,
                                         @Nonnull String key,
                                         @Nonnull String value) throws UnknownChannelException {
        return counterToCheck(
                channelResolver.resolve(channel, subChannel),
                key,
                value);
    }

    @LuaBinding
    public CounterBinding counterToCheck(@Nonnull String channel,
                                         @Nonnull String key,
                                         @Nonnull String value) throws UnknownChannelException {
        return counterToCheck(
                channelResolver.resolve(channel, null),
                key,
                value);
    }

    @LuaBinding
    public CounterToCheck counterToCheck(@Nonnull String key,
                                         @Nonnull String value) {
        return counterToCheck(config, key, value);
    }

    public Map<String, List<CounterToCheck>> getCountersToCheck() {
        return countersByService;
    }

    public static class CounterToCheck {
        @Nonnull
        private final Instant now;
        @Nonnull
        private final String id;
        @Nonnull
        private final String key;
        @Nonnull
        private final String value;
        @Nonnull
        private final String keyValue;
        @Nonnull
        private final ImmutableChannelConfig channelConfig;
        @Nonnull
        protected LuaValue checkedCounter = LuaNil.NIL;
        @Nullable
        protected LuaValue meta;
        @Nullable
        protected SmoothIncrementContext smoothIncrementContext = null;

        public CounterToCheck(@Nonnull Instant now,
                              @Nonnull String key,
                              @Nonnull String value,
                              @Nullable LuaValue meta,
                              @Nonnull ImmutableChannelConfig channelConfig) {
            this.now = now;
            this.key = key;
            this.value = value;
            this.keyValue = key + '_' + value;

            this.meta = meta;
            this.id = makeCountersId(keyValue, channelConfig);
            this.channelConfig = channelConfig;
        }

        @Nonnull
        public ImmutableChannelConfig channelConfig() {
            return channelConfig;
        }

        @Nonnull
        public Instant now() {
            return now;
        }

        @Nonnull
        public String id() {
            return id;
        }

        @Nonnull
        public String key() {
            return key;
        }

        @Nonnull
        public String value() {
            return value;
        }

        @Nonnull
        public String keyValue() {
            return keyValue;
        }

        @Nullable
        public LuaValue meta() {
            return meta;
        }

        public void checked(@Nonnull LuaValue checkedCounter) {
            this.checkedCounter = checkedCounter;
        }

        @Nullable
        public SmoothIncrementContext smoothIncrementContext() {
            return smoothIncrementContext;
        }
    }

    public class CounterBinding extends CounterToCheck {

        public CounterBinding(@Nonnull Instant now, @Nonnull String key, @Nonnull String value,
                              @Nullable LuaValue meta, @Nonnull ImmutableChannelConfig channelConfig) {
            super(now, key, value, meta, channelConfig);
        }

        @LuaBinding
        public void touch() {
            updateRequestConsumer.accept(new CountersSingleUpdateRequest(this));
        }

        @LuaBinding
        public void touch(LuaTable luaMeta) {
            meta = luaMeta;
            touch();
        }

        @LuaBinding
        public void inc(long count, long tau) {
            smoothIncrementContext = new SmoothIncrementContext(count, tau);
        }

        @LuaBinding
        @Nonnull
        public LuaValue checked() {
            return checkedCounter;
        }
    }

    public static class SmoothIncrementContext {
        private final long count;
        private final long tau;

        public SmoothIncrementContext(long count, long tau) {
            this.count = count;
            this.tau = tau;
        }

        public long getCount() {
            return count;
        }

        public long getTau() {
            return tau;
        }
    }
}
