package ru.yandex.antifraud.artefacts;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.lua_context_manager.LuaBinding;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.storage.ListItemSearchRequest;
import ru.yandex.antifraud.storage.ListItemsSearchRequest;
import ru.yandex.antifraud.storage.ListUpdateRequest;

public class PreparedLists {
    private final List<ValueToCheck> valuesToCheck = new ArrayList<>();
    private final Map<String, ListItemsSearchRequest> searchRequests = new HashMap<>();
    @Nonnull
    private final ChannelResolver channelResolver;
    @Nonnull
    private final Consumer<ListUpdateRequest> listUpdateRequestConsumer;

    public PreparedLists(@Nonnull ChannelResolver channelResolver,
                         @Nonnull Consumer<ListUpdateRequest> listUpdateRequestConsumer) {
        this.channelResolver = channelResolver;
        this.listUpdateRequestConsumer = listUpdateRequestConsumer;
    }

    public ValueToCheck add(@Nonnull Instant now,
                            @Nonnull String channel,
                            @Nullable String subChannel,
                            @Nonnull String listName,
                            @Nonnull String value) throws UnknownChannelException {
        final ImmutableChannelConfig channelConfig =
                channelResolver.resolve(channel, subChannel);
        final ValueToCheck valueToCheck = new ValueToCheck(channelConfig, now, listName, value);
        valuesToCheck.add(valueToCheck);
        searchRequests
                .computeIfAbsent(channelConfig.storageService(), ListItemsSearchRequest::new)
                .accept(valueToCheck);
        return valueToCheck;
    }

    public Map<String, ListItemsSearchRequest> getSearchRequests() {
        return searchRequests;
    }

    public List<ValueToCheck> getValuesToCheck() {
        return valuesToCheck;
    }

    public class ValueToCheck extends ListItemSearchRequest {
        @Nonnull
        private final Instant now;

        private boolean isInList = false;

        public ValueToCheck(@Nonnull ImmutableChannelConfig channelConfig,
                            @Nonnull Instant now,
                            @Nonnull String listName,
                            @Nonnull String value) {
            super(channelConfig, listName, value);
            this.now = now;
        }

        public void setIsInList(boolean isInList) {
            this.isInList = isInList;
        }

        @LuaBinding
        public boolean isInList() {
            return isInList;
        }

        @LuaBinding
        public void putInList(
                @Nonnull String author,
                @Nonnull String reason,
                int days
        ) throws TimeRange.IllegalTimeRangeException {
            listUpdateRequestConsumer.accept(new ListUpdateRequest(
                    channelConfig(),
                    listName(),
                    author,
                    reason,
                    Collections.singletonList(value()),
                    new TimeRange(now, now.plus(Duration.ofDays(days)))));
        }
    }
}
