package ru.yandex.antifraud.artefacts;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;

import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.lua.util.JsonUtils;

public class Resolution {
    private final JsonList tags;
    private final List<String> reason;
    private final String txId;
    @Nonnull
    private ResolutionCode resolutionCode;

    @Nonnull
    private final LuaTable customFields;

    public Resolution(@Nonnull String txId) {
        this.txId = txId;
        tags = new JsonList(BasicContainerFactory.INSTANCE);
        reason = new ArrayList<>();
        resolutionCode = ResolutionCode.ALLOW;
        customFields = new LuaTable();
    }

    public Resolution(@Nonnull JsonMap src) throws JsonException {
        tags = src.getList("tags");
        reason = new ArrayList<>(Arrays.asList(src.getString("reason").split("\n")));
        txId = src.getString("tx_id");
        resolutionCode = ResolutionCode.valueOf(src.getString("action"));
        customFields = JsonUtils.jsonToLua(src).checktable();
    }

    public AsJson asJson() {
        return new AsJson();
    }

    public JsonMap asJson(ContainerFactory containerFactory) {
        final JsonMap map = new JsonMap(containerFactory);

        map.put("action", new JsonString(resolutionCode.name()));

        map.put("tags", tags);

        map.put("reason", new JsonString(String.join("\n", reason)));

        map.put("risk_score", JsonLong.valueOf(0));

        map.put("rule_score", JsonLong.valueOf(0));

        map.put("status", new JsonString("success"));

        map.put("txId", new JsonString(txId));

        for (LuaValue key : customFields.keys()) {
            map.put(key.tojstring(), JsonUtils.luaToJson(customFields.get(key), containerFactory));
        }

        return map;
    }

    @Nonnull
    public ResolutionCode getResolutionCode() {
        return resolutionCode;
    }

    @Nonnull
    public JsonList getTags() {
        return tags;
    }

    @Nonnull
    public List<String> getReason() {
        return reason;
    }

    public void deny() {
        resolutionCode = ResolutionCode.DENY;
    }

    public void allow() {
        resolutionCode = ResolutionCode.ALLOW;
    }

    @Nonnull
    public ResolutionCode resolutionCode() {
        return resolutionCode;
    }

    @Nonnull
    public LuaTable customFields() {
        return customFields;
    }

    public class AsJson implements JsonValue {

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();

            writer.key("action");
            writer.value(resolutionCode.name());

            writer.key("tags");
            writer.value((JsonValue) tags);

            writer.key("reason");
            writer.value(String.join("\n", reason));

            writer.key("risk_score");
            writer.value(0);

            writer.key("rule_score");
            writer.value(0);

            writer.key("status");
            writer.value("success");

            writer.key("tx_id");
            writer.value(txId);

            for (LuaValue key : customFields.keys()) {
                writer.key(key.tojstring());
                writer.value(new JsonUtils.LuaAsJson(customFields.get(key)));
            }

            writer.endObject();
        }
    }
}
