package ru.yandex.antifraud.channel;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.logging.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.TransactionStatus;
import ru.yandex.antifraud.lua_context_manager.CurrenciesRatesTuner;
import ru.yandex.antifraud.lua_context_manager.DefaultsTuner;
import ru.yandex.antifraud.lua_context_manager.HypocrisyTuner;
import ru.yandex.antifraud.lua_context_manager.JsonTuner;
import ru.yandex.antifraud.lua_context_manager.LibPhoneNumberTuner;
import ru.yandex.antifraud.lua_context_manager.SolomonTuner;
import ru.yandex.antifraud.lua_context_manager.UaTraitsTuner;
import ru.yandex.antifraud.lua_context_manager.YasmTuner;
import ru.yandex.antifraud.util.JsonWriterSb;
import ru.yandex.logger.HandlersManager;
import ru.yandex.logger.ImmutableLoggerConfig;
import ru.yandex.lua.util.LuaException;
import ru.yandex.parser.config.ConfigException;

public class Channel {

    private final ImmutableChannelConfig channelConfig;

    private final Logger deliveryLogger;

    private final Logger paymentsLogger;

    @Nonnull
    private final EntriesDeque entries;

    private final YasmTuner yasmTuner;

    private final SolomonTuner solomonTuner;

    public Channel(@Nonnull ImmutableChannelConfig channelConfig,
                   @Nullable Path scriptsAdditionalDir,
                   @Nonnull YasmTuner yasmTuner,
                   @Nonnull SolomonTuner solomonTuner,
                   @Nullable UaTraitsTuner uaTraitsTuner,
                   @Nullable CurrenciesRatesTuner currenciesRatesTuner,
                   @Nonnull DefaultsTuner defaultsTuner,
                   @Nonnull JsonTuner jsonTuner) throws IOException,
            ConfigException, LuaException {
        try {
            this.channelConfig = channelConfig;
            this.yasmTuner = yasmTuner;
            this.solomonTuner = solomonTuner;
            {
                final ImmutableLoggerConfig deliveryLogConfig = channelConfig.deliveryLog();
                if (deliveryLogConfig != null) {
                    final HandlersManager handlersManager = new HandlersManager();
                    deliveryLogger = deliveryLogConfig.build(handlersManager);
                } else {
                    deliveryLogger = null;
                }
            }
            {
                final ImmutableLoggerConfig loggerConfig = channelConfig.paymentsLog();
                if (loggerConfig != null) {
                    paymentsLogger = loggerConfig.build(new HandlersManager());
                } else {
                    paymentsLogger = null;
                }
            }
            entries = new EntriesDeque(channelConfig.entry(),
                    scriptsAdditionalDir,
                    Arrays.asList(
                            yasmTuner,
                            solomonTuner,
                            uaTraitsTuner,
                            currenciesRatesTuner,
                            defaultsTuner,
                            LibPhoneNumberTuner.INSTANCE,
                            jsonTuner,
                            HypocrisyTuner.INSTANCE
                    )
            );
            for (EntriesDeque.AppRoot appRoot : EntriesDeque.AppRoot.values()) {
                final EntriesDeque.Entry entry = entries.popEntryOrCreate(appRoot);
                if (entry != null) {
                    entry.close();
                }
            }
        } catch (IOException | ConfigException | LuaException e) {
            e.addSuppressed(new RuntimeException("failed to load channel " + channelConfig.channelUri()));
            throw e;
        }
    }

    @Nonnull
    public ImmutableChannelConfig getConfig() {
        return channelConfig;
    }

    @Nonnull
    public EntriesDeque entries() {
        return entries;
    }

    @Nullable
    public Logger deliveryLogger() {
        return deliveryLogger;
    }

    public YasmTuner yasmTuner() {
        return yasmTuner;
    }

    public SolomonTuner solomonTuner() {
        return solomonTuner;
    }

    public void writePaymentLog(@Nonnull ScoringData scoringData) throws IOException {
        if (paymentsLogger != null && scoringData.getTransactionStatus() == TransactionStatus.OK) {
            final JsonWriterSb paymentData = JsonWriterSb.create();
            try (paymentData) {
                for (Field field : Field.FIELDS_FOR_PAYMENTS_LOG) {
                    final String value = scoringData.getValue(field).value();
                    if (value != null) {
                        paymentData.kv(field.fieldName(), value);
                    }
                }
            }
            paymentsLogger.fine(paymentData.toString());
        }
    }
}
