package ru.yandex.antifraud.channel;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.Globals;
import core.org.luaj.vm2.LuaClosure;
import core.org.luaj.vm2.LuaError;
import core.org.luaj.vm2.LuaFunction;
import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.Prototype;
import jse.org.luaj.vm2.lib.jse.JsePlatform;

import ru.yandex.antifraud.aggregates.Aggregates;
import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.artefacts.PreparedCounters;
import ru.yandex.antifraud.artefacts.PreparedLists;
import ru.yandex.antifraud.data.Counters;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.lua_context_manager.AggregatesTuner;
import ru.yandex.antifraud.lua_context_manager.ArtefactsTuner;
import ru.yandex.antifraud.lua_context_manager.CatboostTuner;
import ru.yandex.antifraud.lua_context_manager.ConfigTuner;
import ru.yandex.antifraud.lua_context_manager.ListTuner;
import ru.yandex.antifraud.lua_context_manager.ListsProvider;
import ru.yandex.antifraud.lua_context_manager.LogTuner;
import ru.yandex.antifraud.lua_context_manager.LuaPackage;
import ru.yandex.antifraud.lua_context_manager.RblTuner;
import ru.yandex.antifraud.lua_context_manager.RequestTuner;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.antifraud.lua_context_manager.VerificationLevelTuner;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.jni.catboost.JniCatboostModel;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.lua.util.LuaException;

public class EntriesDeque {
    private static final String MAIN = "main";
    private static final String PREPARE = "prepare";
    private static final String POST_ACTION = "post_action";
    @Nonnull
    private final List<LuaPackage> packages;
    @Nonnull
    private final Prototype entryModule;
    @Nonnull
    private final Map<String, Prototype> modules;
    @Nonnull
    private final Map<AppRoot, Deque<Entry>> entries = new EnumMap<>(AppRoot.class);

    public EntriesDeque(@Nonnull Path entry,
                        @Nullable Path additionalDir,
                        @Nonnull List<LuaPackage> packages) throws IOException, LuaException {
        this.packages = packages;
        entry = entry.toAbsolutePath();
        additionalDir = additionalDir != null ? additionalDir.toAbsolutePath() : null;

        if (additionalDir != null) {
            modules = readPrototypesFromDir(additionalDir, 0);
            modules.putAll(readPrototypesFromDir(entry.getParent(), Integer.MAX_VALUE));
        } else {
            modules = readPrototypesFromDir(entry.getParent(), Integer.MAX_VALUE);
        }

        entryModule = modules.remove(dropExtension(entry.getFileName().toString()));

        for (AppRoot appRoot : AppRoot.values()) {
            final Deque<Entry> entryDeque = new ConcurrentLinkedDeque<>();
            @Nullable final Entry firstEntry = makeEntry(makeEnvironmentEntry(), entryDeque, appRoot);
            if (firstEntry != null) {
                firstEntry.close();
                entries.put(appRoot, entryDeque);
            }
        }
    }

    static Map<String, Prototype> readPrototypesFromDir(@Nonnull Path dir, int maxDepth, int depth) throws IOException {
        final Map<String, Prototype> prototypeMap = new HashMap<>();

        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(dir)) {
            for (final Path path : dirStream) {
                final File file = path.toFile();
                final BasicFileAttributes basicFileAttributes = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
                if (basicFileAttributes.isRegularFile() && path.toString().endsWith(".lua")) {
                    try (BufferedReader reader = new BufferedReader(new FileReader(file,
                            Charset.defaultCharset()))) {
                        final Path relativePath = dir.relativize(path);
                        final Prototype prototype = JsePlatform.standardGlobals().compilePrototype(reader,
                                relativePath.toString());

                        prototypeMap.put(dropExtension(relativePath.toString()), prototype);
                    }
                } else if (basicFileAttributes.isDirectory() && depth < maxDepth) {
                    prototypeMap.putAll(readPrototypesFromDir(path, maxDepth, depth + 1));
                }
            }
        }

        return prototypeMap;
    }

    private static String dropExtension(String path) {
        if (path.endsWith(".lua")) {
            return path.substring(0, path.length() - 4);
        }
        return path;
    }

    static Map<String, Prototype> readPrototypesFromDir(@Nonnull Path dir, int maxDepth) throws IOException {
        return readPrototypesFromDir(dir, maxDepth, 0);
    }

    @Nullable
    private static Entry makeEntry(Globals environment, @Nonnull Deque<Entry> entryDeque, AppRoot appRoot) throws LuaException {
        try {
            final LuaValue value = environment.get(appRoot.functionName());
            if (value != null && value.isfunction()) {
                return new Entry(value.checkfunction().call().checktable(), entryDeque);
            }
            return null;
        } catch (Exception e) {
            throw new LuaException(e);
        }
    }

    private Globals makeEnvironmentEntry() throws LuaException {
        try {
            final Globals environment = JsePlatform.standardGlobals();
            for (LuaPackage luaPackage : packages) {
                if (luaPackage != null) {
                    environment.load(luaPackage);
                }
            }

            final LuaValue preloaded = environment.get("package").get("preload");

            {
                Set<Map.Entry<String, Prototype>> modulesToLoad = new HashSet<>(modules.entrySet());
                while (!modulesToLoad.isEmpty()) {
                    final Set<Map.Entry<String, Prototype>> failedLoadModules = new HashSet<>();
                    LuaError lastError = null;
                    for (Map.Entry<String, Prototype> kv : modulesToLoad) {
                        final String path = kv.getKey();
                        final Prototype prototype = kv.getValue();

                        try {
                            preloaded.set(path, new LuaClosure(prototype, environment));
                        } catch (LuaError e) {
                            failedLoadModules.add(kv);
                            lastError = e;
                        }
                    }

                    if (failedLoadModules.size() < modulesToLoad.size()) {
                        modulesToLoad = failedLoadModules;
                    } else {
                        throw new LuaException(lastError);
                    }
                }
            }

            new LuaClosure(entryModule, environment).call();
            return environment;
        } catch (LuaError e) {
            throw new LuaException(e);
        }
    }

    @Nullable
    public Entry popEntryOrCreate(AppRoot appRoot) throws LuaException {
        @Nullable final Deque<Entry> entryDeque = entries.getOrDefault(appRoot, null);
        try {
            if (entryDeque != null) {
                final Entry entry = entryDeque.pop();
                entry.open();
                return entry;
            } else {
                return null;
            }
        } catch (NoSuchElementException ignored) {
            return makeEntry(makeEnvironmentEntry(), entryDeque, appRoot);
        }
    }

    public enum AppRoot {
        SCORE("make_app"),
        SAVE("make_app_on_save"),
        ;

        private final String functionName;

        AppRoot(String functionName) {
            this.functionName = functionName;
        }

        public String functionName() {
            return functionName;
        }
    }

    public static class Context {
        @Nonnull
        private final LuaValue context = new LuaTable();
        @Nonnull
        private final Artefacts artefacts;
        @Nonnull
        private final List<ListsProvider> listsProviders = new ArrayList<>();
        @Nonnull
        private final Logger logger;
        @Nonnull
        private final ScoringData scoringData;
        @Nonnull
        private final Channel channel;

        public Context(
                @Nonnull Channel channel,
                @Nonnull ChannelResolver channelResolver,
                @Nonnull ScoringData scoringData,
                @Nonnull Logger logger) {
            this.scoringData = scoringData;
            this.channel = channel;
            this.logger = new PrefixedLogger(logger, scoringData.getExternalId(), ":");

            RequestTuner.INSTANCE.tuneLuaContext(context, scoringData, channel.getConfig());

            artefacts = new Artefacts(
                    scoringData,
                    channel.getConfig(),
                    channelResolver);

            LogTuner.INSTANCE.tuneLuaContext(context, this.logger, artefacts.rootLog(), artefacts.scriptLog());
            ConfigTuner.INSTANCE.tuneLuaContext(context, channel.getConfig());
            ArtefactsTuner.INSTANCE.tuneLuaContext(context, artefacts);

            ListTuner.tuneLuaContext(context,
                    scoringData.getTimestamp(),
                    channel.getConfig().channel(),
                    channel.getConfig().subChannel(),
                    artefacts.getLists());
        }

        public void updateContext(@Nonnull ListsProvider listsProvider,
                                  @Nonnull ChannelResolver channelResolver) {
            listsProviders.add(listsProvider);

            ListTuner.tuneLuaContext(
                    channelResolver,
                    context,
                    scoringData.getTimestamp(),
                    channel.getConfig(),
                    listsProviders,
                    artefacts.getListsChecks(),
                    artefacts.updateRequests());

            VerificationLevelTuner.tuneLuaContext(
                    context,
                    channel.getConfig(),
                    artefacts.updateRequests()
            );
        }

        public void updateContext(@Nonnull String message, @Nonnull Exception e) {
            this.logger.log(Level.WARNING, message, e);
            artefacts.getResolution().getReason().add(e.toString());
        }

        public void updateContext(@Nullable Aggregates aggregates,
                                  @Nullable RblData rblData) throws JsonBadCastException {
            if (aggregates != null) {
                AggregatesTuner.INSTANCE.tuneLuaContext(context, aggregates);
            }
            RblTuner.INSTANCE.tuneLuaContext(context, rblData);
        }

        public void updateContext(@Nonnull Map<String, JniCatboostModel> models) {
            CatboostTuner.INSTANCE.tuneContext(context, models);
        }

        public void updateContext(@Nonnull ListsProvider.ListChecker listChecker) {
            for (PreparedLists.ValueToCheck valueToCheck : artefacts.getLists().getValuesToCheck()) {
                @Nullable final TimeRange timeRange = listChecker.getRange(
                        valueToCheck.channelConfig(),
                        valueToCheck.listName(),
                        valueToCheck.value()
                );

                if (timeRange != null) {
                    valueToCheck.setIsInList(timeRange.contains(scoringData.getTimestamp()));
                }
            }
        }

        public void updateContext(@Nonnull Counters counters) {
            for (var entry : artefacts.getCounters().getCountersToCheck().entrySet()) {
                for (PreparedCounters.CounterToCheck counterToCheck : entry.getValue()) {
                    @Nullable final LuaTable checkedCounter = counters.countersByIds().get(counterToCheck.id());

                    if (checkedCounter != null) {
                        counterToCheck.checked(checkedCounter);
                    }
                }
            }
        }

        @Nonnull
        private LuaValue context() {
            return context;
        }

        @Nonnull
        public ScoringData scoringData() {
            return scoringData;
        }

        @Nonnull
        public Artefacts artefacts() {
            return artefacts;
        }

        @Nonnull
        public Channel channel() {
            return channel;
        }
    }

    public static class Entry implements GenericAutoCloseable<RuntimeException>, Consumer<Context> {
        private final Deque<Entry> entryDeque;
        private final AtomicBoolean closed = new AtomicBoolean(false);
        @Nonnull
        private final LuaTable obj;
        @Nonnull
        private final Deque<Consumer<Context>> stages = new ArrayDeque<>();
        @Nullable
        private final Consumer<Context> prepareStage;
        @Nonnull
        private final LuaFunction main;
        @Nullable
        private final LuaFunction postAction;

        protected Entry(@Nonnull LuaTable obj, Deque<Entry> entryDeque) {
            this.entryDeque = entryDeque;
            this.obj = obj;

            main = obj.get(MAIN).checkfunction();

            {
                final LuaValue prepare = obj.get(PREPARE);
                if (prepare != null && prepare.isfunction()) {
                    final LuaFunction prepareFunction = prepare.checkfunction();
                    prepareStage = (context) -> {
                        try {
                            prepareFunction.call(obj, context.context());
                        } catch (Exception e) {
                            context.updateContext("stage failed ", e);
                        }
                    };
                } else {
                    prepareStage = null;
                }
            }

            {
                final LuaValue postAction = obj.get(POST_ACTION);
                if (postAction != null && postAction.isfunction()) {
                    this.postAction = postAction.checkfunction();
                } else {
                    this.postAction = null;
                }
            }

            open();
        }

        public void open() {
            stages.clear();
            if (prepareStage != null) {
                stages.add(prepareStage);
            }
            closed.set(false);
        }

        @Override
        public void close() {
            checkOpen();

            if (closed.compareAndSet(false, true)) {
                stages.clear();
                entryDeque.push(this);
            }
        }

        @Override
        public void accept(@Nonnull Context context) {
            checkOpen();
            while (!stages.isEmpty()) {
                stages.pop().accept(context);
            }
        }

        public boolean isEmpty() {
            return stages.isEmpty();
        }

        public void runMain(@Nonnull Context context) {
            checkOpen();
            try {
                main.call(obj, context.context());
            } catch (Exception e) {
                context.updateContext("main failed ", e);
            }
        }

        public void addStage(@Nonnull Consumer<Context> stage) {
            checkOpen();
            stages.push(stage);
        }

        public void runPostAction(@Nonnull Context context) {
            checkOpen();
            if (postAction != null) {
                try {
                    postAction.call(obj, context.context());
                } catch (Exception e) {
                    context.updateContext("post action failed", e);
                }
            }
        }

        private void checkOpen() {
            if (closed.get()) {
                throw new RuntimeException("entry closed, but there are " + stages.size() + " stages");
            }
        }
    }
}
