package ru.yandex.antifraud.channel.config;

import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.aggregates_v2.aggregates_config.AggregatorsConfig;
import ru.yandex.antifraud.aggregates_v2.aggregates_config.AggregatorsConfigBuilder;
import ru.yandex.antifraud.channel.route_config.ChannelRouteConfigBuilder;
import ru.yandex.logger.LoggerConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractChannelConfigBuilder
        <T extends AbstractChannelConfigBuilder<T>>
        extends ChannelRouteConfigBuilder
        implements ChannelConfig {
    private Path entry;
    private List<String> listDir;
    private LoggerConfigBuilder deliveryLog;
    private LoggerConfigBuilder paymentsLog;
    private String storageService;
    private String storageLoginIdService;
    private Set<String> fieldsToExclude;
    private List<String> crossChannels;
    private Duration gapToCollapsedAggrsBegin;
    private Duration gapToCollapsedAggrsEnd;
    private boolean storeInDb;

    private AggregatorsConfigBuilder aggregatorsConfig;

    protected AbstractChannelConfigBuilder(final ChannelConfig config) {
        super(config);
        entry = config.entry();
        listDir = config.listDir();
        if (config.deliveryLog() != null) {
            deliveryLog = new LoggerConfigBuilder(config.deliveryLog());
        } else {
            deliveryLog = null;
        }
        if (config.paymentsLog() != null) {
            paymentsLog = new LoggerConfigBuilder(config.paymentsLog());
        } else {
            paymentsLog = null;
        }
        storageService = config.storageService();
        storageLoginIdService = config.storageLoginIdService();
        fieldsToExclude = config.fieldsToExclude();
        crossChannels = config.crossChannels();

        gapToCollapsedAggrsBegin = config.gapToCollapsedAggrsBegin();
        gapToCollapsedAggrsEnd = config.gapToCollapsedAggrsEnd();

        storeInDb = config.storeInDb();

        aggregatorsConfig = new AggregatorsConfigBuilder(config.aggregatorsConfig());
    }

    protected AbstractChannelConfigBuilder(
            @Nonnull Path root,
            @Nonnull IniConfig config,
            @Nonnull ChannelConfig defaults)
            throws ConfigException {
        super(config, defaults);
        entry = Path.of(root.toString(), config.getString(ENTRY));

        listDir =
                LIST_DIR.extract(config, defaults.listDir()).stream().map(d -> Path.of(root.toString(), d).toString()).collect(Collectors.toList());

        {
            final IniConfig section = config.sectionOrNull(DELIVERY_LOG);
            if (section != null) {
                deliveryLog =
                        new LoggerConfigBuilder(
                                section,
                                defaults.deliveryLog());
            } else {
                deliveryLog = null;
            }
        }

        {
            final IniConfig section = config.sectionOrNull(PAYMENTS_LOG);
            if (section != null) {
                paymentsLog =
                        new LoggerConfigBuilder(
                                section,
                                defaults.paymentsLog());
            } else {
                paymentsLog = null;
            }
        }

        fieldsToExclude = new HashSet<>(FIELDS_TO_EXCLUDE.extract(config, defaults.fieldsToExclude()));
        crossChannels = new ArrayList<>(CROSS_CHANNELS.extract(config, defaults.crossChannels()));

        storageService = config.getString(STORAGE_SERVICE, defaults.storageService());
        storageLoginIdService = config.getString(STORAGE_LOGIN_ID_SERVICE, defaults.storageLoginIdService());

        gapToCollapsedAggrsBegin = Duration.ofMillis(config.getNegativelyLongDuration(COLLAPSED_AGGRS_BEGIN,
                defaults.gapToCollapsedAggrsBegin().toMillis()));

        gapToCollapsedAggrsEnd = Duration.ofMillis(config.getNegativelyLongDuration(COLLAPSED_AGGRS_END,
                defaults.gapToCollapsedAggrsEnd().toMillis()));

        if (gapToCollapsedAggrsEnd.compareTo(gapToCollapsedAggrsBegin) <= 0) {
            throw new ConfigException("end of range=" + gapToCollapsedAggrsEnd + " less or equal then begin=" + gapToCollapsedAggrsBegin);
        }

        storeInDb = config.getBoolean(STORE_IN_DB, defaults.storeInDb());

        aggregatorsConfig = new AggregatorsConfigBuilder(config.section(AGGREGATORS));

    }

    @Nonnull
    @Override
    public Path entry() {
        return entry;
    }

    public void entry(Path entry) {
        this.entry = entry;
    }

    @Nullable
    @Override
    public List<String> listDir() {
        return listDir;
    }

    public void listDir(List<String> listDir) {
        this.listDir = listDir;
    }

    @Nullable
    @Override
    public LoggerConfigBuilder deliveryLog() {
        return deliveryLog;
    }

    public void deliveryLog(LoggerConfigBuilder deliveryLog) {
        this.deliveryLog = deliveryLog;
    }

    @Nullable
    @Override
    public LoggerConfigBuilder paymentsLog() {
        return paymentsLog;
    }

    public void paymentsLog(LoggerConfigBuilder paymentsLog) {
        this.paymentsLog = paymentsLog;
    }

    @Nonnull
    @Override
    public String storageService() {
        return storageService;
    }

    public void storageService(@Nonnull String storageService) {
        this.storageService = storageService;
    }

    @Nonnull
    @Override
    public String storageLoginIdService() {
        return storageLoginIdService;
    }

    public void storageLoginIdService(@Nonnull String storageLoginIdService) {
        this.storageLoginIdService = storageLoginIdService;
    }

    @Nonnull
    @Override
    public Set<String> fieldsToExclude() {
        return fieldsToExclude;
    }

    public void fieldsToExclude(@Nonnull Set<String> fieldsToExclude) {
        this.fieldsToExclude = fieldsToExclude;
    }

    @Nonnull
    @Override
    public List<String> crossChannels() {
        return crossChannels;
    }

    public void crossChannels(@Nonnull List<String> crossChannels) {
        this.crossChannels = crossChannels;
    }

    @Override
    @Nonnull
    public Duration gapToCollapsedAggrsBegin() {
        return gapToCollapsedAggrsBegin;
    }

    public void gapToCollapsedAggrsBegin(@Nonnull Duration gapToCollapsedAggrsBegin) {
        this.gapToCollapsedAggrsBegin = gapToCollapsedAggrsBegin;
    }

    @Override
    @Nonnull
    public Duration gapToCollapsedAggrsEnd() {
        return gapToCollapsedAggrsEnd;
    }

    public void gapToCollapsedAggrsEnd(@Nonnull Duration gapToCollapsedAggrsEnd) {
        this.gapToCollapsedAggrsEnd = gapToCollapsedAggrsEnd;
    }

    @Override
    public boolean storeInDb() {
        return storeInDb;
    }

    public void storeInDb(boolean storeInDb) {
        this.storeInDb = storeInDb;
    }

    @Nonnull
    @Override
    public AggregatorsConfigBuilder aggregatorsConfig() {
        return aggregatorsConfig;
    }

    public void aggregatorsConfig(AggregatorsConfig aggregatorsConfig) {
        this.aggregatorsConfig = new AggregatorsConfigBuilder(aggregatorsConfig);
    }
}

