package ru.yandex.antifraud.channel.config;

import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.aggregates_v2.aggregates_config.ImmutableAggregatorsConfig;
import ru.yandex.antifraud.channel.route_config.ImmutableChannelRouteConfig;
import ru.yandex.logger.ImmutableLoggerConfig;
import ru.yandex.logger.LoggerConfig;
import ru.yandex.parser.config.ConfigException;

public class ImmutableChannelConfig extends ImmutableChannelRouteConfig
        implements ChannelConfig {
    private final Path entry;
    private final List<String> listDir;
    private final ImmutableLoggerConfig deliveryLog;
    private final ImmutableLoggerConfig paymentsLog;
    private final ServiceChannelUri serviceChannelUri;
    private final String storageLoginIdService;
    private final Set<String> fieldsToExclude;
    private final List<String> crossChannels;
    private final Duration gapToCollapsedAggrsBegin;
    private final Duration gapToCollapsedAggrsEnd;
    private final boolean storeInDb;

    private final ImmutableAggregatorsConfig aggregatorsConfig;

    public ImmutableChannelConfig(final ChannelConfig config) throws ConfigException {
        super(config);
        entry = config.entry();
        listDir = config.listDir();
        {
            final LoggerConfig deliveryLogConfig = config.deliveryLog();
            if (deliveryLogConfig != null) {
                deliveryLog = new ImmutableLoggerConfig(deliveryLogConfig);
            } else {
                deliveryLog = null;
            }
        }
        {
            final LoggerConfig loggerConfig = config.paymentsLog();
            if (loggerConfig != null) {
                paymentsLog = new ImmutableLoggerConfig(loggerConfig);
            } else {
                paymentsLog = null;
            }
        }

        serviceChannelUri = new ServiceChannelUri(channelUri(), config.storageService());
        storageLoginIdService = config.storageLoginIdService();
        fieldsToExclude = config.fieldsToExclude();
        crossChannels = config.crossChannels();

        gapToCollapsedAggrsBegin = config.gapToCollapsedAggrsBegin();
        gapToCollapsedAggrsEnd = config.gapToCollapsedAggrsEnd();

        storeInDb = config.storeInDb();

        aggregatorsConfig = new ImmutableAggregatorsConfig(config.aggregatorsConfig());
    }

    @Nonnull
    @Override
    public Path entry() {
        return entry;
    }

    @Nullable
    @Override
    public List<String> listDir() {
        return listDir;
    }

    @Nullable
    @Override
    public ImmutableLoggerConfig deliveryLog() {
        return deliveryLog;
    }

    @Nullable
    @Override
    public ImmutableLoggerConfig paymentsLog() {
        return paymentsLog;
    }

    @Nonnull
    @Override
    public String storageService() {
        return serviceChannelUri.getStorageService();
    }

    @Nonnull
    @Override
    public String storageLoginIdService() {
        return storageLoginIdService;
    }

    @Nonnull
    @Override
    public Set<String> fieldsToExclude() {
        return fieldsToExclude;
    }

    @Nonnull
    @Override
    public List<String> crossChannels() {
        return crossChannels;
    }

    @Override
    @Nonnull
    public Duration gapToCollapsedAggrsBegin() {
        return gapToCollapsedAggrsBegin;
    }

    @Override
    @Nonnull
    public Duration gapToCollapsedAggrsEnd() {
        return gapToCollapsedAggrsEnd;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof ImmutableChannelConfig) {
            final ImmutableChannelConfig other = (ImmutableChannelConfig) o;
            return Objects.equals(getServiceChannelUri(), other.getServiceChannelUri()) &&
                    Objects.equals(serviceIds(), other.serviceIds());
        } else {
            return false;
        }
    }

    public ServiceChannelUri getServiceChannelUri() {
        return serviceChannelUri;
    }

    @Override
    public int hashCode() {
        int hash = channel().hashCode();

        {
            final String subChannel = subChannel();
            if (subChannel != null) {
                hash ^= subChannel.hashCode();
            }
        }

        hash ^= serviceIds().hashCode();

        return hash;
    }

    @Override
    public boolean storeInDb() {
        return storeInDb;
    }

    @Override
    @Nonnull
    public ImmutableAggregatorsConfig aggregatorsConfig() {
        return aggregatorsConfig;
    }

}

