package ru.yandex.antifraud.currency;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Currency;
import java.util.HashMap;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;

public class CurrencyRateMap implements ToRubConverter {

    @Nonnull
    private static final Currency RUB_CURRENCY = Objects.requireNonNull(CurrencyMap.INSTANCE.getItemByName("RUB"));

    @Nonnull
    private static final Currency USD_CURRENCY = Objects.requireNonNull(CurrencyMap.INSTANCE.getItemByName("USD"));

    @Nonnull
    private final HashMap<Currency, HashMap<Currency, Double>> rates = new HashMap<>();
    @Nonnull
    private final HashMap<Currency, Double> ratesToRub;
    @Nonnull
    private final HashMap<Currency, Double> ratesToUsd;

    @Nonnull
    public static CurrencyRateMap make(@Nonnull Path path) throws IOException, JsonException {
        try (final BufferedReader reader = Files.newBufferedReader(path, Charset.defaultCharset())) {
            return new CurrencyRateMap(reader);
        }
    }

    public CurrencyRateMap(@Nonnull BufferedReader reader) throws IOException, JsonException {
        final BasicGenericConsumer<JsonObject, JsonException> jsonConsumer =
                new BasicGenericConsumer<>();

        final JsonParser jsonParser = TypesafeValueContentHandler.prepareParser(jsonConsumer);

        while (reader.ready()) {
            final String line = reader.readLine();
            if (line == null) {
                break;
            }
            jsonParser.parse(line);

            final JsonMap rateRelation = jsonConsumer.get().asMap();

            final int baseCcNum = rateRelation.getInt("base_cc_num");
            final int ccNum = rateRelation.getInt("cc_num");

            final Currency srcCurrency =
                    CurrencyMap.INSTANCE.getItemByNumCode(baseCcNum);

            final Currency dstCurrency =
                    CurrencyMap.INSTANCE.getItemByNumCode(ccNum);
            if (srcCurrency == null || dstCurrency == null) {
                continue;
            }

            final double reversedRate = rateRelation.getDouble("rate");
            final double rate = 1. / reversedRate;

            rates.computeIfAbsent(srcCurrency, ignored -> new HashMap<>())
                    .put(srcCurrency, 1.0);
            rates.computeIfAbsent(dstCurrency, ignored -> new HashMap<>())
                    .put(dstCurrency, 1.0);

            rates.computeIfAbsent(srcCurrency, ignored -> new HashMap<>())
                    .put(dstCurrency, rate);
            rates.computeIfAbsent(dstCurrency, ignored -> new HashMap<>())
                    .put(srcCurrency, reversedRate);

            jsonParser.reset();
        }
        ratesToRub = Objects.requireNonNull(rates.get(RUB_CURRENCY));
        ratesToUsd = Objects.requireNonNull(rates.get(USD_CURRENCY));
    }

    public Double rate(@Nonnull Currency from, @Nonnull Currency to) {
        final HashMap<Currency, Double> tos = rates.getOrDefault(from, null);
        if (tos == null) {
            return null;
        }

        {
            final Double rate = tos.getOrDefault(to, null);
            if (rate != null) {
                return rate;
            }
        }

        if (from != USD_CURRENCY && to != USD_CURRENCY) {
            final Double rate = rateWithBuffer(tos, to, USD_CURRENCY, ratesToUsd);
            if (rate != null) {
                return rate;
            }
        }

        if (from != RUB_CURRENCY && to != RUB_CURRENCY) {
            final Double rate = rateWithBuffer(tos, to, RUB_CURRENCY, ratesToRub);
            if (rate != null) {
                return rate;
            }
        }

        return null;
    }

    public Double rateWithBuffer(@Nonnull final HashMap<Currency, Double> tos,
                                 @Nonnull Currency to,
                                 @Nonnull Currency bufferCurrency,
                                 @Nonnull HashMap<Currency, Double> bufferRates) {
        final Double toTo = tos.getOrDefault(bufferCurrency, null);
        final Double fromTo = bufferRates.getOrDefault(to, null);
        if (toTo != null && fromTo != null) {
            return toTo * fromTo;
        }
        return null;
    }

    public Double rateToRub(@Nullable Currency from) {
        if (from != null) {
            return rate(from, RUB_CURRENCY);
        } else {
            return null;
        }
    }

    @Nullable
    @Override
    public Long apply(@Nonnull Amount amount) {
        final Double rate = rateToRub(amount.currency());
        if (rate != null) {
            return (long) (rate * amount.amount());
        } else {
            return null;
        }
    }
}
