package ru.yandex.antifraud.data;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaTable;

import ru.yandex.antifraud.aggregates.Aggregates;
import ru.yandex.antifraud.aggregates.AggregatesBatch;
import ru.yandex.antifraud.aggregates.AggregatesBuilder;
import ru.yandex.antifraud.aggregates.StatsMergeException;
import ru.yandex.antifraud.aggregates_v2.aggregates_config.ImmutableAggregatorsConfig;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.lua_context_manager.SolomonTuner;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;

public class AggregatedData {
    @Nonnull
    private final List<StorageData> datum = new ArrayList<>();
    @Nonnull
    private final Set<String> ids = new HashSet<>();

    public int process(@Nonnull JsonObject data) throws JsonException {
        final JsonList hitsArray = data.asMap().getList("hitsArray");

        for (final JsonObject rawObject : hitsArray) {
            final RequestType type = RequestType.valueOf(rawObject.asMap().getString("type",
                    RequestType.MAIN.toString()));

            final String id = rawObject.asMap().getString("id");
            if (!ids.add(id)) {
                continue;
            }

            if (type == RequestType.MAIN) {
                final StorageData storageData = new StorageData(rawObject.asMap());
                datum.add(storageData);
            }
        }

        return hitsArray.size();
    }

    @Nonnull
    public static Aggregates calcAggregates(
            @Nonnull ImmutableChannelConfig channelConfig,
            @Nonnull List<Map.Entry<ImmutableChannelConfig, AggregatesBatch>> aggregatesBatches,
            @Nonnull ScoringData request,
            @Nullable RblData rblData,
            @Nonnull SolomonTuner solomonTuner
    ) throws JsonBadCastException, StatsMergeException {
        if (aggregatesBatches.isEmpty()) {
            return Aggregates.EMPTY;
        }

        Aggregates headAggregates = null;

        final JsonMap shortCross = new JsonMap(BasicContainerFactory.INSTANCE);
        final LuaTable fullCross = new LuaTable();

        for (Map.Entry<ImmutableChannelConfig, AggregatesBatch> kv : aggregatesBatches) {
            if (kv.getValue() == null) {
                continue;
            }

            final ImmutableChannelConfig config = kv.getKey();
            final AggregatesBatch batch = kv.getValue();
            final AggregatesBuilder builder = new AggregatesBuilder(
                    request,
                    batch.getCounters(),
                    rblData,
                    channelConfig.aggregatorsConfig(),
                    !batch.areCollapsedAggrsEmpty(),
                    !batch.isAggregatedDataEmpty(),
                    !batch.areCountersEmpty()
            );

            for (StorageData data : batch.getAggregatedData().datum) {
                builder.consume(data);
            }

            for (CollapsedAggregates data : batch.getCollapsedAggregatesResponse().getAggregatesList()) {
                builder.consume(data);
            }

            final Aggregates aggregates = builder.build();

            if (Objects.equals(config, channelConfig)) {
                headAggregates = aggregates;
                builder.stat(solomonTuner);
            } else {
                shortCross.put(config.channelUri(), aggregates.asShortJson());
                fullCross.set(config.channelUri(), aggregates.asFull());
            }
        }

        if (headAggregates == null) {
            return Aggregates.EMPTY;
        }

        headAggregates.asShortJson().put("cross", shortCross);
        headAggregates.asFull().set("cross", fullCross);

        return headAggregates;

    }

    @Nonnull
    public Aggregates calcAggregates(
            @Nonnull ImmutableAggregatorsConfig aggregatorsConfig,
            @Nonnull ScoringData request,
            @Nonnull CollapsedAggregatesResponse collapsedAggrs,
            @Nonnull Counters counters,
            @Nullable RblData rblData
    ) throws JsonBadCastException, StatsMergeException {
        final AggregatesBuilder builder = new AggregatesBuilder(
                request,
                counters,
                rblData,
                aggregatorsConfig,
                !collapsedAggrs.getAggregatesList().isEmpty(),
                !datum.isEmpty(),
                !counters.isEmpty());

        for (StorageData data : datum) {
            builder.consume(data);
        }

        for (CollapsedAggregates data : collapsedAggrs.getAggregatesList()) {
            builder.consume(data);
        }

        return builder.build();
    }
}
