package ru.yandex.antifraud.data;

import java.time.Instant;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.aggregates.Aggregator;
import ru.yandex.antifraud.aggregates.MultiStructuredStat;
import ru.yandex.antifraud.aggregates.StatFactory;
import ru.yandex.antifraud.aggregates.Stats;
import ru.yandex.antifraud.aggregates.StructuredAggregator;
import ru.yandex.antifraud.aggregates.StructuredStat;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;

public class CollapsedAggregates {
    @Nonnull
    private final Stats stats;
    @Nonnull
    private final Map<StructuredAggregator, StructuredStat> structuredStats;
    @Nonnull
    private final Instant timestamp;
    @Nonnull
    private final String fieldPrefix;

    public CollapsedAggregates(@Nonnull JsonMap src) throws JsonException {
        timestamp = Instant.ofEpochMilli(src.getLong("txn_timestamp"));
        fieldPrefix = src.getString("key");

        final BasicGenericConsumer<JsonObject, JsonException> consumer =
                new BasicGenericConsumer<>();
        final JsonParser jsonParser = TypesafeValueContentHandler.prepareParser(consumer);
        jsonParser.parse(src.getString("data"));
        jsonParser.reset();

        final JsonMap aggregates = consumer.get().asMap();
        stats = new Stats();
        for (Map.Entry<String, JsonObject> entry : aggregates.entrySet()) {
            stats.put(Aggregator.parse(entry.getKey()), StatFactory.INSTANCE.fromJson(entry.getValue()));
        }
        final String rawStructuredAggregates = src.getString("user_context", null);
        if (rawStructuredAggregates != null) {
            jsonParser.parse(rawStructuredAggregates);
            jsonParser.reset();

            structuredStats = new EnumMap<>(StructuredAggregator.class);
            for (var entry : consumer.get().asMap().entrySet()) {
                structuredStats.put(StructuredAggregator.parse(entry.getKey()),
                        MultiStructuredStat.parse(entry.getValue()));
            }
        } else {
            structuredStats = Collections.emptyMap();
        }
    }

    @Nonnull
    public Stats stats() {
        return stats;
    }

    @Nonnull
    public Map<StructuredAggregator, StructuredStat> structuredStats() {
        return structuredStats;
    }

    @Nonnull
    public Instant getTimestamp() {
        return timestamp;
    }

    @Nonnull
    public String fieldPrefix() {
        return fieldPrefix;
    }
}
